import logging

import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry


logger = logging.getLogger(__name__)


class ApiBase(object):
    def __init__(self, token):
        self._session = requests.Session()
        self._session.headers.update({'Authorization': 'OAuth {}'.format(token)})
        retries = Retry(total=3, backoff_factor=0.5)
        adapter = HTTPAdapter(max_retries=retries)
        self._session.mount('http://', adapter)
        self._session.mount('https://', adapter)

    def _make_request(self, method, *args, **kwargs):
        logger.info('Make request: method=[%s], args=[%r], kwargs=[%r]', method, args, kwargs)
        return getattr(self._session, method)(*args, **kwargs)

    def _process_text_query(self, method, *args, **kwargs):
        response = self._make_request(method, *args, **kwargs)
        try:
            response.raise_for_status()
        except requests.HTTPError:
            logger.exception('Api Error')
            logger.error('Response body: %s', response.text)
            raise

        return response.text

    def _process_json_query(self, method, *args, **kwargs):
        response = self._make_request(method, *args, **kwargs)
        try:
            response.raise_for_status()
        except requests.HTTPError:
            logger.exception('Api Error')
            logger.error('Response body: %s', response.text)
            raise

        return response.json()


class QloudPublicApi(ApiBase):
    URL = 'https://platform.yandex-team.ru/api/v1/'

    def list_environments(self, application_id):
        url = self.URL + 'application/{}'.format(application_id)
        return self._process_json_query('get', url)

    def dump_environment(self, environment_id):
        url = self.URL + 'environment/dump/{}'.format(environment_id)
        return self._process_json_query('get', url)

    def upload_environment(self, environment_dump, verbose=False):
        path = 'environment/upload/return-header' if verbose else 'environment/upload'
        return self._process_json_query('post', self.URL + path, json=environment_dump)

    def create_environment(self, application_id, environment_name, engine='platform'):
        url = self.URL + 'environment/new'
        return self._process_json_query('post', url, data={
            'applicationId': application_id,
            'environmentName': environment_name,
            'engine': engine,
        })

    def get_environment_info(self, environment_id):
        url = self.URL + 'environment/stable/{}'.format(environment_id)
        return self._process_json_query('get', url)

    def delete_environment(self, environment_id):
        url = self.URL + 'environment/stable/{}'.format(environment_id)
        self._process_text_query('delete', url)

    def environment_exists(self, environment_id):
        try:
            url = self.URL + 'environment/dump/{}'.format(environment_id)
            response = self._make_request('get', url)
            if response.status_code == 404:
                return False
            response.raise_for_status()
        except requests.HTTPError:
            logger.exception('Api Error')
            logger.error('Response body: %s', response.text)
            raise
        else:
            return True

    def get_component_info(self, component_id):
        environment_id, component_name = component_id.rsplit('.', 1)
        dump = self.get_environment_info(environment_id)

        for current_component_name, component in dump.get('components', {}).items():
            if current_component_name == component_name or component_name == '*':
                return component
        return None
