import logging
import requests

import sandbox.projects.common.string as str_utils


class RequestHelper(object):
    API_URL = None
    DEFAULT_TIMEOUT = 30  # sec
    CHECK_RESPONSE = True

    def __init__(self, api_url=None, timeout=DEFAULT_TIMEOUT, ssl_verify=False):
        self.api_url = api_url or self.API_URL
        self.logger = logging.getLogger(__name__)
        self.timeout = timeout
        self.ssl_verify = ssl_verify
        self.headers = None

    def _check_response(self, response, request_type):
        """
        Log response url. Fail on bad status codes.

        :return: response in json
        """
        if not self.CHECK_RESPONSE:
            return response
        self.logger.debug(response.url)
        if response.status_code == requests.codes.ok:
            return response.json()
        else:
            self.logger.error("Return code %s. Content: '%s'", response.status_code, response.content)
            raise requests.HTTPError("{} request fails with status code {}: {}".format(
                request_type, response.status_code, str_utils.all_to_str(response.text)
            ))

    def _do_get(self, url, params=None):
        method = "GET"
        full_url = self.api_url + url
        self.logger.debug("%s %s | PARAMS: %s", method, full_url, params)
        r = requests.get(
            full_url,
            params=params,
            timeout=self.timeout,
            headers=self.headers,
            verify=self.ssl_verify
        )
        return self._check_response(r, method)

    def _do_post(self, url, data=None, params=None, json=None, files=None):
        method = "POST"
        full_url = self.api_url + url
        self.logger.debug("%s %s | PARAMS: %s", method, full_url, params)
        post_request = requests.Request(
            method=method,
            url=self.api_url + url,
            params=params,
            data=data,
            files=files,
            headers=self.headers,
            json=json or {},
        )
        with requests.sessions.Session() as session:
            prepped = session.prepare_request(post_request)
            self.logger.debug("REQUEST: %s", format_prepped_request(prepped))
            r = session.send(prepped, timeout=self.timeout, verify=self.ssl_verify)
        return self._check_response(r, method)

    def _do_delete(self, url):
        method = "DELETE"
        full_url = self.api_url + url
        self.logger.debug("%s %s", method, full_url)
        r = requests.delete(
            url=full_url,
            timeout=self.timeout,
            headers=self.headers,
            verify=self.ssl_verify,
        )
        return self._check_response(r, method)

    def _do_put(self, url):
        method = "PUT"
        full_url = self.api_url + url
        self.logger.debug("%s %s", method, full_url)
        r = requests.put(
            url=full_url,
            timeout=self.timeout,
            headers=self.headers,
            verify=self.ssl_verify,
        )
        return self._check_response(r, method)

    def exists(self, url):
        r = requests.head(
            self.api_url + url,
            timeout=self.timeout,
            headers=self.headers,
            verify=self.ssl_verify
        )
        if r.status_code == requests.codes.ok:
            return True
        self.logger.error("Bad status code: %s", r.status_code)
        return False


def format_prepped_request(prepped):
    return "\n".join([
        "{} {}".format(prepped.method, prepped.path_url),
        '\n'.join(['{}: {}'.format(*hv) for hv in prepped.headers.items()]),
        "",
        str(prepped.body),
    ])
