import logging
import six
import typing
import requests

import sandbox.projects.abc.client as abc_client
import sandbox.projects.release_machine.helpers.u_helper as rm_u_helper
import sandbox.projects.release_machine.security as rm_sec
from sandbox.projects.release_machine.components.config_core.responsibility import Responsible, Abc


LOGGER = logging.getLogger(__name__)


def get_responsible_user_login_from_abc_object(abc_object,  token=None):
    # type: (Abc, typing.Optional[str]) -> str
    abc_api = abc_client.AbcClient(token or rm_sec.get_rm_token(None))
    responsible = abc_api.get_responsible(abc_object)

    LOGGER.info("Got responsible from abc: %s", responsible)

    return responsible


def get_responsible_user_login_from_abc(responsible_object, token=None):
    # type: (Responsible, typing.Optional[str]) -> str
    return get_responsible_user_login_from_abc_object(responsible_object.abc, token=token)


def get_responsible_user_login_from_u(responsible_object, token=None):
    # type: (Responsible, typing.Optional[str]) -> str

    u_helper = rm_u_helper.UApi()
    responsible = u_helper.get_on_duty(responsible_object.u)

    LOGGER.info("Got responsible from u: %s", responsible)

    return responsible


def get_responsible_user_login(responsible_object, token=None):
    # type: (Responsible, typing.Optional[str]) -> str

    if isinstance(responsible_object, six.string_types):
        return responsible_object

    for service_name, service_fn in [
        ("abc", get_responsible_user_login_from_abc),
        ("u", get_responsible_user_login_from_u)
    ]:
        if not getattr(responsible_object, service_name, None):
            continue

        try:

            responsible = service_fn(responsible_object, token)
            if responsible:
                return responsible

            LOGGER.warning("Unable to find user in %s", service_name)

        except requests.HTTPError:
            LOGGER.exception("Unable to get responsible user from %s due to an unexpected error", service_name)
            continue

    LOGGER.warning("Fallback to hardcoded user login: %s", responsible_object.login)

    return responsible_object.login


def get_responsible_abc_service_name(responsible_object, token=None):
    # type: (typing.Union[Responsible, typing.AnyStr], typing.Optional[str]) -> typing.Optional[str]
    if isinstance(responsible_object, six.string_types):
        LOGGER.warning("Responsible does not contain info about abc: %s", responsible_object)
        return
    if not responsible_object.abc:
        LOGGER.warning("Abc service not specified: %s", responsible_object)
        return
    if responsible_object.abc.service_name:
        return responsible_object.abc.service_name
    if not responsible_object.abc.component_id:
        LOGGER.warning("Abc component_id not specified: %s", responsible_object.abc)
        return
    abc_api = abc_client.AbcClient(token or rm_sec.get_rm_token(None))
    return abc_api.get_service_info(responsible_object.abc.component_id)["slug"]
