# -*- coding: utf-8 -*-

import logging

import sandbox.sdk2 as sdk2
import sandbox.common.errors as err
from sandbox.projects.common import link_builder as lb
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.helpers import arcanum_helper
from sandbox.projects.release_machine.helpers import svn_helper
from sandbox.projects.release_machine.helpers import commit


def create_reviews(task, action_items, revs, commit_info, base_url, do_commit, merge_from_trunk=False):
    common_path = svn_helper.get_common_path(revs, base_url, merge_from_trunk)

    for action_item in action_items:
        c_info = action_item.c_info
        user = task.Parameters.commit_user or task.author  # We check commit_user, when task is launched from RM UI
        granted = c_info.check_merge_permissions(task, user) if (c_info and do_commit) else True
        if not granted:
            task.set_info(
                c_info.merges_cfg__not_granted_message(user, c_info.get_responsible_for_component()),
            )
        for merge_type in action_item.merge_types:
            if granted:
                review_result = commit.create_review(
                    task=task,
                    merge_path=merge_type,
                    common_path=common_path,
                    revs=revs,
                    checkout_rev=task.Parameters.revision_to_checkout,
                    ignore_ancestry=task.Parameters.force_merge,
                    commit_info=commit_info,
                    descr=task.Parameters.description,
                    commit_user=task.Parameters.commit_user,
                )
            else:
                review_result = rm_const.ReviewResult(rm_const.CommitStatus.failed)
            yield (merge_type, review_result)


def wait_for_tests(task):
    reviews = task.Context.reviews
    arcanum_api = arcanum_helper.ArcanumApi(token=rm_sec.get_rm_token(task))
    for review_id in reviews:
        if not review_id:
            continue
        review_info = arcanum_api.get_review_tests_statuses(review_id)['result']
        tests_status = commit.check_review_tests_statuses(review_info)
        if tests_status == rm_const.CiTestResult.failed:
            logging.info("Tests in review {review_id} failed".format(review_id=review_id))
        elif tests_status == rm_const.CiTestResult.running:
            logging.info("Tests in review {review_id} are still running".format(review_id=review_id))
            task.Context.reviews_check_attempt += 1
            task.Context.save()
            if task.Context.reviews_check_attempt >= 5:
                task.Context.reviews_check_attempt = 0
                task.Context.save()
                stop_msg = (
                    "We've been waiting 3 hours for tests in review {review_link}, "
                    "now we move task to STOPPED status".format(
                        review_link=lb.review_link(review=review_id)
                    )
                )
                task.set_info(stop_msg, do_escape=False)
                raise err.TaskStop(stop_msg)
            raise sdk2.WaitTime(30 * 60)  # Wait tests for 30 minutes
    logging.info("All tests in reviews are finished")
