from datetime import datetime
import logging
import six

import sandbox.sdk2 as sdk2
from sandbox.projects.common import decorators
from sandbox.projects.common import requests_wrapper
import sandbox.projects.release_machine.core.const as rm_const


RM_PROJECT_ID = "release_machine"
RM_SERVICE_NOTIFY_NAME = "notify"
RM_SERVICE_MONITOR_EXEC_TIME_NAME = "monitor_tasks_exec_time"
RM_SERVICE_NUMBER_COMMITS = "number_commits_in_release"
RM_SERVICE_UI_SPEED = "ui_speed_monitoring"
RM_CLUSTER_NAME_DEFAULT = "default"
RM_CLUSTER_NAME_TESTING = "testing"
RM_CLUSTER_NAME = RM_CLUSTER_NAME_DEFAULT

DATE_FMT = "%Y-%m-%dT%H:%M:%SZ"


class SolomonApi(object):
    def __init__(self, token=None, timeout=30):
        self.__token = token or sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME)
        self.timeout = timeout
        self.headers = self._set_headers(self.__token)
        self.logger = logging.getLogger(__name__)

    @staticmethod
    def _set_headers(token):
        return {
            "Content-Type": "application/json",
            "Authorization": "OAuth {}".format(token)
        }

    def _push_sensors(self, sensors, service=RM_SERVICE_NOTIFY_NAME, cluster=RM_CLUSTER_NAME):
        self.logger.debug("Post request to Solomon.\nSENSORS:\n%s\nCLUSTER: %s\nSERVICE: %s", sensors)
        return requests_wrapper.post(
            rm_const.Urls.SOLOMON_PUSH_API_URL,
            params={"project": RM_PROJECT_ID, "service": service, "cluster": cluster},
            headers=self.headers,
            json={
                "commonLabels": {"host": "solomon-push"},
                "metrics": sensors,
            },
        )

    def _add_kind_and_ts(self, sensor, kind="IGAUGE", ts=None):
        sensor["type"] = kind
        sensor["ts"] = ts or datetime.utcnow().strftime(DATE_FMT)

    def post_alert_to_solomon(self, exc):
        self.logger.exception("Post alert to Solomon: %s", exc)
        sensor = {
            "labels": {"name": "TG_notify_failures"},
            "value": 1,
        }
        self._add_kind_and_ts(sensor)
        self._push_sensors([sensor])

    def post_number_commits_in_release_to_solomon(self, num_commits, component_name):
        sensor = {
            "labels": {"name": "commits_number", "component_name": str(component_name)},
            "value": num_commits,
        }
        self._add_kind_and_ts(sensor)
        return self._push_sensors([sensor])

    def post_ui_speed_to_solomon(self, exec_time, url, ctype):

        parsed_url = six.moves.urllib_parse.urlparse(url)

        host = "{}://{}".format(parsed_url.scheme, parsed_url.netloc)
        path = parsed_url.path

        cluster = RM_CLUSTER_NAME_DEFAULT if ctype == "production" else RM_CLUSTER_NAME_TESTING

        sensor = {
            "labels": {
                "name": str(url),
                "host": host,
                "path": path,
            },
            "value": exec_time,
        }

        self._add_kind_and_ts(sensor, kind="DGAUGE")

        return self._push_sensors([sensor], service=RM_SERVICE_UI_SPEED, cluster=cluster)

    @decorators.retries(max_tries=3, delay=3, default_instead_of_raise=True)
    def post_task_exec_time_to_solomon(self, exec_time, task_type_str, component_name):
        sensor = {
            "labels": {"name": str(task_type_str), "component_name": component_name},
            "value": exec_time,
        }
        self._add_kind_and_ts(sensor)
        return self._push_sensors([sensor])
