from __future__ import unicode_literals


import json
import logging
from requests.exceptions import Timeout, HTTPError, ConnectionError

from sandbox.projects.common.decorators import retries
from sandbox.projects.release_machine.helpers.request_helper import RequestHelper

LOGGER = logging.getLogger(__name__)
SOY_API = "https://soyproxy.yandex.net/{}/soy_api/"


class SoyApi(RequestHelper):
    # will retry 6 times with sleeps (2 6 18 54 162 486) seconds consequently. 728 seconds of wait time total
    retries_decorator = retries(max_tries=6, delay=2, backoff=3, exceptions=(Timeout, HTTPError, ConnectionError))

    def __init__(
        self,
        timeout=30,
        ssl_verify=False,
        token=None,
        server="hahn",
    ):
        soy_api = SOY_API.format(server)
        super(SoyApi, self).__init__(soy_api, timeout, ssl_verify)
        self.headers = {
            'Content-type': 'application/json',
            'Accept': 'application/json',
        }
        if token:
            self.headers['Authorization'] = 'OAuth: {}'.format(token)

    @retries_decorator
    def create(self, request_json):
        """
        Creates new batch
        :return: Json answer from server
        """
        return self._do_post("create", data=json.dumps(request_json))

    @retries_decorator
    def running_operations(self):
        """
        There will be info about operations in pool
        """
        pass

    @retries_decorator
    def status(self, batch_id):
        """
        Gets status of batch with the specified id
        :return: Json answer from server
        """
        r = self._do_get("status?id={}".format(batch_id))
        LOGGER.debug("RESPONSE: %s", r)
        return r

    @retries(max_tries=3, exceptions=(Timeout, HTTPError))
    def abort(self, batch_id):
        """
        Aborts batch with the specified id
        :return: Json answer from server
        """
        r = self._do_get("abort?id={}".format(batch_id))
        LOGGER.debug("RESPONSE: %s", r)
        return r
