import logging

from sandbox.projects.common import requests_wrapper


class StaffApi(object):
    API_URL = "https://staff-api.yandex-team.ru/v3/"

    STAFF_NOT_FOUND_MESSAGE = "not found"

    def __init__(self, token=None, timeout=30, ssl_verify=False):
        self.__token = token
        self.timeout = timeout
        self.headers = self._set_headers(token)
        self.ssl_verify = ssl_verify
        self.logger = logging.getLogger(__name__)

    @staticmethod
    def _set_headers(token):
        return {
            'Authorization': 'OAuth {}'.format(token)
        }

    def get_persons(self, url, params):
        """
        Gets info from staff api, docs https://staff-api.yandex-team.ru/v3/
        :param url: str, api handle
        :param params: dict, params for request
        :return: json or raise Exception
        Retries 3 times on bad http codes, except (403 and 404)
        """
        return requests_wrapper.get_r(
            self.API_URL + url,
            params=params,
            timeout=self.timeout,
            headers=self.headers,
            verify=self.ssl_verify,
            no_retry_on_http_codes=requests_wrapper.NO_RETRY_CODES,
        ).json()

    def get_person_id(self, url, params):
        persons = self.get_persons(url, params)
        try:
            return persons["result"][0]["uid"]
        except Exception as exc:
            self.logger.error("Unable to get person id: %s", exc)

    def get_robotness(self, logins):
        return {
            i["login"]: i["official"]["is_robot"]
            for i in self.get_persons_properties_bulk(logins, ["official.is_robot", "login"])
        }

    def get_current_employment_status(self, logins):
        """
        :param logins:  str, user login
        :return: dict[str, bool] {user_login: is_currently_employed}
        """
        self.logger.debug("Getting employment statuses for logins: %s", logins)
        return {
            i["login"]: not i["official"]["is_dismissed"]
            for i in self.get_persons_properties_bulk(logins, ["official.is_dismissed", "login"])
        }

    def check_person_is_robot(self, login):
        """
        :param login: str, user login
        :return:
            True if person is robot
            False otherwise.
            None - if login is not found in staff or Staff is not accessible.
        """
        try:
            return self.get_person_profile_property(login, "official.is_robot")["official"]["is_robot"]
        except (IOError, KeyError) as exc:
            self.logger.error("Unable to find out robotness for person '%s': %s", login, exc)

    def check_person_is_external(self, login):
        """
        Return True if person is external, False else
        :param login: str, user login
        :return: True if person is robot, False otherwise, None if user not found or Staff is not accessible.
        """
        try:
            response = self.get_person_profile_property(login, "official.affiliation")
            return response["official"]["affiliation"] == "external"
        except (IOError, KeyError) as exc:
            self.logger.error("Unable to find out if person '%s' is external: %s", login, exc)

    def get_person_profile_property(self, login, property):
        """
        Returns staff api response for request with user login and property
        :param login: str, user login
        :param property: str, property from staff-api. Example: 'official.is_robot'
        :return: response in json format. Raises exception when user not found or Staff is not accessible.
        """
        params = {
            'login': login,
            '_one': 1,
            '_fields': property,
        }
        response = self.get_persons('persons', params)
        self.logger.info("Got response %s for staff api request with params %s", response, params)
        return response

    def get_persons_properties_bulk(self, logins, fields):
        """
        Returns staff api response for request with user login and property
        :param logins: Iterable[str]
        :param fields: str, property from staff-api. Example: 'official.is_robot'
        :return: response in json format.
        """
        params = {
            'login': ",".join(logins),
            '_fields': ",".join(fields),
        }
        response = self.get_persons('persons', params)
        self.logger.info("Got response %s for staff api request with params %s", response, params)
        return response.get("result", [])

    def get_person_by_telegram_login(self, telegram_login):
        """
        Returns staff login by telegram_login if found, None else
        :param telegram_login: str
        :return: str or None
        """
        params = {
            "_one": 1,
            "_query": "accounts==match({{'type':'telegram','value':'{telegram_login}'}})".format(
                telegram_login=telegram_login,
            ),
            "_fields": "login",
        }
        response = self.get_persons("persons", params)
        self.logger.info("Got response %s for staff api request with params %s", response, params)
        try:
            return response["login"]
        except (IOError, KeyError) as exc:
            self.logger.error("Unable to find out person with telegram login '%s': %s", telegram_login, exc)
            return None

    def get_user_groups(self, login):
        groups = self.get_persons('persons', {
            'login': login,
            '_one': 1,
            '_fields': 'groups.group.url',
        })
        return (x['group']['url'] for x in groups['groups'])

    def get_users_from_group(self, group):
        persons = self.get_persons('groupmembership', {
            'group.url': group,
            '_fields': 'person.login',
        })
        return (x['person']['login'] for x in persons['result'])

    def is_user_employed(self, staff_login):
        staff_info = self.get_person_profile_property(staff_login, "official.is_dismissed")
        if staff_info.get("error_message") == self.STAFF_NOT_FOUND_MESSAGE:
            return False

        return not staff_info['official']['is_dismissed']

    def filter_externals(self, logins):
        return [i for i in logins if self.check_person_is_external(i) is False]
