"""
    Here are helpers for easy usage of startrek api.
    For release-machine purposes.
    Maintainer: ilyaturuntaev@, glebov-da@
"""
import logging
import functools as ft
import requests
import six
import time

import sandbox.projects.release_machine.core as rm_core
import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.common import decorators
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import string as cs
from sandbox.projects.common import requests_wrapper
from sandbox.projects.release_machine.helpers import wiki_helper
from sandbox.projects.release_machine.helpers import staff_helper
from sandbox.common.errors import TemporaryError

LOGGER = logging.getLogger(__name__)

ISSUE_TIME_FMT = "%Y-%m-%dT%H:%M:%S.%f+0000"


def find_tickets_by_filter(st_client, c_info, st_filter):
    query = ["(Queue: {})".format(c_info.notify_cfg__st__queue)]
    query.extend(st_filter)
    query = " AND ".join(query) + " \"Sort By\": Key DESC"
    LOGGER.info("Process query: [%s]", query)
    issues = st_client.issues.find(query) or []
    LOGGER.info("Issues found: %s", ",".join([found_issue.key for found_issue in issues]))
    return issues


def find_tickets_by_tags(st_client, c_info, st_tags, open_only=False):
    query = ["(Queue: {})".format(c_info.notify_cfg__st__queue)] + ["Tags: {}".format(st_tag) for st_tag in st_tags]
    if open_only:
        query.append("Resolution: empty()")
    query = " AND ".join(query) + " \"Sort By\": Key DESC"
    LOGGER.info("Process query: [%s]", query)
    issues = st_client.issues.find(query) or []
    LOGGER.info("Issues found: %s", ",".join([found_issue.key for found_issue in issues]))
    return issues


class STHelper(object):
    ISSUE_CREATED = "created"
    ISSUE_FOUND = "found"
    ISSUE_UPDATED = "updated"
    RELEASE_NUM_TAG = rm_const.STARTREK_RELEASE_NUMBER_TAG_TEMPLATE

    def __init__(self, token, useragent=rm_const.ROBOT_RELEASER_USER_NAME):
        """
            Requirements and task tags are in RM consts file.
            in task class
        """
        self.__token = token
        self.headers = self._set_headers(token)

        try:
            import startrek_client
            self.st_client = startrek_client.Startrek(token=token, useragent=useragent)
        except ImportError:
            import yandex_tracker_client
            # `useragent` seems to be legacy parameter and was dropped
            self.st_client = yandex_tracker_client.TrackerClient(token=token)

        self.st_client._connection.session.verify = False

    @staticmethod
    def _set_headers(token):
        headers = {}
        if token:
            headers["Authorization"] = "OAuth " + token
        return headers

    @decorators.retries(3)
    def check_robot_access_for_queue(self, st_queue):
        try:
            response = requests_wrapper.get_r(
                "https://st-api.yandex-team.ru/v2/queues/{queue}/checkPermissions/write".format(
                    queue=st_queue
                ),
                headers=self.headers,
            )
        except Exception as exc:
            eh.log_exception("Unable to check st_queue permissions", exc)
            return False
        if response.status_code != requests.codes.ok:
            raise TemporaryError
        response = response.json()
        LOGGER.debug("Api permissions response: %s", response)
        has_access = response.get("users", False)
        return has_access

    @decorators.retries(3, delay=5)
    def create_issue(self, task, c_info, add_followers, add_descr, release_num):
        st_tags = c_info.st_tags
        st_tags.append(self.RELEASE_NUM_TAG.format(release_num))
        followers = c_info.get_followers(self.__token) + (add_followers or [])
        filtered_followers = staff_helper.StaffApi(self.__token).filter_externals(followers)
        kwargs = {
            "queue": c_info.notify_cfg__st__queue,
            "assignee": (
                task.author if c_info.notify_cfg__st__use_task_author_as_assignee else c_info.st_assignee
            ),
            "followers": filtered_followers,
            "summary": c_info.st_summary(release_num),
            "description": u"{}\n{}\n\n//Ticket was created by task: {task}//".format(
                c_info.st_description(release_num), add_descr, task=lb.sb_item_wiki_link(task.id, "task")
            ),
            "tags": st_tags,
            "components": c_info.notify_cfg__st__components,
            "deadline": c_info.notify_cfg__st__deadline_date,
            "followingMaillists": c_info.notify_cfg__st__following_mails,
            "unique": "_".join(st_tags)
        }
        if c_info.notify_cfg__st__ticket_type:
            kwargs["type"] = {"name": c_info.notify_cfg__st__ticket_type}
        if c_info.notify_cfg__st__abc_service:
            kwargs["abcService"] = c_info.notify_cfg__st__abc_service
        return self.st_client.issues.create(**kwargs)

    def patch_filter(self, query, st_filter):
        requests_wrapper.patch_r(
            "{}filters/{}".format(rm_const.Urls.STARTREK_API, st_filter),
            json={"query": query},
            headers={"Authorization": "OAuth {}".format(self.__token)}
        )

    def comment_results_to_st(self, issue_key, st_comment_id, text):
        issue = self.get_ticket_by_key(issue_key)
        for comment in issue.comments.get_all():
            if comment.id == st_comment_id:
                full_text = "{}\n{}".format(comment.text, text)
                LOGGER.info("Comment text: '%s'", full_text)
                comment.update(text=full_text)
                return

    @decorators.retries(3, delay=10)
    def get_ticket_by_key(self, st_key):
        return self.st_client.issues[st_key]

    def find_ticket_by_release_number(self, release_num, c_info, fail=True, open_only=False):
        if not release_num:
            LOGGER.info("Release number not specified, cannot find ticket")
            return

        if not c_info.notify_cfg__use_startrek:
            return

        st_tags = c_info.st_tags + [self.RELEASE_NUM_TAG.format(release_num)]
        issues = find_tickets_by_tags(self.st_client, c_info, st_tags, open_only=open_only)

        if not issues:
            LOGGER.debug("Cannot find issue with release_num: %s", release_num)
            if fail:
                eh.fail(u"Cannot find issue with filter: {}".format(release_num).encode("utf-8"))
            return

        if len(issues) > 1:
            LOGGER.warning("Too many issues found: %s.\nTry to use the latest one", issues)
            found_issue = max(issues, key=lambda x: int(x.key.split("-")[-1]))
        elif len(issues) == 1:
            found_issue = issues[0]

        LOGGER.info("Issue found: %s", found_issue.key)  # todo: `found_issue` can be referenced before assignment

        return found_issue

    @decorators.retries(3, delay=10)
    def find_tickets_by_filter(self, c_info, st_filter):
        if not c_info.notify_cfg__use_startrek:
            return
        if not isinstance(st_filter, (list, tuple)):
            LOGGER.debug("Bad st_filter format in Component Info, st_filter should be a list")
            return
        return find_tickets_by_filter(self.st_client, c_info, st_filter)

    @decorators.retries(3, delay=10)
    def find_tickets_by_tags(self, c_info, st_tags, open_only=False):
        if not c_info.notify_cfg__use_startrek:
            return
        if not isinstance(c_info.st_tags, (list, tuple)):
            LOGGER.debug("Bad st_tags format in Component Info, st_tags should be a list")
            return
        return find_tickets_by_tags(self.st_client, c_info, st_tags, open_only=open_only)

    def comment(self, release_num, text, c_info, summonees=None, maillist_summonees=None, fail=True):
        issue = self.find_ticket_by_release_number(release_num, c_info, fail=fail)
        if not issue:
            return
        self.create_comment(
            issue,
            text,
            summonees,
            maillist_summonees,
            c_info.notify_cfg__st__notify_on_robot_comments_to_tickets,
        )
        return issue

    def comment_by_key(self, issue_key, text):
        issue = self.get_ticket_by_key(issue_key)
        self.create_comment(issue, text)
        return issue

    @staticmethod
    @decorators.retries(3, delay=10)
    def create_comment(issue, text, summonees=None, maillist_summonees=None, notify=True):
        comment = issue.comments.create(
            params=dict(notify=notify),
            text=text,
            summonees=summonees,
            maillistSummonees=maillist_summonees
        )
        LOGGER.info("Comment added to issue [%s]: %s", issue.key, comment)

    def find_comment(self, release_num, comment_start, c_info):
        """
        :return: tuple of (Optional[comment], Optional[issue])
        """
        issue = self.find_ticket_by_release_number(release_num, c_info, fail=False)
        if issue:
            comment = self.find_comment_in_issue(issue, comment_start)
            return comment, issue
        return None, issue

    @decorators.retries(3, delay=2, backoff=1)
    def find_comment_in_issue(self, issue, comment_start):
        """
        Find comment by starting substring from issue
        :param issue: issue key or object
        :param comment_start: substring with comment start
        :return: comment object
        """
        issue = self._to_issue_obj(issue)
        for comment in issue.comments.get_all():
            if comment.text.startswith(comment_start):
                LOGGER.debug('Found comment `%s` in issue `%s`', comment, issue)
                return comment
        LOGGER.error('Cannot find comment starting with `%s` in issue `%s`', comment_start, issue)

    def _to_issue_obj(self, issue):
        if isinstance(issue, (six.text_type, six.binary_type)):
            return self.get_ticket_by_key(issue)
        elif issue is None:
            raise ValueError("Issue should not be None")
        return issue

    @decorators.retries(3, delay=10)
    def write_grouped_comment(self, group_name, title, content, release_num, c_info, summonees=None):
        """
        :param group_name: Title for grouped comment, used to find comment in ticket
        :param title: Title for hidden block
        :param content: Hidden block content
        :param release_num: Ticket release number
        :param c_info: Component, release number and component are used to find ticket
        :param summonees: Summon responsible users
        :return:
        """
        issue = self.find_ticket_by_release_number(release_num, c_info, fail=False)
        if not issue:
            LOGGER.error("Unable to update table, no issue found")
            return
        self.write_grouped_comment_in_issue(
            issue, group_name, title, content, summonees, c_info.notify_cfg__st__notify_on_robot_comments_to_tickets
        )

    def write_grouped_comment_in_issue(self, issue, group_name, title, content, summonees=None, notify=True):
        issue = self._to_issue_obj(issue)
        comment = self.find_comment_in_issue(issue, group_name)
        if title:
            message = "<{{{title}\n{content}\n}}>\n\n".format(
                title=cs.all_to_str(title),
                content=cs.all_to_str(content),
            )
        else:
            message = "{content}\n\n".format(
                content=cs.all_to_str(content),
            )
        if not comment:
            self.create_comment(
                issue, "{}\n\n{}".format(group_name, message),
                summonees=summonees,
                notify=notify,
            )
        else:
            if six.PY2:
                message = message.decode('utf-8')
            comment.update(text=comment.text + message, summonees=summonees)

    def replace_or_write_grouped_table(self, group_name, rows, release_num, c_info, wiki_text_upd_function=None):
        comment, issue = self.find_comment(release_num, group_name, c_info)
        if not issue:
            LOGGER.error("Unable to update table, no issue found")
            return None
        table_header = rm_const.TicketGroups.TicketTableHeaders.get(group_name, [" "])
        if not comment:
            new_table = u"{}\n{}".format(group_name, wiki_helper.format_table(table_header, rows))
            self.create_comment(issue, new_table, notify=c_info.notify_cfg__st__notify_on_robot_comments_to_tickets)
        else:
            if not wiki_text_upd_function:
                wiki_text_upd_function = wiki_helper.replace_table_in_text
            updated_comment = wiki_text_upd_function(comment.text, table_header, rows)
            comment.update(text=updated_comment)
        return issue

    @decorators.retries(3, delay=10)
    def replace_grouped_table(self, group_name, rows, release_num, c_info):
        return self.replace_or_write_grouped_table(
            group_name,
            rows,
            release_num,
            c_info,
            wiki_text_upd_function=wiki_helper.replace_table_in_text,
        )

    @decorators.retries(3, delay=10)
    def write_grouped_table(self, group_name, rows, release_num, c_info):
        return self.replace_or_write_grouped_table(
            group_name,
            rows,
            release_num,
            c_info,
            wiki_text_upd_function=wiki_helper.append_table_rows_in_text,
        )

    @decorators.retries(3, delay=10)
    def update_grouped_table(self, group_name, row, release_num, c_info, compare_position=0):
        if len(row) <= compare_position:
            eh.check_failed("Wrong parameters, row '{}' is too short, required length is {}.".format(
                row, compare_position + 1,
            ))
        return self.replace_or_write_grouped_table(
            group_name,
            [row],
            release_num,
            c_info,
            wiki_text_upd_function=ft.partial(wiki_helper.split_and_update_table, compare_position=compare_position),
        )

    def comment_task_problem(self, task, release_num, c_info):
        if release_num and not task.ctx.get("comment_on_break"):
            self.comment(
                release_num=release_num,
                text="Unexpected problem in task {}: {}".format(task.type, lb.task_wiki_link(task.id)),
                c_info=c_info,
            )
            task.ctx["comment_on_break"] = True

    def get_release_num_from_tags(self, tags):
        release_num = None
        for tag in tags:
            if not self.RELEASE_NUM_TAG.format('') in tag:
                continue
            release_num = tag.split(self.RELEASE_NUM_TAG.format(''))[1]
        return int(release_num) if release_num else release_num

    @decorators.retries(3, delay=10)
    def close_prev_tickets(self, c_info, release_num, custom_message):
        """ Close tickets with release numbers less than release_num """
        LOGGER.info("Try to close old tickets")

        if not c_info.notify_cfg__use_startrek:
            LOGGER.info("The component does not use startrek. Skipping")
            return

        issues = self.find_tickets_by_tags(c_info, c_info.st_tags, open_only=True)
        if not issues:
            return
        try:
            for issue in issues[1:]:
                issue_release_num = self.get_release_num_from_tags(issue.tags)
                LOGGER.debug(
                    "Check issue %s with founded release_num=%s against release_num=%s",
                    issue.key, issue_release_num, release_num,
                )
                if issue_release_num and issue_release_num >= release_num:
                    LOGGER.info(
                        "Don't close ticket '%s', release number is higher than %s", issue.key, release_num
                    )
                    continue
                self.close_issue(issue, custom_message)
        except Exception as exc:
            eh.log_exception("Something goes wrong with startrek", exc)

    @decorators.retries(3, delay=10)
    def close_linked_tickets(self, c_info, release_num):
        """ Close tickets linked to release_ticket if they are already solved """
        LOGGER.info("Try to close linked tickets")
        issue = self.find_ticket_by_release_number(release_num, c_info, fail=False, open_only=True)
        if not issue:
            return
        try:
            linked_issues = issue.links.get_all()
            for linked_issue in linked_issues:
                if linked_issue.status.key == "resolved":
                    self.close_issue(
                        self.get_ticket_by_key(linked_issue.object.key),
                        "Close ticket because it has status 'resolved' and it was released with {}".format(issue.key),
                    )
        except Exception as exc:
            eh.log_exception("Something goes wrong with startrek", exc)
        LOGGER.info("All linked and resolved tickets has been closed")

    @staticmethod
    @decorators.retries(3, delay=5, backoff=1)
    def close_issue(issue, custom_message):
        if issue.status.key == rm_const.Workflow.CLOSED:
            LOGGER.info("Ticket %s is already closed!", issue.key)
            return False
        LOGGER.info("Ticket %s has status '%s'. Closing it!", issue.key, issue.status.key)

        available_transitions = [tr.id for tr in issue.transitions.get_all()]
        if rm_const.Workflow.CLOSE in available_transitions:
            transition = issue.transitions[rm_const.Workflow.CLOSE]
        else:
            # Some tickets have closed if for closing transactions
            transition = issue.transitions[rm_const.Workflow.CLOSED]
        transition.execute(
            comment=custom_message,
            resolution='fixed',
        )
        return True

    @decorators.retries(5, delay=5)
    def execute_transition(self, c_info, release_num, transition, task=None):
        issue = self.find_ticket_by_release_number(release_num, c_info, fail=False)
        if not issue:
            return
        # Don't change production status for acceptance ticket
        if issue.status.key == transition or issue.status.key in [
            rm_const.Workflow.PRODUCTION,
            rm_const.Workflow.CLOSED,
        ]:
            LOGGER.debug("Ticket already has status '%s' or ticket has production/closed status", transition)
            return True

        task_marker = "[done by task: {}]".format(lb.task_wiki_link(task.id)) if task else ""
        for _ in range(len(c_info.notify_cfg__st__workflow)):
            # Execute transition if it exists in available transitions
            available_transitions = [tr.id for tr in issue.transitions.get_all()]
            LOGGER.debug(
                "Issue status %s, available_transitions %s", issue.status.key, ", ".join(available_transitions)
            )
            if transition in available_transitions:
                LOGGER.info("Execute transition '%s'.", transition)
                self.create_comment(
                    issue,
                    "Move ticket status from {prev_status} to {current_status}\n{task_marker}".format(
                        prev_status=issue.status.key,
                        current_status=transition,
                        task_marker=task_marker,
                    ),
                    notify=c_info.notify_cfg__st__notify_on_robot_comments_to_tickets,
                )
                self._execute_transition(issue, transition)
                return True

            # Choose and execute next transition in workflow to set desirable transition
            c_transition = c_info.notify_cfg__st__workflow.get(issue.status.key, None)
            LOGGER.debug(
                "Issue status %s, available_transitions %s", issue.status.key, ", ".join(available_transitions)
            )
            if c_transition not in available_transitions:
                LOGGER.warning("Bad workflow for transition '%s'", c_transition)
                return False

            LOGGER.info("Execute transition '%s'.", c_transition)
            self._execute_transition(issue, c_transition)
            time.sleep(2)
            issue = self.st_client.issues[issue.key]
        return False

    @decorators.retries(3)
    def _execute_transition(self, issue, transition):
        if transition == rm_const.Workflow.CLOSE:
            issue.transitions[transition].execute(resolution="fixed")
        else:
            issue.transitions[transition].execute()

    @decorators.retries(3, delay=5, default_instead_of_raise=True, default_value=rm_core.Error())
    def change_assignee(self, c_info, release_num, assignee=None):
        issue = self.find_ticket_by_release_number(release_num, c_info, fail=False)
        if issue is None:
            msg = "Can't update assignee. Issue is not found"
            LOGGER.warning(msg)
            return rm_core.Error(msg)
        if not assignee:
            assignee = c_info.get_responsible_for_release()
            if c_info.notify_cfg__st__assignee_after_acceptance:
                assignee = c_info.notify_cfg__st__assignee_after_acceptance
        LOGGER.info("Update assignee from '%s' to '%s'.", issue.assignee.login, assignee)
        issue.update(assignee=assignee)
        return rm_core.Ok()

    @classmethod
    def generate_ticket_creation_link(
        cls, queue='', summary='', description='', issue_type='', assignee='', priority=''
    ):
        """
        Returns a link to Startrek ticket creation form with fields pre-set according to the kwargs values

        :param queue: the name of Startrek queue
        :param summary: ticket title
        :param description: ticket description
        :param issue_type: ticket type (integer)
        :param assignee: ticket assignee
        :param priority: ticket priority
        """
        url = "{host}/{path}".format(
            host=rm_const.Urls.STARTREK.strip('/'),
            path='createTicket',
        )
        query_dict = {
            'queue': queue,
            'summary': summary,
            'description': description,
            'type': issue_type,
            'assignee': assignee,
            'priority': priority,
        }

        return "{url}?{params}".format(
            url=url,
            params=six.moves.urllib.parse.urlencode(query_dict),
        )
