"""
    Here are helpers for easy usage of svn.
    For release-machine purposes.
    Maintainer: ilyaturuntaev@, glebov-da@
"""

import abc
import logging
import os
import re
import six

import sandbox.common.errors as err
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import string
from sandbox.projects.release_machine import core as rm_core
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.helpers import merge_helper
from sandbox.sdk2.vcs import svn
from sandbox.sdk2 import ssh

TRUNK = 'trunk'
TRUNK_PATH = os.path.join(svn.Arcadia.ARCADIA_BASE_URL, TRUNK)
BRANCH_PATH = os.path.join(svn.Arcadia.ARCADIA_BASE_URL, "branches")
_VALID_PATHS = [
    # arc
    ':/arc/branches/',
    ':/arc/tags/',
    # robots
    ':/robots/branches/',
    ':/robots/tags/',
]
# The first seven columns in the output are each one character wide,
# and each column gives you information about a different aspect of each working copy item.
# http://svnbook.red-bean.com/en/1.8/svn.ref.svn.c.status.html
FILE_STATUS_LENGTH = 7

LOGGER = logging.getLogger(__name__)


class InitRevision(object):
    __slots__ = ("rev", "msg")

    def __init__(self, rev, msg=None):
        self.rev = rev  # could be negative, representing rollbacks
        self.msg = msg or []

    def __str__(self):
        return str(self.rev)

    __repr__ = __str__


class SvnHelper(object):
    @staticmethod
    def get_last_trunk_revision_before_copy(
        path,
        repo_base_url=svn.Arcadia.ARCADIA_BASE_URL, trunk_url=TRUNK_PATH,
        max_copy_depth=5,  # seems, that 5 nested branches is enough to cover all possible use cases
    ):
        """
        Get last trunk revision contained in branch or tag.

        trunk - r1 - ... - r123 - r124 - ....
        ...                     \
        branch_i                r125 - ... - r222 - ...
        ...                                        \
        path                                   r226 - ...

        Given path, return last trunk revision (r123 in this example)
        """
        LOGGER.info("Try to get last trunk revision for: %s", path)
        LOGGER.info("Looking for copy from: %s", trunk_url)
        if not path.startswith(repo_base_url):
            LOGGER.error("Path [%s] should start with '%s'", path, repo_base_url)
            return None
        result_rev, curr_path = None, path
        for _ in range(max_copy_depth):
            if curr_path.startswith(trunk_url):
                return result_rev
            log_on_copy = svn.Arcadia.log(
                curr_path,
                revision_from="r0", revision_to="HEAD",
                stop_on_copy=True, track_copy=True, limit=1
            )
            if not log_on_copy:
                return None
            first_commit_after_copy = log_on_copy[0]
            copies = first_commit_after_copy.get("copies")
            LOGGER.info("Got copies: %s", copies)
            if not copies:
                return None
            result_rev = int(copies[0][1])
            curr_path = repo_base_url + copies[0][0]
        return None

    @classmethod
    def is_branching(cls, vcs_info):
        return "[branch:" in vcs_info["msg"]  # todo: find better heuristic

    @classmethod
    def is_tagging(cls, vcs_info):
        return "[tag:" in vcs_info["msg"]  # todo: find better heuristic

    @classmethod
    def get_initial_revs(cls, vcs_info):
        if cls.is_branching(vcs_info):
            return []
        if cls.is_tagging(vcs_info):
            return []
        revs = [InitRevision(int(vcs_info["revision"]))]
        prev_indent = 0

        # find merges
        for line in vcs_info["msg"].split("\n"):
            include = rm_const.INCLUDE_REV_RE.search(line)
            if include is not None:
                indent = len(include.group(1).replace(" ", ""))
                rev = include.group(2)
                if indent > prev_indent:
                    # this revision is not initial, so we don't need it
                    revs.pop()
                revs.append(InitRevision(int(rev)))
                prev_indent = indent
            else:  # append message
                line = line.strip("> .-")
                if line:
                    revs[-1].msg.append(line)

        # find rollbacks among found revs
        revs = list(cls._get_rollbacks(revs))
        # todo: find rollbacks of rollbacks (it will require requests to vcs for some info)

        LOGGER.debug("Initial revisions for %s: %s", vcs_info["revision"], revs)
        return revs

    @classmethod
    def _get_rollbacks(cls, revs):
        for rev in revs:
            rollbacks = merge_helper.get_rollbacks_from_commit_message("\n".join(rev.msg))
            if rollbacks:
                for rollback_init_rev in rollbacks:
                    yield InitRevision(-rollback_init_rev)
            else:
                yield rev

    @classmethod
    def extract_commit_info(cls, revs, svn_path):
        diff_resolvers = set()
        committers = set()
        rb_review_ids = set()

        for rev in revs:
            rev_log = svn.Arcadia.log(svn_path, rev, limit=1)
            if not rev_log:
                LOGGER.error("No revision %s for %s: revision log is empty", rev, svn_path)
                raise Exception(
                    "No revision {} for {}: revision log is empty. "
                    "Perhaps you're trying to roll back trunk revision from branch, "
                    "you should specify in-branch revision instead. See SEARCH-4869 "
                    "for details. ".format(rev, svn_path)
                )

            rev_log = rev_log[0]

            LOGGER.debug("Rev log: %s", rev_log)
            stripped_msg = rev_log["msg"].replace(" ", "")
            if rm_const.DIFF_RESOLVER_MASK in stripped_msg:
                # resolver will be pasted automatically
                # with commit message for revision
                rm = rm_const.DIFF_RESOLVER_RE.search(stripped_msg)
                if rm:
                    diff_resolvers.add(rm.group(1))

            committers.add(rev_log["author"])
            review_nums = rm_const.REVIEW_RE.findall(stripped_msg)
            for review_id in review_nums:
                rb_review_ids.add(review_id)

        if not diff_resolvers:
            diff_resolvers = committers

        diff_resolvers = list(diff_resolvers)
        committers = list(committers)
        rb_review_ids = list(rb_review_ids)

        LOGGER.info("Diff resolvers: %s", diff_resolvers)
        LOGGER.info("Committers: %s", committers)
        LOGGER.info("Review ids: %s", rb_review_ids)
        return CommitInfo(diff_resolvers, committers, rb_review_ids)

    @staticmethod
    def revisions_merged(branch_path, revision_to=None):
        """
            Get list of branch revisions
            :param branch_path: string, path to branch in svn
            :param revision_to: revision in branch till which svn log is taken. If None log stops on HEAD.
            :return: list of revisions in branch
        """
        arc_log = svn.Arcadia.log(branch_path, 'r0', revision_to or 'HEAD', stop_on_copy=True)
        return [i["revision"] for i in arc_log]

    @staticmethod
    def iter_tags(branch_path, revision_to=None, reverse=False):
        """
            Get generator of branch revisions (of pairs tag_index, rev)
            :param branch_path: string, path to branch in svn
            :param revision_to: revision in branch till which svn log is taken. If None log stops on HEAD.
            :return: generator of revisions in branch
        """
        revs = SvnHelper.revisions_merged(branch_path, revision_to)
        rev_amount = len(revs)
        iter_range = range(rev_amount, 1, -1) if reverse else range(2, rev_amount + 1)
        for index in iter_range:
            yield index, revs[index - 1]

    @staticmethod
    def list_files_with_pattern(parent_path, pattern):
        """
        :param parent_path: SVN path to inspect (e.g. 'arcadia:/arc/branches/userdata')
        :param pattern: string with regular expression
        :return: list of matched names
        """
        LOGGER.info("Listing folders under '%s' with pattern '%s'..", parent_path, pattern)
        regex = re.compile(pattern)
        result = []
        for folder in svn.Arcadia.list(parent_path, as_list=True):
            m = regex.match(folder)
            if m:
                result.append(m.group(0))
        LOGGER.info("Found %d files with pattern '%s'", len(result), pattern)
        return result

    @staticmethod
    def get_highest_folder(path, regexp="^stable-([0-9]+)/", only_num=False):
        """
            :param path: SVN path to inspect (e.g. 'arcadia:/arc/branches/middle')
            :param regexp: should find all correct folder numbers using findall
            :param only_num: should return full folder name or only number?
            :rtype: string
        """
        LOGGER.info("Get highest folder number using regexp: %s", regexp)
        regex = re.compile(regexp)
        folders = svn.Arcadia.list(path, as_list=True)
        LOGGER.debug("Got %d folders", len(folders))
        folder_matches = (regex.match(folder) for folder in folders)
        existing_folders = [f for f in folder_matches if f]
        if existing_folders:
            # get max folder or folder number
            highest = max(existing_folders, key=lambda x: int(x.group(1))).group(int(only_num))
            LOGGER.info("Max existing folder: %s", highest)
        else:
            highest = '0' if only_num else 'stable-0'
            LOGGER.info("Folder with specified regexp doesn't exist! Return 'default': %s", highest)
        return highest

    @staticmethod
    def read_file(url):
        return svn.Arcadia.cat(url)

    @staticmethod
    def get_parent_branch(branch_path):
        """
        :param branch_path: SVN branch path (e.g. 'arcadia:/arc/branches/userfeat/0')
        :return: SVN path of the parent branch
        """
        prefix = "arcadia:/arc"
        assert branch_path.startswith(prefix)
        relpath = branch_path[len(prefix):]

        commits = svn.Arcadia.log(
            branch_path,
            revision_from="HEAD",
            revision_to="1",
            limit=10000,
            stop_on_copy=True,
            track_copy=True
        )
        LOGGER.info("Retrieved '%s' history of %d commit(s)", branch_path, len(commits))
        copies = commits[-1]["copies"]

        LOGGER.info("Checking %d copy operation(s) of the oldest commit..", len(copies))
        for from_path, from_rev, path in copies:
            if path == relpath:
                LOGGER.info(
                    "Found a suitable copy operation: %s:%s -> %s",
                    from_path, from_rev, path
                )
                return prefix + from_path
        raise ValueError("Failed to find parent of %s" % branch_path)


class CommitInfo(object):
    def __init__(self, diff_resolvers=None, committers=None, rb_review_ids=None):
        self.diff_resolvers = diff_resolvers or []
        self.committers = committers or []
        self.rb_review_ids = rb_review_ids or []


@six.add_metaclass(abc.ABCMeta)
class CommitPath(object):
    """Base interface for merge and rollback."""

    def __init__(self):
        self.ram_path = None
        self.local_path = None

    @abc.abstractproperty
    def path_to(self):
        pass

    @abc.abstractproperty
    def path_from(self):
        pass

    @property
    def local(self):
        if self.ram_path:
            return "{}/{}".format(str(self.ram_path), self.local_path)
        return self.local_path

    def arc_merge_func(self, arc_client, arcadia_src_dir, arc_hash):
        """
        Function to perform cherry_pick or revert with arc_hash

        :param arc_client: instance for working with Arcadia repository through Arc Vcs.
        :param arcadia_src_dir: path where arc mounted
        :param arc_hash: commit hash, etc. to cherry-pick/revert
        """
        pass

    def __repr__(self):
        name = getattr(self, "name", "unknown")
        return "<{} {}>".format(self.__class__.__name__, name)

    def __str__(self):
        return repr(self)


class RollbackPath(CommitPath):
    """ Base interface for different types of rollback. """
    action = merge_helper.Action.ROLLBACK

    @staticmethod
    def format_revs(revs):
        return ['-{}'.format(rev) for rev in revs]

    def path_to(self):
        return self.full

    def path_from(self):
        return self.full

    @property
    def component_name(self):
        return "Arcadia"

    def arc_merge_func(self, arc_client, arcadia_src_dir, arc_hash):
        arc_client.revert(arcadia_src_dir, commit=arc_hash, allow_empty=True)


class MergePath(CommitPath):
    """ Base interface for different types of merge. """

    action = merge_helper.Action.MERGE

    def path_to(self):
        return self.full

    def path_from(self):
        return TRUNK_PATH

    @property
    def component_name(self):
        return "Arcadia"

    @staticmethod
    def format_revs(revs):
        return revs

    def arc_merge_func(self, arc_client, arcadia_src_dir, arc_hash):
        arc_client.cherry_pick(arcadia_src_dir, commit=arc_hash, add_commit_name=True, allow_empty=True)


class RollbackTrunkPath(RollbackPath):
    def __init__(self, c_info=None, ram_path=None):
        self.name = TRUNK
        self.c_info = c_info
        self.ram_path = ram_path
        self.local_path = "local_branch_{}".format(TRUNK)
        self.short = TRUNK
        self.full = TRUNK_PATH


class RollbackReleaseMachinePath(RollbackPath):
    def __init__(self, branch_num, c_info, ram_path=None):
        self.name = str(branch_num)
        self.c_info = c_info
        self.ram_path = ram_path
        self.branch_num = branch_num
        self.local_path = "local_branch_{}".format(self.branch_num)
        self.short = c_info.component_branch_name(self.branch_num)
        self.full = c_info.full_branch_path(self.branch_num)

    @property
    def component_name(self):
        return self.c_info.name


class RollbackCustomPath(RollbackPath):
    def __init__(self, full, ram_path=None):
        self.name = full
        self.c_info = None
        self.ram_path = ram_path
        self.local_path = "local_branch_custom"
        self.short = full
        self.full = full


class MergeReleaseMachinePath(MergePath):
    def __init__(self, branch_num, c_info, ram_path=None):
        self.c_info = c_info
        self.ram_path = ram_path
        self.name = str(branch_num)
        self.branch_num = branch_num
        self.local_path = "local_branch_{}".format(self.branch_num)
        self.short = c_info.component_branch_name(self.branch_num)
        self.full = c_info.full_branch_path(self.branch_num)

    @property
    def component_name(self):
        return self.c_info.name


class MergeCustomPath(MergePath):
    def __init__(self, full, ram_path=None):
        self.name = full
        self.c_info = None
        self.ram_path = ram_path
        self.local_path = "local_branch_custom"
        self.short = full
        self.full = full


class MergeResult(object):
    def __init__(self):
        self.successes = []
        self.failures = []
        self.result_revs = []

    def add_success(self, merge_path):
        self.successes.append(merge_path)

    def add_failure(self, merge_path):
        self.failures.append(merge_path)

    def all(self):
        return {
            merge_helper.CommitResult.SUCCEEDED: self.successes,
            merge_helper.CommitResult.FAILED: self.failures,
        }

    def update(self, commit_status, merge_path, revision=None):
        if commit_status in [rm_const.CommitStatus.success, rm_const.CommitStatus.changeset_is_empty]:
            self.successes.append(merge_path)
        elif commit_status == rm_const.CommitStatus.failed:
            self.failures.append(merge_path)
        if revision:
            self.result_revs.append(str(revision))

    def is_ok(self):
        return not self.failures


def filter_text_mods(diff_paths):
    """
    Return paths from diff_paths which have real text changes (text-mods=True).
    If action for path is `A`, `D` or `R` we should add it anyway. If action is `M` and text-mods == True we also should
    add it. In other cases we should skip this path.
    :param diff_paths: list of dicts with keys `action`, `text-mods`, `prop-mods`, `text`, `kind`.
    :return: list of 2-tuples, (action, path).
    """
    filtered_paths = []
    LOGGER.debug("Got diff paths before filtering %s", diff_paths)
    for path_entry in diff_paths:
        if path_entry["action"] != "M":
            filtered_paths.append((path_entry["action"], path_entry["text"]))
        else:
            if path_entry["text-mods"]:
                filtered_paths.append((path_entry["action"], path_entry["text"]))
    LOGGER.debug("Diff paths after filtering %s", filtered_paths)
    return filtered_paths


def get_paths_for_revisions(full_branch_path, revs):
    all_paths = []
    for rev in revs:
        try:
            full_rev_logs = svn.Arcadia.log(full_branch_path, rev, fullpath=True)
        except svn.SvnTemporaryError as exc:
            eh.log_exception("Temporary svn error. Try to restart task later", exc)
            raise
        except svn.SvnError as exc:
            eh.log_exception("Svn error, something goes wrong with Arcadia.log.", exc)
            eh.check_failed("Svn error. Please, check input parameters")
        eh.ensure(
            full_rev_logs,
            "Revision {rev} does not exist in {path}, did you mistype branch number? "
            "See SEARCH-2101 for possible resolution. ".format(
                rev=rev,
                path=full_branch_path,
            )
        )
        filtered_paths = filter_text_mods(full_rev_logs[0].get("paths"))
        if filtered_paths:
            LOGGER.info("Diff of r%s:\n%s", rev, string.shift_list_right(filtered_paths))
            all_paths.extend(merge_helper.prepare_diff_items(filtered_paths))
        else:
            LOGGER.warning("Diff is empty for revision %s", rev)
    return all_paths


def get_common_path(revs, full_branch_path, merge_from_trunk=False):
    """
        Extract common path that given revisions touch.
        :param revs: Revisions list to analyze
        :param full_branch_path: full path to branch (e.g. arcadia:/arc/branches/middle/stable-126)
        :param merge_from_trunk: True when performing merge from trunk
        :return: common merge path that is always started from 'arcadia' element
    """
    all_paths = get_paths_for_revisions(full_branch_path, revs)
    eh.ensure(all_paths, "No paths to merge")
    max_common_path = merge_helper.get_max_common_path(all_paths)

    if merge_from_trunk:
        # fail if trunk/arcadia is not in common path
        eh.ensure("trunk" in max_common_path, "Cannot merge commit outside of trunk")

    eh.ensure(
        "arcadia_tests_data" not in max_common_path,
        (
            "Cannot commit anything into arcadia_tests_data, it is too heavy. "
            "If you still using arcadia_tests_data, you're doing wrong. "
            "Ask mvel@ if you still have any questions. "
        )
    )
    min_rev = min(revs)
    info = svn.Arcadia.info(os.path.join(svn.Arcadia.ARCADIA_BASE_URL, *max_common_path) + "@{}".format(min_rev))

    if info:
        if info['entry_kind'] == 'file':
            max_common_path = max_common_path[:-1]
            LOGGER.info("Max common path is a file, made level up %s", max_common_path)
    else:
        LOGGER.info("Can't get svn info for path %s, rev %s, probably it has been deleted", max_common_path, min_rev)

    common_path = max_common_path[1:]
    LOGGER.info("Common path: %s", common_path)
    return common_path


def get_ssh_key(task, key_owner, key_name):
    try:
        return ssh.Key(task, key_owner=key_owner, key_name=key_name)
    except err.VaultNotFound:
        raise err.VaultNotFound(
            "Key with name '{key_name}' and owner '{key_owner}' "
            "is not available for task owner '{task_owner}'.\n"
            "Please, ask somebody from '{key_owner}' for access".format(
                key_owner=key_owner,
                key_name=key_name,
                task_owner=task.owner if task else "???"
            )
        )


def save_files_with_conflicts(task, paths):
    """
    Write diff from files with conflicts to resource
    :param task: MergeToStable or RollbackCommit
    :param paths: list with paths
    """
    if not task.Parameters.merge_conflicts.path.exists():
        task.Parameters.merge_conflicts.path.mkdir()
    for path in paths:
        path_diff = svn.Arcadia.diff(path)
        conflict_file_path = task.Parameters.merge_conflicts.path / path.replace('/', '__')
        fu.write_file(conflict_file_path, path_diff)
    LOGGER.debug("All conflict files saved")


def process_svn_status_output(task, status_str, path):
    """
    Parse svn status output, save conflicts to resource and put results in task info
    :param task: MergeToStable or RollbackCommit
    :param status_str: svn.Arcadia.status output
    :param path: local path checkouted from
    :return: CommitStatus entity
    """
    if not status_str:
        task.set_info("Nothing was merged (changeset is empty)")
        return rm_const.CommitStatus.changeset_is_empty
    if "Summary of conflicts:" in status_str:
        LOGGER.debug("Got some conflicts:\n%s", status_str)
        paths_with_conflicts = []
        missing_items = []
        for file_status in status_str.split("\n"):
            item_name = file_status[FILE_STATUS_LENGTH:].strip()
            if file_status[0] == "C":
                paths_with_conflicts.append(item_name)
            elif file_status[0] == "!":
                missing_items.append(item_name)
        task.set_info(
            "Local path was checkouted from: {path}\n"
            "There are some conflicts while merging:\n\nConflicts:\n{conflicts}\n\n"
            "Missing items:\n{missing_items}".format(
                path=path,
                conflicts="\n".join(paths_with_conflicts),
                missing_items="\n".join(missing_items),
            )
        )
        if paths_with_conflicts:
            save_files_with_conflicts(task, paths_with_conflicts)
        return rm_const.CommitStatus.failed
    else:
        LOGGER.debug("There are no conflicts in svn status")
        modified_paths = []
        not_modified_paths = []
        for file_status in status_str.split("\n"):
            if file_status[0] == " ":
                LOGGER.debug("Path %s has no text changes", file_status[8:])
                not_modified_paths.append(file_status)
            else:
                LOGGER.debug("Path %s was modified", file_status[8:])
                modified_paths.append(file_status)
        parsed_status = "Paths with important changes:\n{modified_paths}\n".format(
            modified_paths="\n".join(modified_paths),
        )
        if not_modified_paths:
            parsed_status += (
                "Paths with `no modifications` (see svn status --help for additional info):\n"
                "{not_modified_paths}".format(
                    not_modified_paths="\n".join(not_modified_paths),
                )
            )
        task.set_info("Local path was checkouted from: {path}\n{parsed_status}".format(
            path=path,
            parsed_status=parsed_status,
        ))
        return rm_const.CommitStatus.success


def parse_merged_commit_msg(msg):
    BASE_START = "Merge from trunk: "
    INCLUDE_START = "  INCLUDE:"
    POINTS_SEP = " .................................................................\n"
    ARROW_SEP = " >"
    ENDLINE_SEP = "\n"

    commit_msg = []
    if not msg.startswith(BASE_START) or POINTS_SEP not in msg:
        return msg
    try:
        msg_copy = msg.replace(ARROW_SEP, "")
        msg_copy = msg_copy.split(POINTS_SEP)[1]
        msg_copy = msg_copy.split(ENDLINE_SEP)
        for line in msg_copy:
            if line.startswith(INCLUDE_START):
                continue
            commit_msg.append(line.strip())
        return ENDLINE_SEP.join(commit_msg)
    except Exception as e:
        LOGGER.exception("Failed to parse commit message", e)
        return msg


def make_intermediate_svn_dirs(path, component_name, ssh_key=None):
    path = path.rstrip("/")
    LOGGER.info("Making intermediate svn dirs for path: `%s`", path)

    if not any([i in path for i in _VALID_PATHS]):
        return rm_core.Error("Invalid path: {}. Valid path should contain any of these parts: {}".format(
            path, _VALID_PATHS,
        ))

    ssh_key = ssh_key or get_ssh_key(None, rm_const.COMMON_TOKEN_OWNER, rm_const.ARC_ACCESS_TOKEN_NAME)

    parent_path = os.path.dirname(path)

    with ssh_key:
        # svn mkdir fails when path already exists (even with --parents specified)
        if svn.Arcadia.check(parent_path, user=rm_const.ROBOT_RELEASER_USER_NAME):
            return rm_core.Ok("Parents path already exists, do nothing")

        svn.Arcadia.mkdir(
            parent_path,
            user=rm_const.ROBOT_RELEASER_USER_NAME,
            message='Creating intermediate {dir_type} folder for component {component_name}'.format(
                dir_type='tag' if 'tags' in path else 'branch',
                component_name=component_name,
            ),
            parents=True,
        )
    return rm_core.Ok("Parents path successfully created")
