from __future__ import unicode_literals

from sandbox.projects.common.decorators import retries
from sandbox.projects.release_machine.helpers.request_helper import RequestHelper


class VcsIndexerClient(RequestHelper):
    API_URL = "https://vcs-indexer.yandex-team.ru/"
    DEFAULT_TIMEOUT = 10  # sec
    INTERVAL_BATCH_LIMIT = 10000

    @retries(3)
    def batch_info(self, revisions, base_path=None, sandbox_mode=True, need_paths=True):
        """
        Get info for list of revisions.

        :param revisions: list of revisions or single revision
        :param base_path: path to filter log with
        :param sandbox_mode: compatibility with Arcadia.log format
        :param need_paths: return info with paths
        :return: list of log entry dicts with keys:
            - revision
            - author
            - msg
            - paths:
                - [sandbox_mode=True] a list of pairs (action, path)
                - [sandbox_mode=False] a list of dicts with keys (action, path)
        """
        if not revisions:
            return []
        params = {
            "revisions": ",".join(map(str, revisions)),
            "sandbox_mode": "yes" if sandbox_mode else "no",
            "need_paths": "yes" if need_paths else "no",
        }
        if base_path:
            params["base_path"] = base_path
        return self._do_get("batch_info", params=params).get("logentries", [])

    @retries(3)
    def interval_info(self, start_revision, end_revision, base_path=None, sandbox_mode=True, need_paths=True):
        """
        Get info for interval of revisions.

        :param start_revision: first revision of interval
        :param end_revision: last revision of interval
        :param base_path: path to filter log with
        :param sandbox_mode: compatibility with Arcadia.log format
        :param need_paths: return info with paths
        :return: list of log entry dicts with keys:
            - revision
            - author
            - msg
            - paths:
                - [sandbox_mode=True] a list of pairs (action, path)
                - [sandbox_mode=False] a list of dicts with keys (action, path)
        """
        result = []
        params = {
            "start_revision": start_revision,
            "end_revision": end_revision,
            "sandbox_mode": "yes" if sandbox_mode else "no",
            "need_paths": "yes" if need_paths else "no",
        }
        if base_path:
            params["base_path"] = base_path
        offset = 0
        for _ in range(1000):
            params.update({
                "limit": self.INTERVAL_BATCH_LIMIT,
                "offset": offset
            })
            batch_info = self._do_get("batch_info", params=params)
            if batch_info["error_code"] == "E_FAIL":
                raise RuntimeError(batch_info["error"])
            result.extend(batch_info["logentries"])
            if int(batch_info["total_entries"]) <= len(result):
                return result
            self.logger.info(
                "Result length = %d. Total length = %s. Continue requesting ...",
                len(result), batch_info["total_entries"]
            )
            offset += self.INTERVAL_BATCH_LIMIT
        raise RuntimeError("Probably, infinite loop found: unable to get batch info in 1000 iterations")
