# -*- coding: utf-8 -*-
"""
    Here are helpers for easy wiki posting.
    For release-machine purposes.
    Maintainer: ilyaturuntaev@, glebov-da@
"""
from __future__ import unicode_literals

import json
import logging
import itertools
import six

import sandbox.common.errors as err
import sandbox.projects.common.string as str_utils
import sandbox.projects.release_machine.helpers.changelog_helper as rm_changelog
from sandbox.projects.common import decorators
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.release_machine.helpers.request_helper import RequestHelper

WIKI_API = "https://wiki-api.yandex-team.ru/_api/frontend/"
NO_ISSUE = "NO TRACKER ISSUE SPECIFIED"


def format_table(header, rows):
    return '\n#|\n{}\n{}\n|#\n'.format(format_table_header(header), format_table_rows(rows))


def screen_pipe(row):
    return str_utils.all_to_unicode(row).strip().replace('|', '""|""')


def format_table_row(row):
    return '|| {} ||'.format(' | '.join(map(screen_pipe, row)))


def format_table_rows(rows):
    return "\n".join([format_table_row(r) for r in rows])


def format_table_header(row):
    return '|| {} ||'.format(' | '.join(['**{}**'.format(str_utils.all_to_unicode(i)) if i else '' for i in row]))


def parse_table_row(row):
    return row.strip("||").split(" | ")


def parse_table_into_rows(table):
    return table.replace('||||', '||\n||').strip().split('||\n||')


def replace_table_in_text(text, table_header, rows):
    """
    Finds the first table in `text` (if any) and
    replaces it with new one constructed from the
    given `table_header` and `rows`

    :param text: origin text
    :param table_header: table header (list)
    :param rows: table rows (list of lists)
    :return: `text` with the first table replaced with the new one
    """
    pre_table, _, post_table = text.partition("|#")
    pre_table, _, table = pre_table.partition("#|")
    return "{pre_table}{table}{post_table}".format(
        pre_table=pre_table,
        table=format_table(table_header, rows).strip(),
        post_table=post_table,
    )


def append_table_rows_in_text(text, table_header, rows):
    """
    Attempts to find a table in `text` with header equal to `table_header`.
    Adds `rows` to the existing table if found. Otherwise creates new
    table with the given header and rows and appends it to the given text.

    :param text: original text
    :param table_header: table header (list)
    :param rows: table rows (list of lists)
    :return: `text` with table rows added
    """
    formatted_table_header = format_table_header(table_header)
    if formatted_table_header in text:
        pre_table, _, post_table = text.partition("|#")
        updated_text = "{pre_table}\n{rows}\n|#{post_table}".format(
            pre_table=pre_table.strip(),
            rows=format_table_rows(rows),
            post_table=post_table,
        )
    else:
        updated_text = text + format_table(table_header, rows)
    return updated_text


def get_page_without_redirect(session, url):
    """
    Get 1 redirected url for passed wiki page, if redirect doesn't exist - return passed url
    :param session: Session obj with auth headers and etc.
    :param url: wiki page
    :return: wiki page url
    """
    page = session.get(url).json()
    if "redirect_to_tag" in page["data"]:
        redirect = page["data"]["redirect_to_tag"]
        return WIKI_API + redirect.strip("/") + "/"
    return url


@decorators.retries(3, delay=30)
def get_raw_wiki_page(session, url):
    try:
        url = get_page_without_redirect(session, url)
    except Exception:
        logging.warning("Error while retrieving redirected url")
    raw_url = "{}/.raw".format(url.rstrip("/"))
    raw_text = session.get(raw_url).text
    logging.debug("Raw text wiki page (%s) :\n%s\n", url, raw_text)
    raw_text = json.loads(raw_text)
    if "error" in raw_text:
        logging.error("Error retrieving wiki page text: %s", raw_text["error"]["message"])
    if "data" not in raw_text:
        raise err.TaskFailure("Unable to get data from url: {}".format(url))

    if "body" not in raw_text["data"]:
        logging.error("No 'body' key in wiki response, try to find redirect")
        if "redirect_to_tag" in raw_text["data"]:
            redirect = raw_text["data"]["redirect_to_tag"]
            logging.info("Found redirect %s", redirect)
            result = get_raw_wiki_page(session, WIKI_API + redirect.strip("/") + "/")
            return result
        else:
            raise err.TaskFailure("Unable to get 'body' from url data: {}".format(raw_text["data"]))
    return raw_text["data"]["body"], url


@decorators.retries(3, delay=30)
def post_on_wiki_and_check(session, url, data, fail_on_bad_status=True):
    logging.debug("==== WIKI POST REQUEST ====\n%s\n%s\n\n", url, data)
    response = session.post(url, json=data)
    if fail_on_bad_status:
        response.raise_for_status()
    if int(response.status_code / 100) != 2:
        logging.info("Got strange status code: %s", response.status_code)


def st_tickets(change):
    return sorted(change["startrek_tickets"]) or [NO_ISSUE]


def group_changes_by_tickets(c_info, changes):
    grouped_changes = itertools.groupby(sorted(changes, key=st_tickets), key=st_tickets)
    changes_rows = []
    for tickets, changes_of_ticket in grouped_changes:
        changes_rows.append(["\n".join(["**{}**".format(t) for t in tickets])])
        changes_rows.extend([
            rm_changelog.ChangeLogFormatter.startrek_row_grouped_by_ticket(c_info, i)
            for i in changes_of_ticket
        ])
    return format_table(
        header=rm_changelog.ChangeLogFormatter.startrek_header_grouped_by_ticket,
        rows=changes_rows
    )


def split_and_update_table(text, table_header, rows, compare_position):
    # Split comment message into 3 part, beginning, table body and ending
    pre_table, _, table = text.encode("utf-8").partition("#|")
    table, _, post_table = table.partition('|#')

    if not table:
        return append_table_rows_in_text(text, table_header, rows)

    new_table = table
    for row in rows:
        new_table = insert_or_update_row_in_table(new_table, row, compare_position)

    message = "{p_table}\n#|\n{table}\n|#{pt_table}".format(
        p_table=str_utils.all_to_str(pre_table).strip(),
        table=new_table,
        pt_table=str_utils.all_to_str(post_table),
    )

    return str_utils.all_to_str(message)


def insert_or_update_row_in_table(table, row, compare_position=0):
    parsed_table = parse_table_into_rows(table)
    new_table = []
    # Flag to check if new row was inserted
    inserted_row = False
    for t_row in parsed_table:
        t_row = t_row.strip("||")
        parsed_row = parse_table_row(t_row)
        # Check if row has enough length to be compared
        if len(parsed_row) <= compare_position:
            eh.check_failed("Bad table format, row '{}' is too short. Required length is {}".format(
                parsed_row, compare_position + 1
            ))
        # If elements in compare position are equal - update row
        if parsed_row[compare_position].strip() != row[compare_position].strip():
            new_table.append(str_utils.all_to_str("||{}||".format(t_row)))
            continue
        inserted_row = True
        new_row = []
        # Insert value from new row if value is not null, else use old value
        for f_val, s_val in six.moves.zip_longest(row, parsed_row, fillvalue=""):
            new_row.append(f_val if f_val.strip() else s_val)
        new_table.append(str_utils.all_to_str(format_table_row(new_row)))
    # Append new row to the end of the table
    if not inserted_row:
        new_table.append(str_utils.all_to_str(format_table_row(row)))

    logging.debug("New table %s", new_table)
    return '\n'.join(new_table)


class WikiApi(RequestHelper):
    def __init__(self, token=None, timeout=30, ssl_verify=False):
        super(WikiApi, self).__init__(WIKI_API, timeout, ssl_verify)
        self.__token = token
        self.logger = logging.getLogger(__name__)
        self.headers = self._set_headers(token)

    @staticmethod
    def _set_headers(token):
        headers = {'Content-type': 'application/json'}
        if token:
            headers['Authorization'] = 'OAuth ' + token
        return headers

    @decorators.retries(3, 5)
    def post_page(self, url, data=None):
        self._do_post(url, data)

    @decorators.retries(3, 5, default_instead_of_raise=True)
    def delete_page(self, url):
        self._do_delete(url)

    @decorators.retries(3, 5)
    def check_wiki_path_correctness(self, url):
        logging.debug("Current url %s", url)
        parsed_url = url.strip("/").split("/")
        if not len(parsed_url):
            eh.check_failed("Wiki path is empty!")
        parent_url = "/".join(parsed_url[:-1]) + "/"
        logging.debug("Parsed url %s, parent url %s", parsed_url, parent_url)
        if not self.exists(parent_url):
            eh.check_failed("Parent directory is incorrect: {}".format(parent_url))
        #  Now we see, that parent directory exists, so if our path doesn't exist, we can create it.

    def get_page_owner(self, url):
        try:
            r = self._do_get("{}/.owner".format(url.strip()))
            return r["data"]["owner"]["login"]
        except Exception as exc:
            self.logger.warning("Cannot get current user login: %s", exc)

    def get_page_parent_cluster(self, url):
        """
        Get parent cluster for the given url

        :param url: wikipage url
        :return: A tuple of parent cluster dict and parent url (always followed by '/')
        """
        try:
            parent_url = url.strip("/").split("/")[:-1] + "/"
            r = self._do_get("{}.treecluster?sort_order=title".format("/".join(parent_url)))
            return r.get('data', []), parent_url
        except Exception as ex:
            self.logger.warning("Cannot get tree for {}".format(url), ex)
        return [], ""

    def set_page_authors(self, url, new_authors_list, append=True):
        """
        Set new page authors for the given wiki page (`url`).

        :param url: wiki page url
        :param new_authors_list: list of new authors uids
        :param append: whether to preserve current authors or not
        """
        if not new_authors_list:
            self.logger.warning("Attempt to add/set an empty authors list")
            if not append:
                self.logger.error("Cannot set page authors to an empty list")
                return
        try:
            if append:
                resp = self._do_get("{}/.authors".format(url.strip()))
                current_authors = [author.get('uid') for author in resp.get('data').get('authors', [])]
                new_authors_list = new_authors_list[:]
                new_authors_list = current_authors + new_authors_list
            new_authors_list = list(set(new_authors_list))
            self._do_post(
                "{}/.authors".format(url.strip()),
                data=json.dumps({
                    "authors": new_authors_list
                })
            )
        except Exception as exc:
            eh.log_exception("Cannot set authors", exc)
