"""
Here are all common notification possibilities:
    * e-mails
    * sms
    * startrek comments
WARNING: please, don't put here dependencies from:
    projects/release_machine/components
Otherwise, it could produce cyclic imports and add lots of unnecessary dependencies.
Maintainers: mvel@, ilyaturuntaev@, glebov-da@
"""
from __future__ import unicode_literals


import logging
import six
from six.moves.urllib import (
    request,
    parse,
)

import sandbox.common.rest as sandbox_rest
import sandbox.common.types.notification as ctn
from sandbox.projects.common import decorators
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import string
from sandbox.projects.common.telegram_api import notify
from sandbox.projects.messenger.bot import QBot
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.helpers import solomon_helper
from sandbox.projects.release_machine.helpers.staff_helper import StaffApi


LOGGER = logging.getLogger(__name__)


def mail_recipients(recipients, humans_only=True, task=None):
    """

    :param recipients: group of people in arbitrary format: list, string, set ...
    :param humans_only: if True - return only non-robots according to staff api
    :param task: sandbox task object
    :return: list of logins and mailboxes. Ex: ["release-machine-cc"]
    """
    if isinstance(recipients, (six.text_type, six.binary_type)):
        recipients = {recipients}
    if not hasattr(recipients, '__iter__'):
        raise TypeError(
            "Recipients for e-mail have wrong type (not iterable): type({}) = {}",
            recipients, type(recipients)
        )
    mail_tail = "@yandex-team.ru"
    mail_tail_length = len(mail_tail)
    logins = {i[:-mail_tail_length] if i.endswith(mail_tail) else i for i in recipients}
    if task:
        logins = logins | {task.author}
    if humans_only:
        staff_api = StaffApi(rm_sec.get_rm_token(None))
        robotness = staff_api.get_robotness(logins)
        # some logins could be just common mailboxes, robotness is not defined for them
        logins = [login for login in logins if not robotness.get(login)]

    LOGGER.info("E-mail recipients: %s", ", ".join(logins))
    return list(logins)


def mail_body(task, message):
    """
    :param task: Sandbox task object
    :param message: message in HTML format as list of strings or string, separated by \n symbols.
    :return:
    """
    if isinstance(message, (six.text_type, six.binary_type)):
        message = [message.replace("\n", "<br/>\n")]
    tail = [
        "Sandbox task: {}.".format(lb.task_link(task.id)),
        "Task author: {}".format(lb.staff_link(string.all_to_str(task.author))),
        "--",
        "Please do not hesitate to reply to this letter in case of bug reports or any other questions.",
        "Release Machine Team will guide you.",
        "Virtually yours, {}".format(lb.HREF_TO_ITEM.format(link="https://nda.ya.ru/3UW83a", name="Release Machine")),
    ]
    body = "<br/>\n".join(message + tail)
    LOGGER.info("Sending e-mail with body:\n%s", body)
    return "<html><body>{}</body></html>".format(body)


def email2(task, recipients, subject, body):
    """SDK-agnostic e-mail sender for Release-Machine."""
    recipients = mail_recipients(recipients, humans_only=True, task=task)

    if not recipients:
        LOGGER.warning('Recipients are empty (or were filtered), email sending skipped')
        # this notification is for debug weird things like RMDEV-1118, disabled now
        if False:
            telegram(
                task=task,
                message="Recipients are empty, can't send email",
                chat_ids=[rm_const.RM_USERS["rm_maintainers"].telegram],
            )

    try:
        client = sandbox_rest.Client()
        client.notification(
            recipients=recipients + ["release-machine-cc"],
            subject="[release-machine] {}".format(subject),
            body=mail_body(task, message="{}".format(body)),
            headers=[
                "Reply-to: search-components@yandex-team.ru",
            ],
            transport=ctn.Transport.EMAIL,
            type=ctn.Type.HTML,
            task_id=task.id,
        )
        return True, ""
    except Exception as exc:
        LOGGER.exception("Got exception while posting email: %s", exc)
        telegram(
            task=task,
            message='Got exception while posting email',
            chat_ids=[rm_const.RM_USERS["rm_maintainers"].telegram],
        )
        return False, "{}".format(exc)


def sms(message, receivers):
    LOGGER.debug("Sending sms: '%s'", message)
    for user in receivers:
        try:
            sms_url = "http://golem.yandex-team.ru/api/sms/send.sbml?resps={}&msg={}".format(
                user, parse.quote(message, safe="_:/-.&")
            )
            request.urlopen(sms_url)
        except Exception as e:
            eh.log_exception("Unable to send sms to '{}'".format(user), e, logger=LOGGER)


def telegram(task, message, chat_ids=(), people=(), bot=None, pin=False, **kwargs):
    try:
        solomon_post_alert_method = solomon_helper.SolomonApi().post_alert_to_solomon
    except Exception as exc:
        LOGGER.error("Unable to get solomon api for final_hook, don't use it, exc: %s", exc)
        solomon_post_alert_method = None

    @decorators.retries(max_tries=2, default_instead_of_raise=True, final_hook=solomon_post_alert_method)
    def telegram_func(task, message, chat_ids=(), people=(), bot=None, pin=False, **kwargs):
        bot = bot or notify.create_bot(task, rm_const.TELEGRAM_TOKEN_NAME, rm_const.COMMON_TOKEN_OWNER)
        if bot is None:
            LOGGER.error("Unable to get telegram bot. Skip telegram notifications")
            return
        if chat_ids:
            notify.notify_chats(task, bot, list(chat_ids), message, pin, **kwargs)
        else:
            notify.notify_people(task, bot, list(people), message, pin, **kwargs)

    return telegram_func(task=task, message=message, chat_ids=chat_ids, people=people, bot=bot, pin=pin, **kwargs)


def q_messenger(task, message, chat_ids):
    token = rm_sec.get_rm_token(task, token_name=rm_const.Q_MESSENGER_TOKEN_NAME)
    bot = QBot(token)
    for chat_id in chat_ids:
        bot.send_message(chat_id, message)
