import typing  # noqa
import json
import logging

from six import string_types
from sandbox.projects.release_machine.components.config_core import release_approvements
from sandbox.projects.release_machine.components.config_core import responsibility
from sandbox.projects.release_machine.helpers import responsibility_helper


LOGGER = logging.getLogger(__name__)


class ApprovementDescriptionFormat(object):
    KEY_TAG_NAME = "tag_name"
    KEY_COMPONENT_NAME = "component_name"

    def __init__(self, description):
        self._description = description

    def format(self, **kwargs):

        format_dict = {key: kwargs.pop(key, "") for key in {self.KEY_TAG_NAME, self.KEY_COMPONENT_NAME}}

        LOGGER.info("Format dict: %s", format_dict)

        return self._description.format(**format_dict)


def release_approvement_stage_to_dict(stage):
    """
    :param stage: release approvement stage
    :type stage: typing.Union[release_approvements.ReleaseApprovementStage, string_types, responsibility.Responsible]
    """

    if isinstance(stage, string_types) or isinstance(stage, responsibility.Responsible):
        return release_approvement_stage_to_dict(release_approvements.ReleaseApprovementStage(stage))

    if not isinstance(stage, release_approvements.ReleaseApprovementStage):
        raise TypeError("Unexpected `stage` type {}".format(type(stage)))

    if isinstance(stage.stage, list):
        return {
            stage.NEED_ALL_KEY: stage.need_all,
            stage.STAGES_KEY: [release_approvement_stage_to_dict(s) for s in stage.stage],
        }

    if isinstance(stage.stage, string_types):
        return {
            stage.APPROVER_KEY: stage.stage,
        }

    if isinstance(stage.stage, responsibility.Responsible):
        return {
            stage.APPROVER_KEY: responsibility_helper.get_responsible_user_login(stage.stage),
        }

    raise TypeError("Unexpected `stage.stage` type {}".format(type(stage.stage)))


def release_approvements_settings_to_dict(settings, ticket_key, author, groups=None, **kwargs):
    """
    :param settings: release approvements RM settings
    :type settings: release_approvements.ReleaseApprovementsSettings
    """
    return {
        "type": "tracker",
        "object_id": ticket_key,
        "text": ApprovementDescriptionFormat(settings.description).format(**kwargs),
        "stages": [release_approvement_stage_to_dict(stage) for stage in settings.stages],
        "author": author,
        "groups": groups or [],
        "is_parallel": settings.is_parallel,
    }


def release_approvements_settings_to_json(*args, **kwargs):
    return json.dumps(release_approvements_settings_to_dict(*args, **kwargs))
