from __future__ import unicode_literals

import logging
import json
import six
import re
import typing

LOGGER = logging.getLogger(__name__)


class TableData(object):
    def __init__(self, header, rows):
        # type: (typing.Optional[typing.Tuple], typing.List[typing.Tuple]) -> None
        row_lengths = list(set(len(row) for row in rows))
        if len(row_lengths) > 1:
            raise ValueError("All rows should have same number of cells, got length: {}".format(row_lengths))
        if header and row_lengths and len(header) != row_lengths[0]:
            raise ValueError("Header should have same number of cells as rows")
        self.header = header
        self.rows = rows

    def update(self, other, using_column_as_key=None):
        # type: (TableData, typing.Optional[int]) -> None
        self.header = other.header
        if using_column_as_key is None:
            LOGGER.info("Update table by replacing rows")
            self.rows = other.rows
        else:
            LOGGER.info("Update table by updating rows using column #%s as key", using_column_as_key)
            rows_dict_new = {row[using_column_as_key]: row for row in other.rows}
            updated = set()
            for i in range(len(self.rows)):
                key = self.rows[i][using_column_as_key]
                if key in rows_dict_new:
                    self.rows[i] = rows_dict_new[key]
                    updated.add(key)
            for new_key in set(rows_dict_new.keys()) - updated:
                self.rows.append(rows_dict_new[new_key])

    @classmethod
    def from_json(cls, table_data_in_json):
        return cls(tuple(table_data_in_json["header"]), [tuple(i) for i in table_data_in_json["rows"]])

    def as_dict(self):
        return {
            "header": self.header,
            "rows": self.rows,
        }


class TableFormatter(object):
    @staticmethod
    def _format_table(table_header, table_body):
        # type: (str, str) -> str
        raise NotImplementedError

    @staticmethod
    def _format_row(table_row):
        raise NotImplementedError

    def _format_header_cell(self, header_cell_content):
        raise NotImplementedError

    def _format_cell(self, cell_content):
        raise NotImplementedError

    @staticmethod
    def _stack_rows(rows):
        return "\n".join(rows)

    @staticmethod
    def _stack_cells(cells):
        return "".join(cells)

    def format_table(self, table_data):
        # type: (TableData) -> str
        table_header = self._format_row(
            table_row=self._stack_cells(
                [self._format_header_cell(header_cell) for header_cell in table_data.header]
            )
        )
        table_body = self._stack_rows(
            [
                self._format_row(
                    table_row=self._stack_cells([self._format_cell(cell) for cell in row])
                ) for row in table_data.rows
            ]
        )
        return self._format_table(table_header, table_body)


class TableFormatterWiki(TableFormatter):
    @staticmethod
    def _format_table(table_header, table_body):
        return "\n#|\n{}\n{}\n|#\n".format(table_header, table_body)

    @staticmethod
    def _format_row(table_row):
        return "|| {} ||".format(table_row)

    @staticmethod
    def _stack_cells(cells):
        return " | ".join(cells)

    def _format_header_cell(self, header_cell_content):
        return "**{}**".format(self._format_cell(header_cell_content))

    def _format_cell(self, cell_content):
        return six.text_type(cell_content).strip().replace("|", '""|""')


class TableFormatterHtml(TableFormatter):
    @staticmethod
    def _format_table(table_header, table_body):
        return "<html><body><table border=1>\n{}\n{}\n</table></body></html>".format(table_header, table_body)

    @staticmethod
    def _format_row(table_row):
        return "<tr>{}</tr>".format(table_row)

    def _format_cell(self, cell_content):
        return "<td style=\"padding: 4px\">{}</td>".format(cell_content)

    def _format_header_cell(self, header_cell_content):
        return "<th>{}</th>".format(header_cell_content)


class UpdatableTablesWiki(object):
    _MARKER = "<# <!-- {} --> #>"

    def __init__(self, comment_marker, tables):
        # type: (str, typing.List[typing.Tuple[str, str, TableData]]) -> None
        self.comment_marker = comment_marker
        self.tables = tables

    @property
    def marker(self):
        return self._MARKER.format(self.comment_marker)

    @classmethod
    def from_wiki(cls, tables_in_wiki):
        json_str, _ = re.search(r"%%\(comments\)(.*)%%(.*)", tables_in_wiki, flags=re.DOTALL).groups()
        return cls.from_json(json.loads(json_str.strip()))

    @classmethod
    def from_json(cls, tables_in_json):
        tables = [
            (
                table_data_json["table_key"],
                table_data_json["table_comment"],
                TableData.from_json(table_data_json["table_content"])
            )
            for table_data_json in tables_in_json["tables"]
        ]
        return cls(tables_in_json["comment_marker"], tables)

    def update(self, other, using_column_as_key=0):
        # type: (UpdatableTablesWiki, typing.Optional[int]) -> None
        self.comment_marker = other.comment_marker
        new_tables = []
        new_tables_dict = {i[0]: i for i in other.tables}
        updated = set()
        for old_table in self.tables:
            table_key, _, old_table_content = old_table
            if table_key in new_tables_dict:
                _, new_table_comment, new_table_content = new_tables_dict[table_key]
                old_table_content.update(new_table_content, using_column_as_key=using_column_as_key)
                new_tables.append((table_key, new_table_comment, old_table_content))
                updated.add(table_key)
        for new_key in set(new_tables_dict.keys()) - updated:
            new_tables.append(new_tables_dict[new_key])
        self.tables = new_tables

    def _to_json(self):
        return {
            "comment_marker": self.comment_marker,
            "tables": [
                {
                    "table_comment": table_comment,
                    "table_content": table_data.as_dict(),
                    "table_key": table_key,
                } for table_key, table_comment, table_data in self.tables
            ]
        }

    def _to_wiki(self):
        return "===== {}\n\n{}".format(
            self.comment_marker,
            "\n".join(
                "{}{}".format(table_comment, TableFormatterWiki().format_table(table_data))
                for _, table_comment, table_data in self.tables
            )
        )

    def __str__(self):
        return "\n".join([
            self.marker,
            "%%(comments)\n{}\n%%".format(
                json.dumps(self._to_json(), indent=2, sort_keys=True, separators=(',', ': '))
            ),
            self._to_wiki(),
            self.marker,
        ])
