from sandbox.projects.release_machine import input_params2 as rm_params
from sandbox.projects.release_machine.core import task_env
from sandbox.projects.release_machine.tasks import base_task as rm_bt

from sandbox import sdk2
from sandbox.projects.common import binary_task


MODE_TESTID_APPROVEMENTS = "testid_approvements"
MODE_EXPERIMENT_APPROVEMENTS = "experiment_approvements"


class AbExperimentsApprovementCreator(rm_bt.BaseReleaseMachineTask):

    class Requirements(task_env.TinyRequirements):
        pass

    class Parameters(rm_params.DefaultReleaseMachineParameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)

        mode = sdk2.parameters.String(
            "Approvement creator mode",
            choices=[
                (MODE_TESTID_APPROVEMENTS, MODE_TESTID_APPROVEMENTS),
                (MODE_EXPERIMENT_APPROVEMENTS, MODE_EXPERIMENT_APPROVEMENTS),
            ],
            default_value=MODE_TESTID_APPROVEMENTS,
        )

        ticket_key = sdk2.parameters.String("Ticket Key", required=True)

        ok_secret = sdk2.parameters.YavSecret(
            "OK API token secret",
            default_value=sdk2.parameters.YavSecret.cast("sec-01ejz9hcr8tq4mmwga719ecgt0#robot_ok_api_token"),
        )

        approvement_text = sdk2.parameters.String("Approvement text", required=True)
        comment_text = sdk2.parameters.String("StarTrek comment text")

        with mode.value[MODE_TESTID_APPROVEMENTS]:

            scope_number = sdk2.parameters.Integer("Scope number", required=True)
            tag_number = sdk2.parameters.Integer("Tag number (optional)")

            approvers = sdk2.parameters.List("approvers", required=True)

    class Context(rm_bt.BaseReleaseMachineTask.Context):
        approvement_creator_state = "wait_start"
        launch_number = 0

    @property
    def approvement_link_verbose(self):
        return '<a href="https://ok.yandex-team.ru/approvements/{uuid}/">{uuid}</a>'.format(
            uuid=self.Context.approvement_uuid,
        ) if self.Context.approvement_uuid else "NOT CREATED"

    def testid_approvement_header(self):
        approvers = [
            '<a href="https://staff.yandex-team.ru/{approver}">{approver}</a>'.format(approver=approver)
            for approver in self.Parameters.approvers
        ]

        return """
                <div>
                    <div>
                        <a href="https://st.yandex-team.ru/{ticket_key}">{ticket_key}</a>
                    </div>
                    <div>
                        <strong>Approvers:&nbsp;</strong>{approvers}
                    </div>
                    <div>
                        <strong>State:&nbsp;</strong><span style="text-transform: uppercase;">{state}</span>
                    </div>
                    <div>
                        <strong>Approvement:&nbsp;</strong>{approvement_link}
                    </div>
                </div>
                """.format(
            ticket_key=self.Parameters.ticket_key,
            approvers=", ".join(approvers),
            state=self.Context.approvement_creator_state or "finished",
            approvement_link=self.approvement_link_verbose,
        )

    def experiment_approvement_header(self):

        return """
                    <div>
                        <div>
                            <a href="https://st.yandex-team.ru/{ticket_key}">{ticket_key}</a>
                        </div>
                        <div>
                            <strong>State:&nbsp;</strong><span style="text-transform: uppercase;">{state}</span>
                        </div>
                        <div>
                            <strong>Approvement:&nbsp;</strong>{approvement_link}
                        </div>
                    </div>
                """.format(
            ticket_key=self.Parameters.ticket_key,
            state=self.Context.approvement_creator_state or "finished",
            approvement_link=self.approvement_link_verbose,
        )

    @sdk2.header(title="General Info")
    def header(self):

        if self.Parameters.mode == MODE_EXPERIMENT_APPROVEMENTS:
            return self.experiment_approvement_header()

        return self.testid_approvement_header()

    def on_execute(self):

        rm_bt.BaseReleaseMachineTask.on_execute(self)

        from release_machine.public.ok import approvement_creator_state_machine

        self.Context.launch_number += 1
        self.Context.save()

        while self.Context.approvement_creator_state is not None:

            if self.Parameters.mode == MODE_EXPERIMENT_APPROVEMENTS:
                state_module = approvement_creator_state_machine.experiment_approvement_states
            else:
                state_module = approvement_creator_state_machine.testid_approvement_states

            state_cls = state_module.STATES.get(
                self.Context.approvement_creator_state,
            )

            self._check_state_cls(state_cls)

            state_obj = state_cls(task=self, test=self.Parameters.debug_mode)

            state_obj.handle()

    def _check_state_cls(self, state_cls):
        if not state_cls:
            raise ValueError(
                "Unacceptable value for approvement_creator_state context field: {}".format(
                    self.Context.approvement_creator_state,
                )
            )

        from release_machine.public.ok import approvement_creator_state_machine

        if not issubclass(state_cls, approvement_creator_state_machine.base.ApprovementCreatorState):
            raise TypeError("Unexpected creator state class: {}".format(state_cls))
