import re
import logging

import sandbox.sdk2 as sdk2
import sandbox.projects.common.binary_task as binary_task
import sandbox.projects.common.decorators as decorators
import sandbox.projects.common.file_utils as fu
import sandbox.projects.common.time_utils as tu
import sandbox.projects.common.vcs.arc as arc_api
import sandbox.projects.release_machine.arc_helper as arc_helper
import sandbox.projects.release_machine.components.all as rmc
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.input_params2 as rm_params
import sandbox.projects.release_machine.resources as rm_res
import sandbox.projects.release_machine.tasks.base_task as rm_bt

try:
    from release_machine.common_proto import events_pb2, component_state_pb2
    from release_machine.release_machine.proto.structures import message_pb2
    from release_machine.public import events as events_public
except ImportError:
    pass


class ArcMergesCrawler(rm_bt.BaseReleaseMachineTask):
    class Requirements(task_env.TinyRequirements):
        disk_space = 2 * 1024  # 2 Gb

    class Parameters(rm_params.ComponentName2):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        component_name_filter = sdk2.parameters.String("Component name filter")
        branch_number = sdk2.parameters.Integer("Branch number, use only with component_name_filter")
        use_sync_conflict_resolver = sdk2.parameters.Bool(
            "Resolve conflict between svn and arc automatically",
            default=False,
        )

        with sdk2.parameters.Output():
            merge_conflicts = sdk2.parameters.Resource(
                "Directory with merge conflicts",
                resource_type=rm_res.MergeConflicts,
            )

    @decorators.memoized_property
    def rm_event_client(self):
        from release_machine.release_machine.services.release_engine.services.Event import EventClient
        return EventClient.from_address(rm_const.Urls.RM_HOST)

    @decorators.memoized_property
    def arc_client(self):
        return arc_api.Arc(secret_name=rm_const.COMMON_TOKEN_NAME, secret_owner=rm_const.COMMON_TOKEN_OWNER)

    def on_enqueue(self):
        rm_bt.BaseReleaseMachineTask.on_enqueue(self)
        with self.memoize_stage.merge_conflicts_dir:
            self.Parameters.merge_conflicts = rm_res.MergeConflicts(
                task=self,
                description="Directory with merge conflicts",
                path="merge_conflicts",
            )

    def on_finish(self, prev_status, status):
        self._save_merge_conflicts()
        super(rm_bt.BaseReleaseMachineTask, self).on_finish(prev_status, status)

    def on_break(self, prev_status, status):
        self._save_merge_conflicts()
        super(rm_bt.BaseReleaseMachineTask, self).on_break(prev_status, status)

    def _save_merge_conflicts(self):
        if not self.Parameters.merge_conflicts.path.exists():
            fu.write_file(self.Parameters.merge_conflicts.path, "There are no conflicts here")
        sdk2.ResourceData(self.Parameters.merge_conflicts).ready()

    def _get_component_branches(self, all_branches, branch_path_pattern, lower_bound_branch_num):
        component_branch_nums = []
        found_branches = filter(None, (branch_path_pattern.search(branch) for branch in all_branches))
        logging.info(
            "Found branches by pattern '%s': %s",
            branch_path_pattern.pattern, [i.group(0) for i in found_branches]
        )
        for found in found_branches:
            found_branch_num = int(found.group(1))
            logging.debug("FOUND: %s", found_branch_num)
            if self.Parameters.branch_number:
                if found_branch_num == self.Parameters.branch_number:
                    component_branch_nums.append(found_branch_num)
            else:
                if lower_bound_branch_num <= found_branch_num:
                    component_branch_nums.append(found_branch_num)
                else:
                    logging.debug(
                        "Branch with number %s is too old, lower_bound_branch_num is %s",
                        found_branch_num, lower_bound_branch_num,
                    )
        return component_branch_nums

    def _process_branch(self, c_info, branch_num):
        logging.info("Start processing branch with number %s from component %s", branch_num, c_info.name)
        return arc_helper.sync_branch_from_arc(self, self.arc_client, c_info, branch_num)

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)

        all_arc_branches = arc_helper.get_all_branches(self.arc_client)

        name_filter_re = re.compile(self.Parameters.component_name_filter or ".*", re.IGNORECASE)

        for c_name in filter(name_filter_re.match, rmc.get_component_names()):

            c_info = rmc.get_component(c_name)

            if c_info.release_cycle_type != rm_const.ReleaseCycleType.BRANCH:
                self._report_component_state(
                    component_name=c_info.name,
                    messages=[],
                    ok=True,
                )
                continue

            if not c_info.svn_cfg__use_arc:
                continue

            component_process_result = rm_const.CommitResult(rm_const.CommitStatus.success)
            component_process_messages = []

            logging.info("Start processing %s", c_info.name)
            branch_path_pattern = re.compile(c_info.svn_cfg__arc_branch_path_pattern)
            lower_bound_branch_num = c_info.last_branch_num - 5
            component_arc_branch_nums = self._get_component_branches(
                all_arc_branches,
                branch_path_pattern,
                lower_bound_branch_num,
            )

            component_arc_branch_nums += c_info.merges_cfg.lts_branches
            # make sure branch values are unique
            component_arc_branch_nums = set(component_arc_branch_nums)

            for branch_num in component_arc_branch_nums:
                try:

                    current_result = self._process_branch(c_info, branch_num)

                    # We do NOT want to overwrite failures
                    if component_process_result.status != rm_const.CommitStatus.failed:
                        component_process_result = current_result

                    if current_result.status == rm_const.CommitStatus.failed:
                        component_process_messages.append(
                            "Sync failed for branch {}. Problem revision: {}".format(
                                branch_num,
                                current_result.revision,
                            ),
                        )

                except Exception as exc:

                    msg = "Got exception while sync branch {branch} in component {comp_name}: {exc}".format(
                        branch=branch_num,
                        comp_name=c_info.name,
                        exc=exc,
                    )

                    component_process_result = rm_const.CommitResult(rm_const.CommitStatus.failed)
                    component_process_messages.append(msg)

                    self.set_info(msg)

            logging.info("Component process result: %s", component_process_result)

            self._report_component_state(
                component_name=c_info.name,
                messages=component_process_messages,
                ok=component_process_result.status in {
                    rm_const.CommitStatus.success,
                    rm_const.CommitStatus.changeset_is_empty,
                },
            )

    def _report_component_state(self, component_name, messages, ok=False):
        """
        :param component_name: component name
        :type component_name: str
        :param messages: message list
        :type messages: list
        :param ok: whether the state/result is ok or not
        :type ok: bool
        """

        logging.info("Going to post component state (arc sync) for %s", component_name)
        logging.info("Got OK = %s and %d messages", ok, len(messages))

        now = tu.datetime_utc()
        now_ts = tu.datetime_to_timestamp(now)

        logging.debug("Now: %s", now)

        event = events_pb2.EventData(
            general_data=events_pb2.EventGeneralData(
                hash=events_public.get_event_hash(
                    now.isoformat(),
                    component_name,
                    "UpdateComponentState",
                    "arc_sync",
                ),
                component_name=component_name,
                referrer="sandbox_task:{}".format(self.id),
            ),
            task_data=events_pb2.EventSandboxTaskData(
                task_id=self.id,
                status=self.status,
                created_at=self.created.isoformat(),
                updated_at=self.updated.isoformat(),
            ),
            update_component_state_data=events_pb2.UpdateComponentStateData(
                component_state=component_state_pb2.ComponentState(
                    timestamp=int(now_ts),
                    referrer="{}:{}".format(self.type, self.id),
                    arc_sync=component_state_pb2.ComponentSectionState(
                        status=(
                            component_state_pb2.ComponentSectionState.Status.OK if ok
                            else component_state_pb2.ComponentSectionState.Status.CRIT
                        ),
                        info=component_state_pb2.ComponentSectionInfo(
                            title=("ARC->SVN sync errors" if not ok else "All OK"),
                            description="\n".join(messages),
                        ),
                    ),
                ),
            ),
        )

        logging.info("Constructed event: %s", event)

        self.rm_event_client.post_proto_events(message_pb2.PostProtoEventsRequest(events=[event]))
