# -*- coding: utf-8 -*-
import contextlib
import datetime
import os
import tarfile
import time

import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr
from sandbox.projects.common import binary_task
from sandbox.projects.common import task_env
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.search import bugbanner2 as bb2
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.common.yappy import resources as ui_resources
from sandbox.projects.common.yappy.utils import node_manager
from sandbox.projects.release_machine.release_engine import resources as rm_resources
from sandbox import sdk2


class BuildRmUi(nanny.ReleaseToNannyTask2, bb2.BugBannerTask, binary_task.LastBinaryTaskRelease):
    class Parameters(sdk2.Task.Parameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        kill_timeout = 20 * 60  # 20 min
        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl(
            "Svn url for arcadia", required=True,
            default_value="arcadia:/arc/trunk/arcadia/release_machine/release_machine/ui"
        )
        nodejs_archive = sdk2.parameters.Resource(
            "Node JS", resource_type=ui_resources.NodeJsArchive, state=(ctr.State.READY,), required=True
        )
        npm_build_command = sdk2.parameters.String("npm build command", default="build_production")
        dist_path = sdk2.parameters.String("npm build command", default="dist")

    class Requirements(task_env.TinyRequirements):
        dns = ctm.DnsType.DNS64
        disk_space = 1024  # 1Gb

    def on_execute(self):
        super(BuildRmUi, self).on_execute()
        self.add_bugbanner(bb2.Banners.ReleaseMachine)
        with arcadia_sdk.mount_arc_path(
            self.Parameters.checkout_arcadia_from_url,
            use_arc_instead_of_aapi=True
        ) as arcadia_src_dir:
            node = node_manager.NodeManager(self, "http://npm.yandex-team.ru")
            node.setup(arcadia_src_dir, self.Parameters.nodejs_archive)
            node.install_modules(cwd=arcadia_src_dir)
            node.run_script(self.Parameters.npm_build_command, cwd=arcadia_src_dir)

            output_archive_name = "bundle.tar"
            with tarfile.TarFile(output_archive_name, mode="w") as tar:
                tar.add(
                    os.path.join(arcadia_src_dir, self.Parameters.dist_path),
                    self.Parameters.dist_path
                )
            sdk2.ResourceData(rm_resources.ReleaseEngineUI(self, "RM UI", output_archive_name)).ready()

    @contextlib.contextmanager
    def info_section(self, name):
        start = time.time()
        exception = None
        try:
            yield
        except Exception as e:
            exception = e
        finally:
            delta = datetime.timedelta(seconds=time.time() - start)
            delta -= datetime.timedelta(microseconds=delta.microseconds)

            state = 'DONE'
            message = '({}) {{}} {}'.format(delta, name)
            if exception:
                state = 'FAIL'
                message = '{}: {}: {}'.format(message, exception.__class__.__name__, exception)

            try:
                message = message.format(state)
            except Exception as e:
                message = 'info_section ({}) crashed (state={}): {}: {}'.format(name, state, e.__class__.__name__, e)

            self.set_info(message)

            if exception:
                raise exception
