import sys
import logging

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common.search import bugbanner2 as bb2
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import task_env
from sandbox.projects.release_machine.components.configs import all as all_cfg


if sys.version_info[0:2] >= (3, 8):  # python 3.8 or higher
    from functools import cached_property
else:
    from sandbox.projects.common.decorators import memoized_property as cached_property


class CheckStartrekReleaseTicket(bb2.BugBannerTask, binary_task.LastBinaryTaskRelease):
    class Requirements(task_env.StartrekRequirements):
        pass

    class Parameters(binary_task.LastBinaryReleaseParameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.String("Component name", default="") as component_name:
            for c_name in all_cfg.get_all_names():
                component_name.values[c_name] = component_name.Value(value=c_name)

        startrek_ticket = sdk2.parameters.String("Startrek ticket")
        release_number = sdk2.parameters.Integer("Release number")

    @cached_property
    def c_cfg(self):
        return all_cfg.get_config(self.Parameters.component_name)

    @cached_property
    def startrek_token(self):
        return rm_sec.get_rm_token(self)

    @cached_property
    def startrek_client(self):
        import startrek_client
        return startrek_client.Startrek(token=self.startrek_token, useragent=rm_const.ROBOT_RELEASER_USER_NAME)

    @cached_property
    def startrek_release_number_tag(self):
        return rm_const.STARTREK_RELEASE_NUMBER_TAG_TEMPLATE.format(self.Parameters.release_number)

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)
        bb2.BugBannerTask.on_save(self)

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)
        bb2.BugBannerTask.on_execute(self)

        if not self.c_cfg.notify_cfg.use_startrek:
            self.set_info(
                "Component {} does not use Startrek. Not going to run any checks".format(
                    self.Parameters.component_name,
                ),
            )
            return

        startrek_ticket = self._check_if_ticket_exists_and_get_ticket()

        eh.ensure(startrek_ticket is not None, "{} - issue not found".format(self.Parameters.startrek_ticket))
        self.set_info("Ticket {} found".format(self.Parameters.startrek_ticket))

        eh.ensure(
            self._check_ticket_release_tag(startrek_ticket),
            "{} tag not found among {} tags".format(
                self.startrek_release_number_tag,
                self.Parameters.startrek_ticket,
            )
        )
        self.set_info("Tag {} found".format(self.startrek_release_number_tag))

    def _check_if_ticket_exists_and_get_ticket(self):
        """
        Try to get ticket with key `self.Parameters.startrek_ticket`.
        If not found return None. Return the ticket otherwise

        :return: Issue object or None if not found
        """
        logging.info("Check and get issue %s", self.Parameters.startrek_ticket)

        from yandex_tracker_client import exceptions

        try:
            startrek_ticket = self.startrek_client.issues[self.Parameters.startrek_ticket]
            logging.info("Ticket found: %s", startrek_ticket)
            return startrek_ticket
        except exceptions.NotFound:
            logging.exception("%s not found", self.Parameters.startrek_ticket)

        return

    def _check_ticket_release_tag(self, startrek_ticket):
        logging.info("Check release number tag %s presence", self.startrek_release_number_tag)
        ticket_tags = startrek_ticket.tags
        logging.info("Tags found: %s", ticket_tags)
        return self.startrek_release_number_tag in ticket_tags
