# -*- coding: utf-8 -*-
import six
import logging

from sandbox.common import errors
import sandbox.sdk2 as sdk2

import sandbox.projects.release_machine.components.all as rmc
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.tasks.base_task as rm_bt
from sandbox.projects.common import binary_task
from sandbox.projects.common import decorators
from sandbox.projects.common import gsid_parser
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils2
from sandbox.projects.release_machine import input_params2 as rm_params
from sandbox.projects.release_machine import rm_utils
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper


class CheckTasksCorrectness(rm_bt.BaseReleaseMachineTask):
    """Check correctness for input tasks, using SDK2."""
    class Requirements(task_env.StartrekRequirements):
        disk_space = 400  # 400 Mb, 128 is not enough

    class Parameters(rm_params.DefaultReleaseMachineParameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        kill_timeout = 20 * 60  # 20 min
        release_num = sdk2.parameters.Integer("Release number", default=0)
        tasks_to_check = sdk2.parameters.String("Tasks to check before release (divide by comma)")
        autotesting_ok = sdk2.parameters.Bool("Move ticket to autotesting_ok state", default=False)
        with sdk2.parameters.Group("Notifications") as notify:
            notify_ticket = sdk2.parameters.String("Ticket to notify")
            notify_message = sdk2.parameters.String("Message", multiline=True)
            notify_group = sdk2.parameters.String("Message group", default_value="====Check tasks results")

    class Context(rm_bt.BaseReleaseMachineTask.Context):
        type = "CHECK_TASKS_CORRECTNESS"

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)
        task_ids = rm_utils.get_tasks_to_check(self)
        logging.debug("Task ids are {task_ids}".format(task_ids=task_ids))
        if task_ids:
            try:
                utils2.check_tasks_to_finish_correctly(self, task_ids)
            except errors.TaskFailure:
                self._comment_to_ticket("Tests failed")
                raise
            self._comment_to_ticket("Tests OK")

            if self.Parameters.autotesting_ok:
                try:
                    c_info = rmc.get_component(self.Parameters.component_name)
                    logging.debug("Try to go to autotesting_ok in component {comp_name}".format(
                        comp_name=self.Parameters.component_name,
                    ))
                    st_helper = STHelper(rm_sec.get_rm_token(self))
                    issue = st_helper.find_ticket_by_release_number(self.Parameters.release_num, c_info)
                    response = issue.transitions[rm_const.Workflow.AUTOTESTS_OK].execute()
                    logging.debug("The response is {response}".format(response=response))
                except Exception as exc:
                    eh.log_exception("Unable to make transition to AutotestingOk", exc)

    @decorators.retries(3, delay=5, default_instead_of_raise=True)
    def _comment_to_ticket(self, msg):
        if self.Parameters.notify_ticket and self.Parameters.notify_message:
            st_helper = STHelper(rm_sec.get_rm_token(self))
            st_helper.write_grouped_comment_in_issue(
                self.Parameters.notify_ticket,
                self.Parameters.notify_group,
                msg,
                self.Parameters.notify_message,
            )

    def _get_rm_proto_event_specific_data(self, rm_proto_events, event_time_utc_iso, status=None):
        if self.Parameters.debug_mode or self.Parameters.release_num == 0:
            return
        logging.debug("Building specific event data for %s", self.Context.rm_event_type)

        from release_machine.common_proto import test_results_pb2 as rm_test_results

        if status == "SUCCESS":
            test_result_status = rm_test_results.TestResult.TestStatus.OK
        else:
            test_result_status = rm_test_results.TestResult.TestStatus.CRIT

        return {
            "generic_test_data": rm_proto_events.GenericTestData(
                job_name=six.text_type(self.get_job_name_from_gsid()),
                scope_number=six.text_type(self.Parameters.release_num),
                revision=six.text_type(gsid_parser.get_svn_revision_from_gsid(self.Context.__GSID)),
                test_result=rm_test_results.TestResult(
                    status=test_result_status,
                ),
            )
        }

    @sdk2.footer()
    def footer(self):
        task_ids = rm_utils.get_tasks_to_check(self)
        status_field = "status"
        description_field = "description"
        id_field = "id"
        tasks_dicts = [
            rm_utils.get_task_fields(task_id, [status_field, description_field], sandbox_client=self.server)
            for task_id in task_ids
        ]
        statuses = [
            {
                "Task id": lb.task_link(checked_task[id_field]),
                "Description": checked_task[description_field],
                "Status": "<span class='status status_{0}'>{0}</span>".format(checked_task[status_field].lower()),
            }
            for checked_task in tasks_dicts
        ]
        return {"<h3>Task statuses</h3>": statuses}
