# -*- coding: utf-8 -*-
import logging
import uuid

import sandbox.projects.release_machine.components.all as rmc
import sandbox.projects.release_machine.components.components_info as rm_comp
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.helpers.startrek_helper as startrek_helper
import sandbox.projects.release_machine.helpers.svn_helper as rm_svn
import sandbox.projects.common.testenv_client as te_helper
import sandbox.projects.common.error_handlers as eh
import sandbox.projects.common.binary_task as binary_task
import sandbox.projects.release_machine.helpers.wiki_helper as rm_wiki
import sandbox.projects.release_machine.input_params2 as rm_params
import sandbox.projects.release_machine.tasks.base_task as rm_bt
import sandbox.projects.release_machine.security as rm_sec
import sandbox.sdk2 as sdk2


class CleanupFailedReleaseAttempt(rm_bt.BaseReleaseMachineTask):
    """
        For "pre-release" process:
            - Removes branch from repo
            - Removes first tag of specified branch from repo
            - Removes testenv database
            - Closes startrek ticket
            - Removes wiki page with changelog
        For "release" process:
            Not implemented yet
    """
    class Requirements(task_env.StartrekRequirements):
        pass

    class Parameters(rm_params.ComponentName2):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        kill_timeout = 20 * 60  # 20 min
        with sdk2.parameters.RadioGroup('Release action type') as release_action_type:
            release_action_type.values.pre_release = release_action_type.Value('Pre release', default=True)
            release_action_type.values.release = release_action_type.Value('Release')
        with release_action_type.value.pre_release:
            branch_number = sdk2.parameters.Integer("Branch number")
        with release_action_type.value.release:
            tag_number = sdk2.parameters.Integer("Tag number")

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)
        c_info = rmc.get_component(self.Parameters.component_name)
        if self.Parameters.release_action_type == "pre_release":
            self.cleanup_pre_release(c_info)
        elif self.Parameters.release_action_type == "release":
            self.cleanup_release(c_info)
        else:
            self.set_info("Unknown release action type. Nothing to do")

    def cleanup_pre_release(self, c_info):
        if not isinstance(c_info, rm_comp.Branched):
            self.set_info("Component %s is not 'Branched', and does not use 'pre-release' action. Nothing to do")
            return
        logging.info("Try to cleanup '%s' release for %s", self.Parameters.branch_number, c_info.name)
        self._remove_branch(c_info)
        self._remove_tag(c_info)
        self._remove_te_db(c_info)
        self._remove_st_ticket(c_info)
        self._remove_changelog(c_info)

    def _remove_branch(self, c_info):
        branch_path = c_info.full_branch_path(self.Parameters.branch_number)
        if sdk2.svn.Arcadia.check(branch_path):
            logging.info("Removing branch path: %s", branch_path)
            ssh_key = rm_svn.get_ssh_key(self, rm_const.COMMON_TOKEN_OWNER, rm_const.ARC_ACCESS_TOKEN_NAME)
            with ssh_key:
                sdk2.svn.Arcadia.delete(branch_path, message=self._cleanup_msg, user=rm_const.ROBOT_RELEASER_USER_NAME)
        else:
            self._not_exist_message("Branch path: {}".format(branch_path))

    def _remove_tag(self, c_info):
        tag_path = c_info.full_tag_path(1, self.Parameters.branch_number)
        if sdk2.svn.Arcadia.check(tag_path):
            logging.info("Removing tag path: %s", tag_path)
            ssh_key = rm_svn.get_ssh_key(self, rm_const.COMMON_TOKEN_OWNER, rm_const.ARC_ACCESS_TOKEN_NAME)
            with ssh_key:
                sdk2.svn.Arcadia.delete(tag_path, message=self._cleanup_msg, user=rm_const.ROBOT_RELEASER_USER_NAME)
        else:
            self._not_exist_message("Tag path: {}".format(tag_path))

    def _remove_te_db(self, c_info):
        te_db_name = c_info.testenv_cfg__db_template.format(testenv_db_num=self.Parameters.branch_number)
        success = te_helper.TEClient.cleanup_db({"drop_database_names": te_db_name})
        eh.ensure(success, "Cleanup databases failed")

    def _remove_st_ticket(self, c_info):
        if isinstance(c_info, rm_comp.mixin.Startreked):
            logging.info("Try to remove startrek ticket")
            st_helper = startrek_helper.STHelper(rm_sec.get_rm_token(self))
            issue = st_helper.find_ticket_by_release_number(self.Parameters.branch_number, c_info, fail=False)
            if issue is not None:
                logging.info("Closing ticket: %s", issue.key)
                issue.update(tags=[], unique=str(uuid.uuid4()))
                st_helper.close_issue(issue, self._cleanup_msg)
            else:
                self._not_exist_message("Ticket")

    def _remove_changelog(self, c_info):
        if isinstance(c_info, rm_comp.mixin.Changelogged):
            logging.info("Try to remove changelog wiki page")
            wiki_api = rm_wiki.WikiApi(rm_sec.get_rm_token(self))
            wiki_url = c_info.changelog_major_url(self.Parameters.branch_number)
            if wiki_url and wiki_api.exists(wiki_url):
                logging.info("Try to remove wiki page: %s", wiki_url)
                wiki_api.delete_page(wiki_url)
            else:
                self._not_exist_message("Wiki page: {}".format(wiki_url))

    @property
    def _cleanup_msg(self):
        return "Cleanup from task: {}. __BYPASS_CHECKS__".format(self.id)

    @staticmethod
    def _not_exist_message(something):
        logging.info("%s does not exist. Nothing to remove", something)

    def cleanup_release(self, c_info):
        logging.info(
            "Try to cleanup '%s-%s' release for %s",
            self.Parameters.branch_number, self.Parameters.tag_number, c_info.name
        )
        logging.info("This functional is not implemented yet!")
