# -*- coding: utf-8 -*-
import json
import logging
import time

import sandbox.projects.common.link_builder as lb
import sandbox.projects.common.error_handlers as eh
import sandbox.projects.release_machine.components.all as rmc
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.input_params2 as rm_params
import sandbox.projects.release_machine.rm_notify as rm_notify
import sandbox.projects.release_machine.security as rm_sec
import sandbox.projects.release_machine.tasks.base_task as rm_bt
import sandbox.sdk2 as sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common import string
from sandbox.projects.common.testenv_client import TEClient
from sandbox.common import errors
from sandbox.common.types import task as ctt


TE_READONLY_MATCH = "MySQL server is running with the --read-only option"
SLEEP_IF_READONLY_SECONDS = 5 * 60
COMPONENTS_WITH_LONG_AND_HARD_CLONE = {
    "smart_devices",
}


@rm_notify.notify2()
class CloneTestenvDb(rm_bt.BaseReleaseMachineTask):
    """
        **Release-machine**

        Клонирует базу TestEnv. Работает для компонент,
        описанных здесь: projects/release_machine/components/

        Умеет игнорировать заданные тесты при клонировании, а также задавать произвольные частоты выполнения тестов.

        Реализован в рамках SEARCH-1338.
    """

    _c_info = None
    _branch_id = None
    _new_db_name = None

    class Requirements(task_env.TinyRequirements):
        disk_space = 2 * 1024  # 2 Gb
        semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(name='RM_CLONE_TE_DB'),
            ]
        )

    class Parameters(rm_params.ComponentName2):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        kill_timeout = 15 * 60  # 15 min
        branch_path = sdk2.parameters.ArcadiaUrl("Arcadia path to branch", required=True)
        start_revision = sdk2.parameters.Integer("Start revision", required=False)
        autostart_db = sdk2.parameters.Bool("Start DB automatically", default=True)

    @property
    def c_info(self):
        if not self._c_info:
            self._c_info = rmc.get_component(self.Parameters.component_name)
        return self._c_info

    @property
    def branch_id(self):
        if not self._branch_id:
            self._branch_id = self.c_info.get_branch_id_from_path(self.Parameters.branch_path)
        return self._branch_id

    @property
    def new_db_name(self):
        if not self._new_db_name:
            self._new_db_name = self.c_info.testenv_cfg__db_template.format(testenv_db_num=self.branch_id)
        return self._new_db_name

    def release_machine_shard(self):
        """For TESTENV-3978."""
        return '1'  # all switched

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)
        oauth_token = rm_sec.get_rm_token(self)
        te_helper = TEClient(oauth_token=oauth_token)
        trunk_db_sys_info = te_helper.get_sys_info(self.c_info.testenv_cfg__trunk_db)
        if not trunk_db_sys_info or not trunk_db_sys_info.json().get("is_started"):
            eh.fail(
                "Unable to clone: {}. It is not started or does not exist".format(self.c_info.testenv_cfg__trunk_db)
            )
        path_to_branch = self.Parameters.branch_path
        # if revision is not specified - try to use last revision from branch
        revision = self.Parameters.start_revision or sdk2.svn.Arcadia.info(path_to_branch)["commit_revision"]
        logging.info("Clone DB named '%s' in revision %s", self.new_db_name, revision)
        params = {
            'source_db_name': self.c_info.testenv_cfg__trunk_db,
            'new_db_name': self.new_db_name,
            'svn_server': 'arcadia.yandex.ru/arc',
            'svn_path': string.left_strip(path_to_branch, self.c_info.svn_cfg__repo_base_url).lstrip("/"),
            'start_revision': str(revision),
            'set_frequency_equal_each_revision': '1',
            'clone_resources_auto_update_settings': '0',
            'clone_resources_for_ignored_tests': '1',  # TESTENV-3312
            'ignore_test_names': self.c_info.testenv_cfg__job_patch__ignore_match,
            'ignore_test_prefixes': self.c_info.testenv_cfg__job_patch__ignore_prefix,
            'custom_frequencies': json.dumps(self.c_info.testenv_cfg__job_patch__change_frequency),
            'new_task_owner': self.c_info.testenv_cfg__branch_task_owner,
            'created_by': self.author,  # TODO: remove assert in TE handler?
            'clone_owners': '1',
            'add_owners': self.c_info.testenv_cfg__testenv_db_owners + [rm_const.ROBOT_RELEASER_USER_NAME],
            'start_new_db': '1' if self.Parameters.autostart_db else '0',
            'stop_database_names': self.c_info.testenv_db_stop_names(self.branch_id),
            'drop_database_names': self.c_info.testenv_db_drop_names(self.branch_id),
            'release_machine': '1',  # see also TESTENV-2590
            'set_release_machine_lazy_frequency': '1',
            'release_machine_shard': self.release_machine_shard(),
        }
        if self.c_info.testenv_cfg__merge_on_clone:
            params['add_tests_from'] = self.c_info.testenv_cfg__merge_on_clone
            if self.c_info.testenv_add_tests:
                params["test_names"] = self.c_info.testenv_add_tests
                params["use_all_tests_from_source_db"] = "1"
        if self.c_info.testenv_cfg__job_graph and self.c_info.testenv_cfg__job_graph__graph:  # JobGraph enabled
            params['clone_all_resources'] = '1'  # RMDEV-1233

        # DEVTOOLSSUPPORT-16568
        if self.Parameters.component_name in COMPONENTS_WITH_LONG_AND_HARD_CLONE:
            clone_response = te_helper.clone_db_no_retries(
                params,
                return_bool=False,
                timeout=300,
                perform_cleanup_on_failure=False,
            )
        else:
            clone_response = te_helper.clone_db(params, return_bool=False)

        success = clone_response.status_code == 200

        if TE_READONLY_MATCH in clone_response.text:
            time.sleep(SLEEP_IF_READONLY_SECONDS)
            raise errors.TemporaryError("TestEnv database is in read-only mode: {}".format(clone_response.text))

        if not success:
            eh.fail("Clone database failed!")

        created_db_url = rm_const.Urls.te_db_screen(self.new_db_name, "settings")
        link_to_db = lb.HREF_TO_ITEM.format(link=created_db_url, name=self.new_db_name)
        self.set_info("Testenv db created: {}".format(link_to_db), do_escape=False)
        self.deactivate_jobs(te_helper, self.c_info, self.branch_id)

    @staticmethod
    def deactivate_jobs(te_helper, c_info, branch_id):
        try:
            te_db = c_info.testenv_cfg__db_template.format(testenv_db_num=branch_id - 1)
            for job_to_deactivate in c_info.testenv_cfg__job_patch__deactivate:
                te_helper.stop_job(te_db, job_to_deactivate)
        except Exception as e:
            eh.log_exception("Unable to deactivate jobs", e)

    def _get_rm_proto_event_hash_items(self, event_time_utc_iso, status=None):
        return (
            self.Parameters.component_name,
            'TestenvDbClone',
            self.branch_id,
            self.new_db_name,
            "TASK_{}".format(status or self.status),
            str(self.id),
        )

    def _get_rm_proto_event_specific_data(self, rm_proto_events, event_time_utc_iso, status=None):

        specific_data = rm_proto_events.TestenvDbCloneData(
            branch_testenv_db=self.new_db_name,
            scope_number=str(self.branch_id),
            job_name=self.get_job_name_from_gsid(),
        )

        return {
            "testenv_db_clone_data": specific_data,
        }
