# -*- coding: utf-8 -*-
import collections
import logging
import re
import six
import time

import sandbox.common.errors as err
import sandbox.common.types.task as ctt
import sandbox.projects.common.link_builder as lb
import sandbox.projects.common.sdk_compat.task_helper as th
import sandbox.projects.common.error_handlers as eh
import sandbox.projects.common.vcs.arc as arc
import sandbox.projects.release_machine.components.all as rmc
from sandbox.projects.release_machine import core as rm_core
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.helpers.arcadia as rm_arc
import sandbox.projects.release_machine.arc_helper as arc_helper
import sandbox.projects.release_machine.notify_helper as notify_helper
import sandbox.projects.release_machine.helpers.startrek_helper as rm_st
import sandbox.projects.release_machine.helpers.svn_helper as rm_svn
import sandbox.projects.release_machine.input_params2 as rm_params
import sandbox.projects.release_machine.rm_notify as rm_notify
import sandbox.projects.release_machine.security as rm_sec
import sandbox.projects.release_machine.tasks.base_task as rm_bt
import sandbox.sdk2 as sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common import decorators
from sandbox.projects.common.arcadia.re_svn import SVN_COMMIT_RE
from sandbox.sdk2.vcs import svn

_VALID_PATHS = [
    # arc
    ':/arc/branches/',
    ':/arc/tags/',
    # robots
    ':/robots/branches/',
    ':/robots/tags/',
]


BranchToTagArcadiaData = collections.namedtuple(
    'BranchToTagArcadiaData',
    (
        'branch_num',
        'full_branch_path',
        'revs',
        'tag_id',
        'full_new_tag_path',
    ),
)

RELEASE_TICKET_WAIT_TIME_SEC = 60


@rm_notify.notify2()
class CreateBranchOrTag(rm_bt.BaseReleaseMachineTask):
    """
        The purpose of this task is automated creation of branches and tags for releases, powered by Release Machine.

        Task owner needs to upload her SSH key for Arcadia to Sandbox Vault under the name 'arc_access_key'.
        Bots should use the Sandbox group 'SEARCH-RELEASERS' and the user 'robot-srch-releaser', except
        tasks using TRUNK mode, which doesn't require authorized actions.

        The task can:

        - create nothing, but return the trunk as a result (TRUNK mode)
        - create a tag from the trunk (TRUNK_TO_TAG mode)
        - create a branch from the trunk (TRUNK_TO_BRANCH mode)
        - create a tag from a branch (BRANCH_TO_TAG mode)
        - create a branch from a branch (BRANCH_TO_BRANCH mode)
        - shut down betas in TRUNK_TO_BRANCH mode
        - use last revision in source branch by default

        Output: string with created path.
    """
    class Requirements(task_env.StartrekRequirements):
        disk_space = 2 * 1024  # 2 Gb

    class Context(rm_bt.BaseReleaseMachineTask.Context):
        creation_time = None
        scope_number = None
        base_commit_id = None
        arc_hash = None

    class Parameters(rm_params.ComponentName2):
        _lbrp = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.String("Choose branch/tag creation mode:") as tag_or_branch_mode:
            rm_params.set_choices(tag_or_branch_mode, rm_const.BranchingMode.all)

        with tag_or_branch_mode.value[rm_const.BranchingMode.BRANCH_TO_TAG]:
            branch_number_for_tag = sdk2.parameters.Integer("Custom branch number (0=last)", default=0)
            wait_for_release_ticket = sdk2.parameters.Bool(
                "Wait until release ticket is created (USEREXP-14158)",
                default_value=False,
            )  # [USEREXP-14158] Prevents race condition between new tag and release ticket creation

        with tag_or_branch_mode.value[rm_const.BranchingMode.TRUNK_TO_TAG]:
            custom_tag_number = sdk2.parameters.Integer("Custom tag number (0=infer from last existing + 1)")

        with tag_or_branch_mode.value[rm_const.BranchingMode.TRUNK_TO_BRANCH]:
            custom_branch_number = sdk2.parameters.Integer("Custom branch number (0=infer from last existing + 1)")
            svn_branch_only = sdk2.parameters.Bool(
                "Create SVN branch only",
                description="Forces the task to create only SVN branch (and no Arc branch). This flag is used to "
                            "support some complex cases of TE -> CI transition. See RMDEV-2584",
                default_value=False,
            )

        revision_for_trunk = sdk2.parameters.Integer("Custom revision (not required)")
        custom_comment = sdk2.parameters.String("Custom comment to commit message", multiline=True)
        fail_on_existing_tag = sdk2.parameters.Bool("Fail on existing tag", default_value=True)

        with sdk2.parameters.Output:
            result_path = sdk2.parameters.String("Created path")
            result_url = sdk2.parameters.String("Created path URL")
            result_revision = sdk2.parameters.Integer("Result revision")
            checkout_arcadia_from_url = sdk2.parameters.String("Svn url for arcadia")
            new_tag_number = sdk2.parameters.Integer("New tag number")
            new_branch_number = sdk2.parameters.Integer("New branch number")
            release_ticket = sdk2.parameters.String("Startrek release ticket key")

    @decorators.memoized_property
    def _arc(self):
        return arc.Arc(secret_name=rm_const.COMMON_TOKEN_NAME, secret_owner=rm_const.COMMON_TOKEN_OWNER)

    @decorators.memoized_property
    def c_info(self):
        return rmc.get_component(self.Parameters.component_name)

    @decorators.memoized_property
    def rev(self):
        return self.Parameters.revision_for_trunk or svn.Arcadia.get_revision(self.c_info.svn_cfg__main_url)

    @decorators.memoized_property
    def rm_token(self):
        return rm_sec.get_rm_token(self)

    @decorators.memoized_property
    def st_helper(self):
        return rm_st.STHelper(self.rm_token)

    @decorators.memoized_property
    def branch_to_tag_arcadia_data(self):  # -> BranchToTagArcadiaData
        branch_num = self.Parameters.branch_number_for_tag or self.c_info.last_branch_num
        full_branch_path = self.c_info.full_branch_path(branch_num)
        # RMINCIDENTS-25
        revs = rm_svn.SvnHelper.revisions_merged(full_branch_path)
        logging.debug("Collect revs: %s", str(revs))
        tag_id = revs.index(self.rev) + 1 if self.rev in revs else "fake-{}".format(self.rev)
        full_new_tag_path = self.c_info.full_tag_path(tag_id, branch_num)

        return BranchToTagArcadiaData(
            branch_num=branch_num,
            full_branch_path=full_branch_path,
            revs=revs,
            tag_id=tag_id,
            full_new_tag_path=full_new_tag_path,
        )

    @property
    def base_commit_id(self):

        if not self.Context.base_commit_id:
            self.Context.base_commit_id = self.rev
            self.Context.save()

        return self.Context.base_commit_id

    @property
    def new_tag_number(self):

        if self.Parameters.new_tag_number:
            return self.Parameters.new_tag_number

        if self.Parameters.tag_or_branch_mode == rm_const.BranchingMode.TRUNK_TO_TAG:

            if self.Parameters.custom_tag_number:
                self.Parameters.new_tag_number = self.Parameters.custom_tag_number
            else:
                self.Parameters.new_tag_number = self.c_info.last_tag_num() + 1

        elif self.Parameters.tag_or_branch_mode == rm_const.BranchingMode.BRANCH_TO_TAG:

            if self.rev in self.branch_to_tag_arcadia_data.revs:
                self.Parameters.new_tag_number = self.branch_to_tag_arcadia_data.tag_id

        return self.Parameters.new_tag_number

    @property
    def new_branch_number(self):

        if self.Parameters.new_branch_number:
            return self.Parameters.new_branch_number

        if self.Parameters.tag_or_branch_mode == rm_const.BranchingMode.TRUNK_TO_BRANCH:

            if self.Parameters.custom_branch_number:
                self.Parameters.new_branch_number = self.Parameters.custom_branch_number
            else:
                self.Parameters.new_branch_number = self.c_info.next_branch_num

        elif self.Parameters.tag_or_branch_mode == rm_const.BranchingMode.BRANCH_TO_BRANCH:

            self.Parameters.new_branch_number = self.c_info.last_branch_num + 1

        return self.Parameters.new_branch_number

    @property
    def scope_number(self):

        if not self.Context.scope_number:

            if self.Parameters.tag_or_branch_mode == rm_const.BranchingMode.TRUNK_TO_TAG:

                self.Context.scope_number = self.new_tag_number

            elif self.Parameters.tag_or_branch_mode in (
                rm_const.BranchingMode.TRUNK_TO_BRANCH,
                rm_const.BranchingMode.BRANCH_TO_BRANCH,
            ):

                self.Context.scope_number = self.new_branch_number

            elif self.Parameters.tag_or_branch_mode == rm_const.BranchingMode.BRANCH_TO_TAG:

                self.Context.scope_number = self.branch_to_tag_arcadia_data.branch_num

            self.Context.save()

        return self.Context.scope_number

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)

        if self.Parameters.wait_for_release_ticket:
            ticket = self._set_release_ticket()
            if not ticket:
                raise sdk2.WaitTime(RELEASE_TICKET_WAIT_TIME_SEC)

        if self.Parameters.result_path:
            self.set_info("Path already created: {}".format(self.Parameters.result_path))
            return

        c_info = rmc.get_component(self.Parameters.component_name)

        self.send_rm_proto_event(self.status)

        arc_hash = arc_helper.get_arc_hash_for_revision(self._arc, self.rev)
        logging.info("Got arc_hash `%s` for revision %s", arc_hash, self.rev)

        mode = self.Parameters.tag_or_branch_mode
        self.set_info("Revision for {} is {}".format(mode, self.rev))
        ssh_key = rm_svn.get_ssh_key(self, rm_const.COMMON_TOKEN_OWNER, rm_const.ARC_ACCESS_TOKEN_NAME)

        if mode == rm_const.BranchingMode.TRUNK:
            result_path, result_revision = self._make_trunk(self.rev)
        elif mode == rm_const.BranchingMode.TRUNK_TO_TAG:
            result_path, result_revision = self._make_new_tag_from_trunk(c_info, self.rev, ssh_key)
            self.Context.tag_name = c_info.svn_cfg__tag_folder_name(self.new_tag_number)
        elif mode == rm_const.BranchingMode.TRUNK_TO_BRANCH:
            result_path, result_revision = self._make_new_branch(c_info, self.rev, ssh_key)
        elif mode == rm_const.BranchingMode.BRANCH_TO_TAG:
            result_path, result_revision = self._make_new_tag_from_branch(c_info, self.rev, ssh_key)
            self.Context.tag_name = c_info.svn_cfg__tag_folder_name(
                self.Parameters.branch_number_for_tag,
                self.new_tag_number,
            )
        elif mode == rm_const.BranchingMode.BRANCH_TO_BRANCH:
            result_path, result_revision = self._make_new_branch_from_branch(c_info, self.rev, ssh_key)
        else:
            raise err.TaskError("Unknown task mode: '{}'".format(mode))
        self.Context.creation_time = int(time.time())
        result_url = self._get_result_url(result_path)
        checkout_arcadia_from_url = rm_arc.construct_arc_url(result_path, result_revision)
        self.Parameters.result_path = result_path
        self.Parameters.result_revision = result_revision
        self.Parameters.result_url = result_url
        self.Parameters.checkout_arcadia_from_url = checkout_arcadia_from_url
        # fix for testenv:
        self.Context.result_path = result_path
        self.Context.result_revision = result_revision
        self.Context.result_url = result_url
        self.Context.checkout_arcadia_from_url = checkout_arcadia_from_url

    @staticmethod
    def parse_revision_from_stdout(stdout):
        return int(SVN_COMMIT_RE.findall(stdout)[0])

    @staticmethod
    def _make_trunk(rev):
        logging.info("Making new path from trunk")
        return "{}@{}".format(rm_svn.TRUNK_PATH, rev), rev

    @staticmethod
    def _get_result_url(result_path=None):
        if result_path and isinstance(result_path, six.string_types):
            arcadia_url_obj = svn.Arcadia.parse_url(result_path)
            # Since `Arcadia.parse_url` won't work properly with '/arc/tags/...'
            # not followed by 'arcadia/...' we need to get tag name manually
            # Improvement expected here: RMDEV-481
            m = re.findall(r".*/arc/tags/(.*)|.*/arc/branches/(.*)", arcadia_url_obj.path)
            short_url = m[0][0] or m[0][1] if m else None
            return lb.href_to_arcadia(arcadia_url_obj.path, short_url)
        return "unknown_path {}".format(result_path)

    def _make_new_tag_from_trunk(self, c_info, rev, ssh_key):
        logging.info("Making new tag from trunk")

        new_tag_num = self.new_tag_number
        next_tag_path = c_info.full_tag_path(new_tag_num)
        self._make_dirs(next_tag_path, c_info.name, ssh_key)

        path_info = svn.Arcadia.check(next_tag_path)

        if path_info:
            msg = "Next tag path already exists: {}\n{}".format(next_tag_path, path_info)
            logging.info(msg)
            if self.Parameters.fail_on_existing_tag:
                raise err.TaskFailure(msg)
            else:
                result_revision = int(path_info["commit_revision"])
        else:
            with ssh_key:
                returned_stdout = svn.Arcadia.create_tag(
                    c_info.svn_cfg__main_url,
                    next_tag_path,
                    user=rm_const.ROBOT_RELEASER_USER_NAME,
                    message='[tag:{}] from {}, (r{}, sandboxtask:{}).\n{}'.format(
                        next_tag_path, c_info.svn_cfg__main_url, rev, self.id,
                        self.Parameters.custom_comment
                    ),
                    revision=rev,
                ).stdout
                result_revision = self.parse_revision_from_stdout(returned_stdout)
            logging.info("New tag created: %s from trunk", next_tag_path)
            self._add_tag_log_to_context(c_info=c_info, base_commit_id=rev)

        return next_tag_path, result_revision

    def _make_new_branch_arc(self, c_info, rev):
        arc_full_branch_path = c_info.svn_cfg__arc_branch_path(self.new_branch_number)
        logging.info("Creating new branch %s in arc", arc_full_branch_path)
        with self._arc.mount_path(None, None, fetch_all=False, extra_params=["--vfs-version", "2"]) as arcadia_src_dir:
            all_branches = arc_helper.get_all_branches(self._arc)
            if arc_full_branch_path in all_branches:
                logging.debug("Arc branch %s already exists", arc_full_branch_path)
                return
            self._arc.checkout(
                arcadia_src_dir,
                branch=arc_full_branch_path,
                create_branch=True,
                start_point="r{}".format(rev),
            )
            self._arc.push(
                arcadia_src_dir,
                upstream=arc_full_branch_path,
            )
            logging.info("New arc branch created: %s", arc_full_branch_path)

    def _make_new_branch(self, c_info, rev, ssh_key):
        logging.info("Making new branch from trunk")

        full_new_branch_path = c_info.full_branch_path(self.new_branch_number)

        with ssh_key:
            self._make_dirs(full_new_branch_path, c_info.name, ssh_key)
            if svn.Arcadia.check(full_new_branch_path):
                branch_log = svn.Arcadia.log(full_new_branch_path, revision_from='HEAD', revision_to='1', limit=1)
                if rm_svn.SvnHelper.is_branching(branch_log[0]):
                    # prevent possible testenv-sandbox race TESTENV-2605
                    logging.warning(
                        "Branch already exists, but stamped from same trunk revision, "
                        "it seems TESTENV-2605 strikes back!"
                    )
                    result_revision = branch_log[0]["revision"]
                else:
                    raise err.TaskFailure(
                        "Branch path for revision {} already exists: {}".format(rev, full_new_branch_path)
                    )
            else:
                returned_stdout = svn.Arcadia.copy(
                    c_info.svn_cfg__main_url,
                    full_new_branch_path,
                    message="[branch:{}] (r{}, sandboxtask:{}) __BYPASS_CHECKS__\n{}".format(
                        c_info.svn_cfg__branch_name, rev, self.id, self.Parameters.custom_comment
                    ),
                    user=rm_const.ROBOT_RELEASER_USER_NAME,
                    revision=rev,
                ).stdout
                result_revision = self.parse_revision_from_stdout(returned_stdout)

        if c_info.svn_cfg__use_arc:

            if not self.Parameters.svn_branch_only:

                try:
                    self._make_new_branch_arc(c_info, rev)
                except Exception as exc:
                    eh.check_failed("Got exception while creating arc branch: {}".format(exc))

            else:

                logging.info(
                    "Arc branch not going to be created due to the task launch configuration (svn_branch_only = True)",
                )

        th.ctx_field_set(
            self,
            rm_notify.TM_MESSAGE,
            "New branch created: {branch_path} (r{revision}, SB:{task_link})\n#branch #r{branch_number}".format(
                branch_path="/".join(str(c_info.next_branch_path).split("/")[-2:]),
                revision=lb.revision_link(rev),
                task_link=lb.task_link(self.id),
                branch_number=self.new_branch_number,
            )
        )
        logging.info("New branch created: %s", full_new_branch_path)
        self._on_new_branch(c_info)
        return full_new_branch_path, result_revision

    def _on_new_branch(self, c_info):
        # todo: move everything to c_info.on_new_branch() method
        try:
            self.send_tm_notify()
            rm_token = rm_sec.get_rm_token(self)
            c_info.update_config_wiki_pages(rm_token)
            st_helper = rm_st.STHelper(rm_token)
            min_allowed_release_num = c_info.min_allowed_release_branch_num()
            st_helper.close_prev_tickets(
                c_info,
                min_allowed_release_num,
                "New branch was created, so close ticket because its release number is less than {}".format(
                    min_allowed_release_num
                )
            )  # SEARCH-6016
        except Exception as exc:
            self.set_info("Non-blocking error: {}\nPlease, check logs for details.".format(exc))
            eh.log_exception(
                'On new branch error. '
                'Though it is important, it should not block main release process.',
                exc
            )

    def _make_dirs(self, full_path, component_name, ssh_key):
        # RMDEV-821 and RMDEV-1084
        result = rm_svn.make_intermediate_svn_dirs(full_path, component_name, ssh_key)
        if result.ok:
            logging.info(result.result)
        else:
            logging.error(result.result)
            notify_helper.telegram(
                task=self,
                message=result.result,
                chat_ids=[rm_const.RM_USERS["rm_maintainers"].telegram],
            )

    def _make_new_tag_from_branch(self, c_info, rev, ssh_key):
        self.c_info.check_testenv_db(self)
        logging.info("Making new tag from branch")

        self._make_dirs(
            self.branch_to_tag_arcadia_data.full_new_tag_path,
            self.c_info.name,
            ssh_key,
        )

        if svn.Arcadia.check(self.branch_to_tag_arcadia_data.full_new_tag_path):
            # Arcadia.log is buggy here, see RMDEV-271
            tag_log = svn.Arcadia.log(
                self.branch_to_tag_arcadia_data.full_new_tag_path,
                revision_from='HEAD',
                revision_to='1',
                limit=1,
            )
            if "copy-revision:{}".format(rev) in tag_log[0]["msg"]:
                # prevent possible testenv-sandbox race TESTENV-2605
                logging.warning(
                    "Tag already exists, but stamped from same branch revision, it seems TESTENV-2605 strikes back!"
                )
                result_revision = tag_log[0]["revision"]
            else:
                raise err.TaskFailure(
                    "Tag path for revision {} already exists: {}".format(
                        rev,
                        self.branch_to_tag_arcadia_data.full_new_tag_path,
                    )
                )
        else:
            with ssh_key:
                returned_stdout = svn.Arcadia.create_tag(
                    self.branch_to_tag_arcadia_data.full_branch_path,
                    self.branch_to_tag_arcadia_data.full_new_tag_path,
                    revision=rev,
                    message='[tag:{}] from branch {}, sandboxtask:{}, copy-revision:{}.\n{}'.format(
                        self.branch_to_tag_arcadia_data.full_new_tag_path,
                        self.branch_to_tag_arcadia_data.full_branch_path,
                        self.id,
                        rev,
                        self.Parameters.custom_comment,
                    ),
                    user=rm_const.ROBOT_RELEASER_USER_NAME,
                    parents=True,
                ).stdout
                result_revision = self.parse_revision_from_stdout(returned_stdout)
            logging.info(
                "New tag created: %s from branch: %s",
                self.branch_to_tag_arcadia_data.full_new_tag_path,
                self.branch_to_tag_arcadia_data.full_branch_path,
            )

        if c_info.svn_cfg__use_arc:
            try:
                arc_tag_from_branch_result = arc_helper.make_new_arc_tag_from_branch(
                    self._arc, c_info,
                    self.branch_to_tag_arcadia_data.branch_num,
                    self.branch_to_tag_arcadia_data.tag_id,
                    check_with_svn_rev=rev,
                )

                self.Context.arc_hash = arc_tag_from_branch_result.get_context_field("arc_hash")
                self.Context.save()

            except Exception as e:
                arc_tag_from_branch_result = rm_core.Error(
                    "Unable to create tag in arc! Please, contact RM DEV team to solve this!\n{}".format(e)
                )
            self.set_info(arc_tag_from_branch_result)

        self._on_new_tag_from_branch()
        self._add_tag_log_to_context(c_info=c_info, base_commit_id=rev)
        return self.branch_to_tag_arcadia_data.full_new_tag_path, result_revision

    def _set_release_ticket(self):
        ticket = self.st_helper.find_ticket_by_release_number(
            self.branch_to_tag_arcadia_data.branch_num,
            self.c_info,
            fail=False,
        )

        if ticket:
            self.Parameters.release_ticket = ticket.key

        return ticket

    def _on_new_tag_from_branch(self):
        # todo: move everything to c_info.on_new_tag() method
        # Used for release machine events
        try:

            self.st_helper.execute_transition(
                self.c_info,
                self.branch_to_tag_arcadia_data.branch_num,
                rm_const.Workflow.AUTOTESTING,
                self,
            )

            if not self.Parameters.release_ticket:
                self._set_release_ticket()

        except Exception as exc:
            self.set_info("Unblocking error: {}\nPlease, check logs for details.".format(exc))
            eh.log_exception(
                'On new tag error. '
                'Though it is important, it should not block main release process.',
                exc
            )

    def _make_new_branch_from_branch(self, c_info, rev, ssh_key):
        source_branch = c_info.source_branch_url
        eh.verify(source_branch is not None, "Source branch is undefined for component: {}".format(c_info.name))
        full_new_branch_path = c_info.next_branch_path

        logging.info("Making branch %s from branch %s", full_new_branch_path, source_branch)

        with ssh_key:
            returned_stdout = svn.Arcadia.copy(
                source_branch,
                full_new_branch_path,
                message="[branch:{}] (r{}, sandboxtask:{}) __BYPASS_CHECKS__\n{}".format(
                    c_info.svn_cfg__branch_name, rev, self.id, self.Parameters.custom_comment
                ),
                user=rm_const.ROBOT_RELEASER_USER_NAME,
            ).stdout
            result_revision = self.parse_revision_from_stdout(returned_stdout)

        logging.info("New branch created: %s", full_new_branch_path)

        self._on_new_branch(c_info)

        return full_new_branch_path, result_revision

    def _add_tag_log_to_context(self, c_info, base_commit_id):
        try:
            tag_log = svn.Arcadia.log(c_info.svn_cfg__repo_base_url, revision_from=base_commit_id, limit=1)[0]
            tag_log['date'] = tag_log['date'].isoformat()
            self.Context.tag_log = tag_log
            return
        except Exception as e:
            logging.exception("Unable to retrieve tag log info:\n%s", e)
        self.Context.tag_log = {}

    @property
    def footer(self):
        return "<h3>Result path: {}</h3>".format(self.Parameters.result_path or "-")

    def _get_rm_proto_event_hash_items(self, event_time_utc_iso, status=None):
        if self.Parameters.tag_or_branch_mode in (
            rm_const.BranchingMode.BRANCH_TO_TAG,
            rm_const.BranchingMode.TRUNK_TO_TAG,
        ):
            return (
                self.Parameters.component_name,
                "NewTag",
                self.Parameters.result_path.replace("arcadia:/", ""),
                str(self.scope_number),
                str(self.new_tag_number),
                self.base_commit_id,
                self.Parameters.result_revision,
                "TASK_{}".format(status or self.status),
            )
        return (
            "TASK_{}".format(status or self.status),
            self.Parameters.result_path.replace("arcadia:/", ""),
            str(self.scope_number),
            self.base_commit_id,
            self.Parameters.result_revision,
        )

    def _get_rm_proto_event_specific_data(self, rm_proto_events, event_time_utc_iso, status=None):

        logging.debug("Building specific event data for %s for task status %s", self.Context.rm_event_type, status)

        if status == ctt.Status.EXECUTING:

            if self.Parameters.tag_or_branch_mode in {
                rm_const.BranchingMode.BRANCH_TO_TAG,
                rm_const.BranchingMode.TRUNK_TO_TAG,
            }:
                logging.debug('Constructing NewTagCreationStartedData')

                return {
                    'new_tag_creation_started_data': rm_proto_events.NewTagCreationStartedData(
                        base_commit_id=str(self.base_commit_id),
                        scope_number=str(self.scope_number),
                        tag_number=str(self.new_tag_number),
                        job_name=self.get_job_name_from_gsid(),
                    ),
                }

            elif self.Parameters.tag_or_branch_mode in {
                rm_const.BranchingMode.TRUNK_TO_BRANCH,
                rm_const.BranchingMode.BRANCH_TO_BRANCH,
            }:

                logging.debug('Constructing NewBranchCreationStartedData')

                return {
                    'new_branch_creation_started_data': rm_proto_events.NewBranchCreationStartedData(
                        base_commit_id=str(self.base_commit_id),
                        scope_number=str(self.scope_number),
                        job_name=self.get_job_name_from_gsid(),
                    ),
                }

        elif status == ctt.Status.SUCCESS:

            if self.Parameters.tag_or_branch_mode in {
                rm_const.BranchingMode.BRANCH_TO_TAG,
                rm_const.BranchingMode.TRUNK_TO_TAG,
            }:
                logging.debug('Constructing NewTagData')

                return {
                    'new_tag_data': rm_proto_events.NewTagData(
                        base_commit_id=str(self.base_commit_id),
                        first_commit_id=str(self.Parameters.result_revision),
                        scope_number=str(self.scope_number),
                        tag_number=str(self.new_tag_number),
                        tag_name=str(self.Context.tag_name),
                        arcadia_path=self.Parameters.result_path.replace("arcadia:/", ""),
                        tag_created_at_timestamp=int(self.Context.creation_time),
                        job_name=self.get_job_name_from_gsid(),
                        tag_log=(
                            rm_proto_events.TagLog(
                                datetime_iso=self.Context.tag_log.get('date'),
                                commit_message=self.Context.tag_log.get('msg', ''),
                                revision=str(self.Context.tag_log.get('revision', '')),
                                author=self.Context.tag_log.get('author', ''),
                            ) if self.Context.tag_log else None
                        ),
                    ),
                }

            elif self.Parameters.tag_or_branch_mode in {
                rm_const.BranchingMode.TRUNK_TO_BRANCH,
                rm_const.BranchingMode.BRANCH_TO_BRANCH,
            }:
                logging.debug('Constructing NewBranchData')

                return {
                    'new_branch_data': rm_proto_events.NewBranchData(
                        base_commit_id=str(self.base_commit_id),
                        first_commit_id=str(self.Parameters.result_revision),
                        scope_number=str(self.scope_number),
                        job_name=self.get_job_name_from_gsid(),
                        arcadia_path=self.Parameters.result_path.replace("arcadia:/", ""),
                        branch_created_at_timestamp=int(self.Context.creation_time),
                    ),
                }

        return {}
