import logging
import requests
import functools

from sandbox.projects.release_machine.tasks import base_task as rm_bt
from sandbox.projects.release_machine import core as rm_core
from sandbox.projects.release_machine.core import task_env
from sandbox.projects.release_machine import input_params2 as rm_params
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.helpers import startrek_helper as rm_st
from sandbox.projects.release_machine.components import all as rm_components
from sandbox.projects.release_machine import release_approvements_formatter

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common import decorators
from sandbox.projects.common import error_handlers
from sandbox.projects.common import link_builder


def log_result(func):
    """Logs decorated function result before returning it"""

    @functools.wraps(func)
    def wrapper(*args, **kwargs):

        result = func(*args, **kwargs)

        logging.info("%s call\nArgs: %s\nKwargs: %s\nResult: %s", func.__name__, args, kwargs, result)

        return result

    return wrapper


MODE_RELEASE = "release"  # Release approvement. Requires Release Machine
MODE_CUSTOM = "custom"    # Create a custom approvement. No special requirements

DEFAULT_OK_API_TOKEN_SECRET_ID = "sec-01ejz9hcr8tq4mmwga719ecgt0"
DEFAULT_OK_API_TOKEN_SECRET_KEY = "robot_ok_api_token"

APPROVEMENT_TYPE_TRACKER = "tracker"
APPROVEMENT_TYPE_GENERAL = "general"


class CreateStartrekOkApprovement(rm_bt.BaseReleaseMachineTask):
    """
    Creates OK approvement for the given Startrek ticket

    Two modes are available -- 'release' and 'custom'. In 'release' mode the task creates a standard release approvement
    for the given Release Machine component (provided with the `component_name` input parameter). Release approvements
    are fully described by the respective component config settings (see Releases.approvements config section). 'Custom'
    mode can be used to create a custom approvement specifying all the required parameters as a task input.

    Note that Startrek-bound approvements have a major restriction: only one active approvement can be bound to a single
    Startrek ticket. This means that it is not possible to start a new approvement for some ticket unless all previous
    approvements bound to this ticket are closed. You might want to set the
    `finish_successfully_if_approvement_already_exists` input parameter to prevent task from failing when it
    encounters an existing approvement in the target ticket.
    """

    class Requirements(task_env.TinyRequirements):
        pass

    class Parameters(rm_params.DefaultReleaseMachineParameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)

        ticket_key = sdk2.parameters.String(
            "Ticket Key. When blank the task uses the release ticket found using major_release_num and "
            "minor_release_num parameters",
        )

        ok_secret = sdk2.parameters.YavSecret(
            "OK API token secret",
            default_value=sdk2.parameters.YavSecret.cast(
                "{}#{}".format(
                    DEFAULT_OK_API_TOKEN_SECRET_ID,
                    DEFAULT_OK_API_TOKEN_SECRET_KEY,
                ),
            ),
        )

        approvement_type = sdk2.parameters.String(
            "Approvement type",
            required=True,
            default=APPROVEMENT_TYPE_TRACKER,
            choices=[
                (appr_type, appr_type) for appr_type in [APPROVEMENT_TYPE_TRACKER, APPROVEMENT_TYPE_GENERAL]
            ],
        )

        with approvement_type.value[APPROVEMENT_TYPE_GENERAL]:
            approvement_object_id = sdk2.parameters.String("Approvement object_id")

        major_release_num = sdk2.parameters.Integer("Major release number")
        minor_release_num = sdk2.parameters.Integer("Minor release number")

        finish_successfully_if_approvement_already_exists = sdk2.parameters.Bool(
            "Finish successfully if approvement already exists in the target ticket",
            description="If True then the task finishes successfully even if an active approvement already exists in "
                        "the target Startrek ticket (and thus the task cannot create a new approvement).",
        )

        mode = sdk2.parameters.String(
            "Approvement creation mode",
            choices=[(_, _) for _ in [MODE_RELEASE, MODE_CUSTOM]],
            default_value=MODE_RELEASE,
        )

        with mode.value[MODE_RELEASE]:
            with sdk2.parameters.Group("Release approvement settings") as ras:
                check_tag = sdk2.parameters.Bool("Check component's tag number", default_value=True)

        with mode.value[MODE_CUSTOM]:
            with sdk2.parameters.Group("Custom settings") as cos:
                approvers = sdk2.parameters.List("Approver list")
                approvers_json = sdk2.parameters.JSON("Approvers custom JSON (overrides `approvers` param)")
                author = sdk2.parameters.String("Who will become approvement author")
                text = sdk2.parameters.String("Approvement description")
                comment_text = sdk2.parameters.String(
                    "Some text the appears along with the approvement iframe in Startrek comment",
                )
                approve_groups = sdk2.parameters.List("Groups of approvers")

        with sdk2.parameters.Output:
            approvement_uuid = sdk2.parameters.String("Created OK approvement UUID")
            approvement_already_exists = sdk2.parameters.Bool(
                "Approvement already exists for the specified ticket",
                default_value=False,
            )

    @decorators.memoized_property
    def c_info(self):
        return rm_components.get_component(self.Parameters.component_name)

    @decorators.memoized_property
    def tag_name(self):
        if self.c_info.is_tagged:
            return self.c_info.svn_cfg.tag_folder_name(self.Parameters.major_release_num)

        return self.c_info.svn_cfg.tag_folder_name(
            self.Parameters.major_release_num,
            self.Parameters.minor_release_num,
        )

    @decorators.memoized_property
    def ticket_key(self):

        if self.Parameters.ticket_key:
            return self.Parameters.ticket_key

        issue = self.st_helper.find_ticket_by_release_number(
            self.Parameters.major_release_num,
            self.c_info,
        )

        return issue.key if issue else None

    @decorators.memoized_property
    def st_helper(self):
        st_auth_token = rm_sec.get_rm_token(self)
        st_helper = rm_st.STHelper(st_auth_token)
        return st_helper

    @decorators.memoized_property
    def approvement_object_id(self):
        if self.Parameters.approvement_type == APPROVEMENT_TYPE_TRACKER:
            return self.ticket_key
        return self.Parameters.approvement_object_id

    def on_execute(self):

        rm_bt.BaseReleaseMachineTask.on_execute(self)

        parameters_check_result = self.check_parameters()

        if not parameters_check_result.ok:
            error_handlers.check_failed(str(parameters_check_result))

        if self.Parameters.mode == MODE_RELEASE:

            tag_number_check_result = self.check_tag_number()

            if not tag_number_check_result:
                self.set_info(
                    "{name} does not require approvements for tag {tag}. "
                    "No approvement is going to be started, the task is going to finish correctly".format(
                        name=self.c_info.name,
                        tag=self.tag_name,
                    )
                )
                return

            self.start_release_approvement()

        elif self.Parameters.mode == MODE_CUSTOM:

            self.start_custom_approvement()

        else:

            self.set_info(
                "Neither release nor custom approvement requested (according to the input parameters). "
                "Seems like it's a dry run. Doing nothing"
            )

    def on_break(self, prev_status, status):
        # will be removed/refactored in RMDEV-2580
        try:
            self.send_rm_proto_event(status=status)
        except Exception as exc:
            self.log_exception("Event sending FAILED!", exc)
        rm_bt.BaseReleaseMachineTask.on_finish(self, prev_status, status)

    @log_result
    def check_parameters(self):

        logging.info("Checking input parameters...")

        if self.Parameters.mode == MODE_RELEASE and not self.c_info.releases_cfg__approvements:
            return rm_core.Error(
                "No approvements configured for component {} (but release approvement requested)".format(
                    self.Parameters.component_name,
                ),
            )

        if not self.ticket_key:
            return rm_core.Error("No ticket provided")

        if not self.st_helper.get_ticket_by_key(self.ticket_key):
            return rm_core.Error("Ticket {} cannot be found".format(self.ticket_key))

        return rm_core.Ok("Input parameters OK")

    @log_result
    def check_tag_number(self):

        logging.info("Checking tag number...")

        if not self.Parameters.check_tag:
            return rm_core.Ok("The task was told not to check tag number")

        if not self.c_info.releases_cfg__approvements.first_tag_only:
            return rm_core.Ok("The component requires approvement for all tags (first_tag_only = False)")

        if self.Parameters.minor_release_num != 1:
            return rm_core.Error(
                "The component requirest approvement for the 1st tag only. Current tag is {}".format(
                    self.tag_name,
                ),
            )

        return rm_core.Ok("Tag number is OK")

    def _get_startrek_approvement_comment_text(self, approvement_uuid):

        from library.python import ok_client

        okc = ok_client.OkClient(
            token=self.Parameters.ok_secret.data()[self.Parameters.ok_secret.default_key],
        )

        url = okc.get_embed_url(approvement_uuid)

        comment_text = self.Parameters.comment_text or ""

        if comment_text:
            comment_text += "\n"

        return (
            '{{{{iframe'
            ' src="{url}"'
            ' frameborder=0'
            ' width=100%'
            ' height=400px'
            ' scrolling=no'
            '}}}}\n'
            '\n'
            '{additional_comment}'
            '//Posted by task {task_link}//\n'
        ).format(
            url=url,
            task_link=link_builder.task_wiki_link(self.id),
            additional_comment=comment_text,
        )

    def _create_approvement(self, approvement_request):

        from library.python import ok_client

        okc = ok_client.OkClient(
            token=self.Parameters.ok_secret.data()[self.Parameters.ok_secret.default_key],
        )

        approvement_uuid = None

        try:

            approvement_uuid = okc.create_approvement(payload=approvement_request)

        except ok_client.ApprovementAlreadyExists as aae:

            self.set_info("An active approvement found in {}".format(self.ticket_key))

            msg = (
                "An active approvement exists in {ticket} so it is not possible to start a new one. "
                "Please close the existing approvement in order to proceed. If this a common/highly expected "
                "situation consider running the task with `finish_successfully_if_approvement_already_exists=True`. "
                "Original error: {err}"
            ).format(
                ticket=self.ticket_key,
                err=aae,
            )

            logging.exception("Error and error data: %s %s", msg, aae.error_data)

            self.Parameters.approvement_already_exists = True
            self.Parameters.approvement_uuid = aae.error_data.get('error')[0].get("params", {}).get("uuid", "")

            if self.Parameters.finish_successfully_if_approvement_already_exists:

                logging.info(
                    "finish_successfully_if_approvement_already_exists = True => The error is going to be suppressed "
                    "and the task i going to finish successfully"
                )

                return

            else:

                raise

        except (ok_client.UnretriableError, requests.HTTPError, ValueError, TypeError):

            msg = "Failed to create OK approvement"
            logging.exception(msg)
            error_handlers.check_failed(msg)

        logging.info("OK approvement created. UUID: %s", approvement_uuid)

        self.Parameters.approvement_uuid = approvement_uuid

        try:

            self.st_helper.comment_by_key(
                issue_key=self.ticket_key,
                text=self._get_startrek_approvement_comment_text(approvement_uuid),
            )

        except requests.HTTPError as http_error:
            error_handlers.log_exception(
                "Unable to post approvement comment to Startrek ticket {}".format(self.ticket_key),
                http_error,
                info=True,
            )
            error_handlers.fail("Failed")

        logging.info("Approvement comment successfully posted to Startrek ticket %s", self.ticket_key)

    def get_custom_approvement_approvers(self):

        if self.Parameters.approvers_json:

            if not isinstance(self.Parameters.approvers_json, list):
                raise ValueError("approvers_json: expected list, got {}".format(type(self.Parameters.approvers_json)))

            return self.Parameters.approvers_json

        from library.python import ok_client

        return [ok_client.StageSimple(approver=approver) for approver in self.Parameters.approvers]

    def start_custom_approvement(self):
        """
        Start custom OK approvement
        """

        logging.info("Start Custom Approvement")

        from library.python import ok_client

        approvement_request = ok_client.CreateApprovementRequest(
            type=str(self.Parameters.approvement_type),
            object_id=str(self.approvement_object_id),
            text=str(self.Parameters.text),
            stages=self.get_custom_approvement_approvers(),
            author=str(self.Parameters.author),
            groups=list(self.Parameters.approve_groups),
            is_parallel=True,
        )

        self._create_approvement(approvement_request)

    def start_release_approvement(self):
        """
        Start OK approvement process in current release ticket.
        This method relies on the `Releases.approvement` config property and does nothing if this property is set to
        None. If `Releases.approvement` is not empty then the method creates an approvement and posts the respective
        comment to the release ticket.

        See https://wiki.yandex-team.ru/releasemachine/blokirovka-relizov/#blokirovkadook-ovotzadannyxsotrudnikov
        for more information
        """

        logging.info("Start Release Approvement")

        try:
            approvement_request = release_approvements_formatter.release_approvements_settings_to_dict(
                settings=self.c_info.releases_cfg__approvements,
                ticket_key=self.ticket_key,
                author=self.c_info.get_responsible_for_release(),
                component_name=self.c_info.name,
                tag_name=self.tag_name,
            )
        except Exception as e:
            error_handlers.log_exception(
                "Unable to build approvement request for the approvement settings {}. "
                "Release approvement is not going to be created.".format(self.c_info.releases_cfg__approvements),
                e,
            )

            error_handlers.fail("Failed to build approvement request")

        logging.info("Approvement request generated: %s", approvement_request)

        self._create_approvement(approvement_request)

    def _get_rm_proto_event_specific_data(self, rm_proto_events, event_time_utc_iso, status=None):

        if not all([
            self.Parameters.component_name,
            self.Parameters.approvement_uuid,
            self.Parameters.major_release_num,
        ]):
            logging.info(
                "Task input parameters do not satisfy the minimal requirements for RM event construction. "
                "RM event is not going to be sent"
            )
            return {}

        from release_machine.common_proto import events_pb2

        return {
            "approvement_created_data": events_pb2.ApprovementCreatedData(
                approvement_uuid=self.Parameters.approvement_uuid,
                ticket_key=self.ticket_key,
                scope_number=str(self.Parameters.major_release_num),
                tag_number=str(self.Parameters.minor_release_num or 0),
            ),
        }
