# -*- coding: utf-8 -*-
import json
import logging
from typing import (  # noqa: UnusedImport
    Any,
    Dict,
    List,
    Tuple,
)

import sandbox.common.types.task as ctt
import sandbox.projects.common.link_builder as lb
import sandbox.projects.common.time_utils as tu
import sandbox.projects.common.utils2 as utils
import sandbox.projects.common.error_handlers as eh
import sandbox.projects.release_machine.components.all as rmc
import sandbox.projects.release_machine.core as rm_core
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.abc.client as abc_client
import sandbox.projects.release_machine.helpers.changelog_formatter as rm_cf
import sandbox.projects.release_machine.helpers.changelog_helper as rm_changelog
import sandbox.projects.release_machine.changelogs as rm_ch
import sandbox.projects.release_machine.notify_helper as nh
import sandbox.projects.release_machine.helpers.startrek_helper as rm_st
import sandbox.projects.release_machine.helpers.wiki_helper as rm_wiki
import sandbox.projects.release_machine.input_params2 as rm_params
import sandbox.projects.release_machine.resources as rm_res
import sandbox.projects.release_machine.rm_notify as rm_notify
import sandbox.projects.release_machine.security as rm_sec
import sandbox.projects.release_machine.users as rm_users
import sandbox.projects.release_machine.tasks.base_task as rm_bt
import sandbox.sdk2 as sdk2

from sandbox.projects.common import binary_task
from sandbox.projects.common import decorators
from sandbox.projects.common.string import all_to_str


@rm_notify.notify2()
class CreateStartrekTicket(rm_bt.BaseReleaseMachineTask):

    class Requirements(task_env.TinyRequirements):
        pass

    class Parameters(rm_params.DefaultReleaseMachineParameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        kill_timeout = 15 * 60  # 15 min
        group_changelog_by_tickets = sdk2.parameters.Bool(
            "Group changelog by ticket", default=False, description="RMDEV-209"
        )
        deduplicate_by_changes = sdk2.parameters.Bool(
            "Deduplicate release items info by changes", default=True, description="RMDEV-1400"
        )
        changelog = sdk2.parameters.Resource("Changelog", resource_type=rm_res.RELEASE_MACHINE_CHANGELOG)
        additional_info = sdk2.parameters.String("Additional info")
        update_issue_if_exists = sdk2.parameters.Bool(
            "Update issue description if issue has been already created",
            default=True,
        )

        with sdk2.parameters.Output:
            startrek_issue = sdk2.parameters.String("Created issue")
            issue_updated = sdk2.parameters.Bool(
                "Issue was NOT created, but this task updated it's description",
                default=False,
            )

    class Context(rm_bt.BaseReleaseMachineTask.Context):
        issue_start_time = 0
        issue_status = ""

    @decorators.memoized_property
    def c_info(self):
        return self.get_component_info()

    @decorators.memoized_property
    def st_helper(self):
        st_auth_token = rm_sec.get_rm_token(self)
        st_helper = rm_st.STHelper(st_auth_token)
        return st_helper

    def generate_description(self, c_info, changelog):
        descr = [self.Parameters.additional_info]
        if self.Parameters.deduplicate_by_changes:
            deduplicated_changelog = rm_ch.deduplicate_changelog_by_changes(changelog["all_changes"])
            for i in deduplicated_changelog:
                descr.append(rm_ch.format_release_items_info(i["release_items"], changelog))
                descr.extend(self.format_important_changes(c_info, i["changes"]))
                descr.append("---")
        else:
            for release_item_changes in changelog["all_changes"]:
                descr.append(rm_wiki.format_table(
                    [
                        "First revision",
                        "Baseline trunk revision",
                        "Baseline revision",
                        "Candidate trunk revision",
                        "Candidate revision"
                    ],
                    [[lb.revision_link(i, link_type=lb.LinkType.wiki) for i in [
                        release_item_changes.get(rm_changelog.ChangeLogFormatter.FIRST_REV),
                        release_item_changes.get("baseline_rev_trunk"),
                        release_item_changes.get("baseline_revision"),
                        release_item_changes.get("candidate_rev_trunk"),
                        release_item_changes.get("candidate_revision"),
                    ] if i is not None]]
                ))
                release_item = release_item_changes.get("release_item")
                if release_item:
                    descr.append("**Compared with** {}".format(rm_core.ReleasedResource.format_from_dict(release_item)))
                formatted_changes = self.format_important_changes(c_info, release_item_changes["changes"])
                descr.extend(formatted_changes)
        return u"\n".join(descr)

    def get_component_info(self):
        return rmc.get_component(self.Parameters.component_name)

    def on_enqueue(self):
        queue = self.c_info.notify_cfg__st__queue
        if queue:
            self.Requirements.semaphores = ctt.Semaphores(
                acquires=[
                    ctt.Semaphores.Acquire(
                        name="CreateStartrekTicket_{}_{}".format(queue, self.Parameters.component_name),  # RMDEV-3238
                        capacity=1,
                    ),
                ],
                release=(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH),
            )

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)

        major_release_num = (
            int(self.Parameters.changelog.major_release_num)
            if self.Parameters.changelog.major_release_num else 0
        )

        self.Context.scope_number = major_release_num
        issue = self.st_helper.find_ticket_by_release_number(major_release_num, self.c_info, fail=False)
        status = self.st_helper.ISSUE_FOUND if issue else "not found"

        create_issue = not issue
        update_issue = issue and self.Parameters.update_issue_if_exists
        if create_issue or update_issue:
            changelog = rm_ch.get_rm_changelog(self.Parameters.changelog)
            description = self.generate_description(self.c_info, changelog)
            logging.info("Description:\n%s", description)

            if update_issue:
                full_description = u"{}\n{}\n\n//Ticket was updated by task: {}//".format(
                    self.c_info.st_description(major_release_num), description, lb.sb_item_wiki_link(self.id, "task")
                )
                logging.info("Full description:\n%s", full_description)
                if not self.Parameters.debug_mode:
                    issue.update(description=full_description)
                    self.Parameters.issue_updated = True
                    status = self.st_helper.ISSUE_UPDATED
                else:
                    logging.info("Debug mode, no need to actually update issue")
            else:
                followers = self.get_followers(self.c_info, changelog)
                logging.debug("Ticket followers: %s", followers)
                all_related_tickets = self.get_related_tickets(self.c_info, changelog)
                if not self.Parameters.debug_mode:
                    # filter external followers: spike for RMDEV-1026
                    blacklist = [
                        'phill',
                    ]
                    followers = [follower for follower in followers if str(follower) not in blacklist]
                    issue = self.st_helper.create_issue(self, self.c_info, followers, description, major_release_num)
                    self.link_related_tickets(self.c_info, issue, all_related_tickets)
                    status = self.st_helper.ISSUE_CREATED
                else:
                    logging.info("Debug mode, no need to actually create issue")

        self.Context.startrek_issue = issue.key
        self.Parameters.startrek_issue = issue.key
        self.Context.issue_start_time = tu.to_unixtime(issue.createdAt)
        self.Context.issue_status = all_to_str(issue.status.name)
        self.Context.save()

        tracker_issue_message = "[{}] Tracker issue {}: {}".format(self.c_info.name, status, lb.st_link(issue.key))
        self.set_info(tracker_issue_message, do_escape=False)

        if status == self.st_helper.ISSUE_CREATED and not self.Parameters.debug_mode:
            self.notify_issue_created(issue, tracker_issue_message)

    def notify_issue_created(self, issue, tracker_issue_message):
        self.notify_telegram(
            self.c_info, issue, "{}\n #issue #r{}".format(tracker_issue_message, self.Context.scope_number)
        )
        self.c_info.custom_st_update(self.st_helper, issue)
        logging.info("Try to send email notification")
        self._create_email_notifications(self.c_info, issue.key)

    def _create_email_notifications(self, c_info, startrek_issue):
        letter = c_info.notify_cfg__mail__get_start_letter(
            major_release_num=c_info.last_scope_num,
            revision=c_info.last_rev,
            st_issue_key=startrek_issue,
            changelog_wiki_page=c_info.changelog_cfg__wiki_page,
        )
        if letter:
            nh.email2(self, **letter)
        else:
            logging.warning("No start letter found for '%s'!", c_info.name)

    def notify_telegram(self, c_info, issue, message):
        need_pin = False
        if hasattr(c_info, "pin_message"):
            message = c_info.pin_message(
                lb.st_link(issue.key),
                self.Context.scope_number if 'scope_number' in self.Context else None,
            )
            need_pin = True
        setattr(self.Context, rm_notify.TM_MESSAGE, message)
        logging.debug("Telegram message %s", message)
        if not self.Parameters.debug_mode:
            try:
                self.send_tm_notify(need_pin)
            except Exception as exc:
                eh.log_exception('Telegram notify error', exc, task=self)

    def format_important_changes(self, c_info, changes):
        result = []
        important_changes, some_important_changes_skipped = self.get_important_changes(c_info, changes)
        if not important_changes:
            return result
        if some_important_changes_skipped:
            result.append(
                "!!Не все важные изменения попали в тикет. "
                "Предположительно, нужно увеличить important_changes_limit в конфиге компоненты!!"
            )
        added_changes = [c for c in important_changes if c["added"]]
        lost_changes = [c for c in important_changes if not c["added"]]
        cut = "+" if c_info.notify_cfg__st__hide_commits_under_cut else ""
        if lost_changes:
            result.extend([
                "====={}Коммиты из предыдущего релиза, предположительно отсутствующие в этом:".format(cut),
                self.format_changes_table(lost_changes, c_info)
            ])
        if added_changes:
            result.extend([
                "====={}Добавленные коммиты:".format(cut),
                self.format_changes_table(added_changes, c_info)
            ])
        return result

    def format_changes_table(self, changes, c_info):
        if self.Parameters.group_changelog_by_tickets:
            return rm_wiki.group_changes_by_tickets(c_info, changes)
        else:
            changelog_table_data = rm_cf.ChangeLogDataWiki(
                c_info,
                [
                    rm_ch.ChangeLogEntry.Attrs.REVISION,
                    rm_ch.ChangeLogEntry.Attrs.REVIEW_IDS,
                    rm_ch.ChangeLogEntry.Attrs.COMMIT_AUTHOR,
                    rm_ch.ChangeLogEntry.Attrs.SUMMARY,
                    rm_ch.ChangeLogEntry.Attrs.STARTREK_TICKETS,
                ],
                changes
            )
            return rm_cf.ChangeLogWiki().format_table(changelog_table_data)

    def get_important_changes(self, c_info, changes):
        # type: (Any, List) -> Tuple[List, bool]

        important_changes, trimmed = rm_ch.get_important_changes(
            release_item_changes=changes,
            commit_importance_threshold=c_info.notify_cfg__st__commit_importance_threshold,
            important_changes_limit=c_info.notify_cfg__st__important_changes_limit,
        )

        if trimmed:
            self.set_info("Show only first {} important commits in startrek. Total amount = {}".format(
                c_info.notify_cfg__st__important_changes_limit, len(important_changes)
            ))

        return important_changes, trimmed

    def get_followers(self, c_info, changelog):
        add_followers = {self.author}  # SEARCH-3278
        if not c_info.notify_cfg__st__add_commiters_as_followers:
            logging.info("Commiters will not be added to followers because 'add_commiters_as_followers' is disabled")
        else:
            abc_commiters = []
            if c_info.notify_cfg__st__abc_followers is not None:
                try:
                    abc_helper = abc_client.AbcClient(rm_sec.get_rm_token(self))
                    abc_commiters = abc_helper.get_people_from_service(
                        c_info.notify_cfg__st__abc_followers.component_id,
                        c_info.notify_cfg__st__abc_followers.role_id,
                    )
                    logging.debug(
                        "All people from abc: %s with role_id %s",
                        abc_commiters,
                        c_info.notify_cfg__st__abc_followers.role_id,
                    )
                except Exception as exc:
                    eh.log_exception("Got exception while init AbcApi\nCannot get people from abc", exc, task=self)
            for release_item_changes in changelog["all_changes"]:
                important_changelog, _ = self.get_important_changes(c_info, release_item_changes["changes"])
                for i in important_changelog:
                    st_queues = set(ticket.split("-")[0] for ticket in i["startrek_tickets"])
                    if st_queues and st_queues.issubset(c_info.notify_cfg__st__banned_queues):
                        continue
                    if i["commit_author"] in c_info.notify_cfg__st__banned_people:
                        continue
                    if abc_commiters and i["commit_author"] not in abc_commiters:
                        continue
                    add_followers.add(i["commit_author"])

        return self.filter_by_user_allowance(self.filter_robots(add_followers), c_info)

    @staticmethod
    def filter_robots(add_followers):
        return (user for user in add_followers if utils.is_human(user))

    @staticmethod
    def filter_by_user_allowance(add_followers, c_info):
        return [
            user for user in add_followers
            if user not in rm_users.RM_USERS or rm_users.RM_USERS[user].allow_ticket_following(c_info.name)
        ]

    def get_related_tickets(self, c_info, changelog):
        all_related_tickets = set()
        for release_item_changes in changelog["all_changes"]:
            important_changelog, _ = self.get_important_changes(c_info, release_item_changes["changes"])
            for changelog_entry in important_changelog:
                all_related_tickets |= c_info.filter_tickets(changelog_entry)

        logging.info("All related tickets to link %s", all_related_tickets)
        return list(all_related_tickets)

    def link_related_tickets(self, c_info, release_ticket, all_related_tickets):
        if not self.Parameters.debug_mode:
            c_info.st_create_links(release_ticket, all_related_tickets)

    def _get_rm_proto_event_hash_items(self, event_time_utc_iso, status=None):
        return (
            self.Parameters.component_name,
            'TicketHistory',
            self.Parameters.startrek_issue,
            json.dumps(
                [{
                    "status": self.Context.issue_status.lower(),
                    "start_time": self.Context.issue_start_time,
                    "end_time": 0,
                }],
                sort_keys=True,
            ),
        )

    def _get_rm_proto_event_specific_data(self, rm_proto_events, event_time_utc_iso, status=None):

        if self.Parameters.issue_updated:
            return

        specific_data = rm_proto_events.TicketHistoryData(
            ticket_key=self.Parameters.startrek_issue,
            scope_number=str(self.Context.scope_number),
            job_name=self.get_job_name_from_gsid(),
            ticket_history_latest_status=self.Context.issue_status.lower(),
        )

        specific_data.items.extend([rm_proto_events.TicketHistoryDataItem(
            status=self.Context.issue_status.lower(),
            start_timestamp=self.Context.issue_start_time,
        )])

        return {
            'ticket_history_data': specific_data,
        }
