# -*- coding: utf-8 -*-
import datetime
import json
import logging
from dateutil.parser import parse as dt_parse

from sandbox.projects.common import binary_task
import sandbox.projects.common.link_builder as lb
import sandbox.projects.release_machine.tasks.base_task as rm_bt
import sandbox.projects.release_machine.helpers.changelog_helper as rm_changelog
import sandbox.projects.release_machine.changelogs as rm_ch
import sandbox.projects.release_machine.helpers.changelog_formatter as rm_cf
import sandbox.projects.release_machine.helpers.staff_helper as rm_staff
import sandbox.projects.release_machine.helpers.wiki_helper as rm_wiki
import sandbox.projects.release_machine.input_params2 as rm_params
import sandbox.projects.release_machine.rm_notify as rm_notify
import sandbox.projects.release_machine.components.all as rmc
import sandbox.projects.release_machine.core as rm_core
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.resources as rm_res
import sandbox.projects.release_machine.security as rm_sec
import sandbox.sdk2 as sdk2


@rm_notify.notify2()
class CreateWikiPage(rm_bt.BaseReleaseMachineTask):
    class Requirements(task_env.TinyRequirements):
        disk_space = 2 * 1024  # 2 Gb

    class Parameters(rm_params.DefaultReleaseMachineParameters):
        kill_timeout = 15 * 60  # 15 min
        _lbrp = binary_task.binary_release_parameters(stable=True)
        changelog = sdk2.parameters.Resource("Changelog", resource_type=rm_res.RELEASE_MACHINE_CHANGELOG)
        startrek_issue = sdk2.parameters.String("Startrek issue")
        group_changelog_by_tickets = sdk2.parameters.Bool(
            "Group changelog by ticket",
            description="RMDEV-209",
            default=False,
        )
        last_rev = sdk2.parameters.Integer(
            "Last revision",
            description="RMDEV-482",
            default=0,
        )

    class Context(rm_bt.BaseReleaseMachineTask.Context):
        wiki_url = ""

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)
        c_info = rmc.get_component(self.Parameters.component_name)
        if not c_info.changelog_cfg__wiki_page:
            self.set_info("Changelog_wiki_page is None. Nothing was done.")
            self.do_set_rm_event_data()
            return None
        key = rm_sec.get_rm_token(self)
        wiki_api = rm_wiki.WikiApi(key)
        staff_api = rm_staff.StaffApi(key)
        changelog_url = self.create_changelog_on_wiki(c_info, wiki_api)
        if not self.Parameters.debug_mode:
            self.delete_overdue_changelogs_on_wiki(c_info, wiki_api, changelog_url)
            self.change_wiki_page_owner(c_info, wiki_api, staff_api, changelog_url)
        self.set_info("Wiki page created: {}".format(lb.wiki_page_link(changelog_url)), do_escape=False)

    @staticmethod
    def delete_overdue_changelogs_on_wiki(c_info, wiki_api, changelog_url):
        if not hasattr(c_info, 'changelog_cfg__wiki_page_ttl_days') or c_info.changelog_cfg__wiki_page_ttl_days is None:
            # Do nothing if no ttl provided or ttl is None (i.e., should live forever)
            return
        logging.info("Going to delete overdue changelog wiki pages")
        parent_cluster, parent_url = wiki_api.get_page_parent_cluster(changelog_url)
        for page in parent_cluster:
            created_at = dt_parse(page['created_at'])
            if created_at + datetime.timedelta(days=c_info.changelog_cfg__wiki_page_ttl_days) > datetime.datetime.now():
                continue
            current_changelog_wiki_url = "{parent_url}{supertag}".format(
                parent_url=parent_url,
                supertag=page['supertag']
            )
            wiki_api.check_wiki_path_correctness(current_changelog_wiki_url)
            wiki_api.delete_page(current_changelog_wiki_url)
            logging.info("{} has been deleted".format(current_changelog_wiki_url))

    def create_changelog_on_wiki(self, c_info, wiki_api):
        changelog = rm_ch.get_rm_changelog(self.Parameters.changelog)
        title = "Changelog for {} (Major release number: {})".format(
            c_info.name, self.Parameters.changelog.major_release_num,
        )
        if int(self.Parameters.changelog.minor_release_num) > 0:
            url = c_info.changelog_minor_url(
                self.Parameters.changelog.major_release_num,
                self.Parameters.changelog.minor_release_num
            )
            title += " (Minor release number: {})".format(self.Parameters.changelog.minor_release_num)
        else:
            url = c_info.changelog_major_url(self.Parameters.changelog.major_release_num)
        data = json.dumps({
            "title": title,
            "body": self.format_log_for_wiki(c_info, changelog),
        })
        self.do_set_rm_event_data(url=url)

        logging.info("Full wiki log: %s", data)
        if not self.Parameters.debug_mode:
            wiki_api.check_wiki_path_correctness(url)
            wiki_api.post_page(url, data)
        return url

    def format_log_for_wiki(self, c_info, changelog):
        release = [str(changelog["major_release_num"])]
        if changelog["minor_release_num"]:
            release.append(str(changelog["minor_release_num"]))
        log_parts = [
            "====Changelog for {}. Release candidate: {}".format(c_info.name, "-".join(release)),
            "//Creation time: {}//".format(changelog.get("creation_time"))
        ]
        if self.Parameters.startrek_issue:
            log_parts.append("Startrek: {}".format(lb.st_link(self.Parameters.startrek_issue, plain=True)))
        for changes in changelog["all_changes"]:
            if "release_items" not in changes:
                # this is the most new changelog format
                release_item = changes.get("release_item")
                if release_item:
                    log_parts.append("\n**Compared with {}**".format(
                        rm_core.ReleasedResource.format_from_dict(release_item)
                    ))
                changelog_table_data = rm_cf.ChangeLogDataWiki(
                    c_info,
                    tuple([
                        rm_ch.ChangeLogEntry.Attrs.REVISION,
                        rm_ch.ChangeLogEntry.Attrs.REVIEW_IDS,
                        rm_ch.ChangeLogEntry.Attrs.COMMIT_AUTHOR,
                        rm_ch.ChangeLogEntry.Attrs.SUMMARY,
                        rm_ch.ChangeLogEntry.Attrs.STARTREK_TICKETS,
                        rm_ch.ChangeLogEntry.Attrs.TE_PROBLEMS,
                        rm_ch.ChangeLogEntry.Attrs.COMMIT_IMPORTANCE,
                        rm_ch.ChangeLogEntry.Attrs.ADDED,
                        rm_ch.ChangeLogEntry.Attrs.REASONS,
                    ]),
                    changes["changes"]
                )
                log_parts.append(rm_cf.ChangeLogWiki().format_table(changelog_table_data))
                continue
            if changes["release_items"]:
                log_parts.append("Items:\n{}".format("\n".join(changes["release_items"])))
            log_parts.append("Revisions: {}-{}".format(changes["first_revision"], changes["last_revision"]))
            if self.Parameters.group_changelog_by_tickets:
                log_parts.append(rm_wiki.group_changes_by_tickets(c_info, changes["changes"]))
            else:
                log_parts.append(rm_wiki.format_table(
                    header=rm_changelog.ChangeLogFormatter.wiki_header,
                    rows=[rm_changelog.ChangeLogFormatter.wiki_row(c_info, i) for i in changes["changes"]]
                ))
        return "\n".join(log_parts)

    def change_wiki_page_owner(self, c_info, wiki_api, staff_api, url):
        new_owner = c_info.get_responsible_for_wiki_page()
        logging.debug("New page owner is %s", new_owner)
        if wiki_api.get_page_owner(url) == new_owner:
            self.set_info("Current user is already {}".format(new_owner))
            return
        new_owner_uid = staff_api.get_person_id("persons", {"login": new_owner})
        wiki_api.set_page_authors(url, [new_owner_uid])

    def do_set_rm_event_data(self, url=None):
        if url:
            self.Context.wiki_url = lb.wiki_page_link(url, plain=True)
            self.Context.save()

    def _get_rm_proto_event_hash_items(self, event_time_utc_iso, status=None):
        return (
            self.Parameters.component_name,
            'NewWikiPage',
            str(self.Parameters.changelog.major_release_num),
            str(self.Parameters.changelog.minor_release_num or self.Parameters.changelog.major_release_num or 0),
            self.Context.wiki_url,
            "TASK_{}".format(status or self.status),
        )

    def _get_rm_proto_event_specific_data(self, rm_proto_events, event_time_utc_iso, status=None):
        specific_data = rm_proto_events.NewWikiPageData(
            scope_number=str(self.Parameters.changelog.major_release_num),
            tag_number=str(self.Parameters.changelog.minor_release_num or self.Parameters.changelog.major_release_num),
            wiki_url=self.Context.wiki_url,
            job_name=self.get_job_name_from_gsid(),
        )

        return {
            'new_wiki_page_data': specific_data,
        }
