import logging
import requests

import sandbox.projects.common.error_handlers as eh
import sandbox.projects.release_machine.components.all as rmc
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.input_params2 as rm_params
import sandbox.projects.release_machine.rm_notify as rm_notify
import sandbox.projects.release_machine.tasks.base_task as rm_bt
import sandbox.sdk2 as sdk2
from sandbox.projects.common import binary_task


@rm_notify.notify2()
class FindBuildFromTrunk(rm_bt.BaseReleaseMachineTask):
    """
        Hack for Branched rm-components. Allows to use trunk builds as builds from tag.
        For first tag only!
    """
    class Requirements(task_env.TinyRequirements):
        pass

    class Parameters(rm_params.ComponentName2):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        kill_timeout = 20 * 60  # 20 min
        branch_num = sdk2.parameters.Integer("Branch number", required=True)
        resource_type_to_find = sdk2.parameters.String("Resource type to find", required=True)
        resource_marker = sdk2.parameters.String("Optional Resource marker", required=False)
        with sdk2.parameters.Output:
            found_resource = sdk2.parameters.Resource("Found resource", required=False)

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)
        c_info = rmc.COMPONENTS[self.Parameters.component_name]()
        eh.ensure(
            c_info.is_branched,
            "This task is only for Branched components"
        )
        branch_folder_name = c_info.svn_cfg__branch_folder_name(self.Parameters.branch_num)
        logging.info("Try to find trunk build for %s, branch %s", c_info.name, branch_folder_name)
        branch_path = c_info.full_branch_path(self.Parameters.branch_num)
        self._ensure_branch_head_relates_to_trunk(branch_path)
        trunk_revision = self._get_trunk_revision(branch_path)
        attrs = {
            "rm_component_name": self.Parameters.component_name,
            "svn_revision": trunk_revision,
        }
        if self.Parameters.resource_marker:
            attrs["resource_name"] = self.Parameters.resource_marker
        found_resource = sdk2.Resource.find(
            type=self.Parameters.resource_type_to_find,
            attrs=attrs
        ).first()
        if not found_resource:
            self.set_info("Resource for branch {} (trunk rev {}) not found".format(
                branch_folder_name, trunk_revision
            ))
        else:
            self.set_resource_attr(found_resource.id, 'major_release_num', self.Parameters.branch_num)
            self.set_resource_attr(found_resource.id, 'minor_release_num', 1)
            self.Parameters.found_resource = found_resource

    def set_resource_attr(self, resource_id, attribute_name, attribute_value):
        try:
            self.server.resource[resource_id].attribute.create(name=attribute_name, value=attribute_value)
        except requests.HTTPError as exc:
            if exc.response.status_code != 409:
                raise
            message = "{} already exists".format(attribute_name)
            logging.debug(message)
            self.set_info(message)

    @staticmethod
    def _get_trunk_revision(branch_path):
        svn_log = sdk2.vcs.svn.Arcadia.log(
            branch_path,
            revision_from="HEAD",
            revision_to="1",
            limit=2,
        )
        trunk_revision = svn_log[-1]["revision"]
        logging.info("Found trunk revision: %s", trunk_revision)
        return trunk_revision

    @staticmethod
    def _ensure_branch_head_relates_to_trunk(branch_path):
        svn_log = sdk2.vcs.svn.Arcadia.log(
            branch_path,
            revision_from="HEAD",
            revision_to="1",
            limit=2,
            stop_on_copy=True
        )
        eh.ensure(len(svn_log) == 1, "There are some changes in branch, unable to use trunk build for it")
