# -*- coding: utf-8 -*-
import io
import json
import logging
import os
import re
import time

import yaml

import sandbox.projects.common.betas.beta_api as beta_api
import sandbox.projects.common.file_utils as fu
import sandbox.projects.common.link_builder as lb
import sandbox.projects.common.error_handlers as eh
import sandbox.projects.common.decorators as deco
import sandbox.projects.release_machine.client as rm_client
import sandbox.projects.release_machine.components.all as rmc
import sandbox.projects.release_machine.components.components_info as rm_comp
import sandbox.projects.release_machine.components.config_core.yappy as yappy_cfg
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.helpers.startrek_helper as sh
import sandbox.projects.release_machine.helpers.svn_helper as rm_svn
import sandbox.projects.release_machine.yappy as yappy_helper
import sandbox.projects.release_machine.input_params2 as rm_params2
import sandbox.projects.release_machine.rm_notify as rm_notify
import sandbox.projects.release_machine.security as rm_sec
import sandbox.projects.release_machine.tasks.base_task as rm_bt
from sandbox.sdk2.vcs import svn
from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common import resource_selectors

logger = logging.getLogger("yappy-generator")


RM_STATE_BETA_NAME_KEY = 'beta_name'
MAX_WAIT_CONSISTENCY_NUM = 15


class YappyBetaConfig(sdk2.Resource):
    """ Resource with config for yappy crawler beta """
    pass


@rm_notify.notify2()
class GenerateYappyBeta(rm_bt.BaseReleaseMachineTask):
    """
        **Release-machine**

        Таск для генерации бет в Yappy для компонент, описанных здесь:
            projects/release_machine/components/
    """
    _WAIT_MULTIPLIER = 180
    _BETA_CREATION_TIME = 600  # 10 min

    class Requirements(task_env.StartrekRequirements):
        disk_space = 128

    class Parameters(rm_params2.ComponentNameResources):
        _lbrp = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.String("Update beta mode") as update_beta_mode:
            update_beta_mode.values.APPEND = update_beta_mode.Value("append")
            update_beta_mode.values.REPLACE = update_beta_mode.Value("replace", default=True)

        with sdk2.parameters.String("Get beta name from") as beta_name_source:
            beta_name_source.values.CURRENT = beta_name_source.Value("current resources", default=True)
            beta_name_source.values.LAST_RELEASED = beta_name_source.Value("Last released resources")
            beta_name_source.values.STRING = beta_name_source.Value("String")
        with beta_name_source.value["STRING"]:
            patch_name = sdk2.parameters.String("Patch name")

        max_wait_num = sdk2.parameters.Integer(
            "Max amount of attempts of waiting consistency",
            default=MAX_WAIT_CONSISTENCY_NUM,
        )

        beta_conf_type = sdk2.parameters.String("Beta configuration type")
        release_number = sdk2.parameters.Integer("Release number")
        force_start_beta = sdk2.parameters.Bool("Force start beta", default=False)
        get_beta_names_from_state = sdk2.parameters.Bool(
            "Get prev beta names from RM state",
            default=False,
            description="When True and update beta mode is REPLACE then beta names are retrieved from Release Machine"
                        "state. Otherwise beta names are constructed using branch and tag numbers"
        )

        with sdk2.parameters.Output:
            new_beta_name = sdk2.parameters.String("Generated beta name")
            new_beta_url = sdk2.parameters.String("Generated beta url (without tld)")

    class Context(rm_bt.BaseReleaseMachineTask.Context):
        branch_num = None
        tag_num = None
        beta_name = None
        wait_num = 0
        start_event_created = False

    @deco.memoized_property
    def token(self):
        return sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.YAPPY_TOKEN_NAME)

    @deco.memoized_property
    def c_info(self):
        return rmc.get_component(self.Parameters.component_name)

    @deco.memoized_property
    def yappy_cfg(self):

        logging.debug(
            "Try to get yappy beta config for component: %s, type: %s",
            self.c_info.name,
            self.Parameters.beta_conf_type,
        )

        yappy_cfg = self.c_info.yappy_cfg.betas.get(self.Parameters.beta_conf_type)

        eh.verify(
            yappy_cfg,
            "Cannot get yappy beta config component: {comp}, type: {beta_type}".format(
                comp=self.c_info.name, beta_type=self.Parameters.beta_conf_type
            ),
        )

        return yappy_cfg

    @deco.memoized_property
    def yappy_api_client(self):
        return beta_api.BetaApi.fromurl(token=self.token, id=self.id)

    @property
    def beta_name(self):

        if not self.Context.beta_name:
            self._set_beta_name()

        return self.Context.beta_name

    @deco.memoized_property
    def job_name(self):
        return self.get_job_name_from_gsid()

    def on_enqueue(self):
        # with self.memoize_stage.build_tasks:
        #     if self.Parameters.use_build_tasks:
        #         finish_statuses = tuple(ctt.Status.Group.FINISH) + tuple(ctt.Status.Group.BREAK)
        #         raise sdk2.WaitTask(filter(None, iter(self.Parameters.build_tasks)), finish_statuses, True)
        pass

    def _set_beta_name(self):
        patch_name = self._get_patch_file_name(self.c_info, self.yappy_api_client)
        self.Context.beta_name = self.yappy_cfg.get_beta_name(patch_name=patch_name)
        logging.debug("Set beta_name=%s", self.Context.beta_name)
        self.Context.save()

    def on_execute(self):
        super(GenerateYappyBeta, self).on_execute()

        self.send_rm_proto_event()

        # Check and log RM Yappy config
        logging.debug(self.yappy_cfg)

        # Check not implemented features
        # eh.verify(not self.Parameters.use_build_tasks, "Not implemented yet")

        # Create and start beta
        self._set_beta_name()
        append_mode = self.Parameters.update_beta_mode == "APPEND"
        self._create_yappy_beta(self.yappy_cfg, self.c_info, self.yappy_api_client, append_mode)

        if self.Parameters.debug_mode:
            self.set_info("Tha task is run in debug mode. Not going to start or wait beta")
            return

        st_helper = sh.STHelper(sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME))

        # RMINCIDENTS-169
        self.set_info(
            'Start and wait for beta consistency: {}'.format(self._beta_link()),
            do_escape=False,
        )

        try:
            time_start = int(time.time())
            if append_mode or self.Parameters.force_start_beta:
                try:
                    # Wait 30 seconds after beta creation to solve the competitive allocation problem
                    time.sleep(30)
                    self.yappy_api_client.yappy.start_beta(self.Context.beta_name)
                    consistent = True
                except RuntimeError:
                    consistent = False
            else:
                # Wait for beta activation until check beta state RMDEV-629
                logging.debug("Sleep for 5 minutes to wait Yappy beta activation")
                time.sleep(300)
                consistent = False

            if not consistent:
                for i in range(self.Parameters.max_wait_num):
                    consistent = self.yappy_api_client.yappy.sleep_for_status(
                        self.Context.beta_name,
                        "CONSISTENT",
                        retries=6,
                        delay=30 * (i + 1),
                        multiplier=1.5,
                    )
                    if consistent:
                        break

                    self.set_info(
                        "Beta {} is not consistent still. "
                        "There might be some problems with yappy or quota for this type of betas. "
                        "Please, check beta state in Yappy via given link "
                        "or check task logs for more details".format(self._beta_link()),
                        do_escape=False,
                    )
                    self.notify_if_too_long(time_start, self.c_info)

            if not consistent:
                self.on_failure_st_comment(self.c_info, st_helper)
                eh.check_failed("Waiting beta consistency timeout")
        except RuntimeError:
            self.on_failure_st_comment(self.c_info, st_helper)

        try:

            self.Parameters.new_beta_name = self.Context.beta_name
            self.Parameters.new_beta_url = self.yappy_api_client.retrieve_api_beta(self.Context.beta_name).get(
                'domainNameWithoutTld',
                '',
            )
            if not self.Parameters.new_beta_url:
                self.set_info("Unable to retrieve domain name for {} from Yappy".format(self.Context.beta_name))

        except Exception as e:
            self.log_exception(
                "Unexpected error while trying to retrieve beta domain for {}".format(self.Context.beta_name),
                e,
            )

        self.on_success_st_comment(self.c_info, st_helper)

    def notify_if_too_long(self, time_start_sec, c_info):
        """
        Send notification if task is running for too long.
        RMDEV-3

        :param time_start_sec: start time in seconds
        :param c_info: component's c_info
        """
        time_now_sec = int(time.time())
        notify_error = rm_notify.notify_if_acceptance_delayed(c_info, self, time_now_sec - time_start_sec)
        if notify_error:
            logging.error(notify_error)

    def on_failure_st_comment(self, c_info, st_helper, status='FAILED', force_fail=True):
        if isinstance(c_info, rm_comp.mixin.Startreked):
            mode_msg = "Updating" if self.Parameters.update_beta_mode == "APPEND" else "Creating"
            st_helper.write_grouped_comment(
                rm_const.TicketGroups.BetaCreation,
                "{} Yappy beta {} !!{}!!".format(mode_msg, self.Context.beta_name or "???", status),
                "See details in {}".format(lb.task_wiki_link(self.id)),
                self.Parameters.release_number,
                c_info,
            )
        if force_fail:
            eh.check_failed("Failed to start beta {}".format(self.Context.beta_name))

    def on_success_st_comment(self, c_info, st_helper):
        mode_msg = "Updating" if self.Parameters.update_beta_mode == "APPEND" else "Creating"
        if isinstance(c_info, rm_comp.mixin.Startreked):
            build_tasks = set()
            for comp_res_key, comp_res_id in self.Parameters.component_resources.items():
                try:
                    binary_res = sdk2.Resource[comp_res_id]
                except Exception as exc:
                    logging.exception("Couldn't get resource with id %s, exception %s", comp_res_id, exc)
                    continue
                build_tasks.add(binary_res.task_id)
            msg = (
                "Generated Yappy Beta UI: (({main_url}b/{name} {name}))\n"
                "links: [{ru}] [{ua}] [{kz}] [{comtr}]\n"
                "Generated via sandbox task {task_link}\n"
            ).format(
                main_url=rm_const.Urls.YAPPY,
                name=self.Context.beta_name,
                ru=self._front_url("ru"),
                ua=self._front_url("ua"),
                kz=self._front_url("kz"),
                comtr=self._front_url("com.tr"),
                task_link=lb.task_wiki_link(self.id),
            )
            if build_tasks:
                msg += "Build tasks: {}".format(
                    "; ".join([lb.task_wiki_link(task_id) for task_id in list(build_tasks)])
                )

            st_helper.write_grouped_comment(
                rm_const.TicketGroups.BetaCreation,
                "{} Yappy beta {} **!!(green)SUCCEED!!**".format(mode_msg, self.Context.beta_name or "???"),
                msg,
                self.Parameters.release_number,
                c_info,
            )
            self.set_info(msg)

    @deco.retries(3, delay=5, exceptions=(svn.SvnError,))
    def _create_yappy_beta(self, yappy_cfg, c_info, yappy_api, append_mode):

        if not append_mode:
            self.stop_prev_betas(c_info, yappy_api)

        if getattr(yappy_cfg, 'new_yappy', False):
            logger.warning('using new Yappy API')

            suffix = self._get_patch_file_name(c_info, yappy_api)

            beta_name = self.Context.beta_name

            patch_map = {}
            for patch in yappy_cfg.patches:
                patch_dir = patch.patch_dir if hasattr(patch, 'patch_dir') else patch.patch_file
                patch_map[patch_dir] = self._process_yappy_patch(patch)

            logger.debug('patch map %s', patch_map)

            # Beta already exists -> update patches
            if yappy_api.beta_exists(beta_name):
                patches = []

                if append_mode:
                    logger.info('update beta {}, case mode is append'.format(beta_name))

                    beta = yappy_api.retrieve_api_beta(beta_name)
                    beta_patch_map = {
                        bc.get('templateId', ''): bc.get('patch', {})
                        for bc in beta.get('components', [])
                    }

                    logger.debug('current betas patch map %s', beta_patch_map)

                    for template_id, patch in patch_map.items():
                        eh.verify(
                            template_id in beta_patch_map,
                            "Can't update beta, component with template id {} doesn't exist".format(template_id),
                        )

                        self._update_config_file(beta_patch_map[template_id], patch)
                        patches.append({
                            'patch': beta_patch_map[template_id],
                            'template_id': template_id,
                        })

                else:
                    for template_id, patch in patch_map.items():
                        patches.append({
                            'template_id': template_id,
                            'patch': patch,       # patch.patch_dir is equal component.template_id
                        })

                api_method, params = yappy_api.update_patches, (beta_name, patches)

            else:
                logger.info('create new beta {}'.format(beta_name))
                template = getattr(yappy_cfg, 'template_name', None)
                api_method, params = yappy_api.create_beta_from_template, (template, suffix, patch_map)

            if not self.Parameters.debug_mode:
                api_method(*params)

            return
        else:
            logger.warning('using old Yappy API')

        # Checkout patches
        svn.Arcadia.checkout(yappy_helper.ARC_PATCHES_DIR, "patches")

        patch_name = self._get_patch_file_name(c_info, yappy_api)

        svn_add_paths = []
        added_paths = []
        for patch in yappy_cfg.patches:
            patch_file = patch.get_patch_name(patch_name=patch_name)
            patch_content = self._process_yappy_patch(patch)
            config_res = self._create_patch_resource(patch_file)
            if append_mode:
                patch_content = self._get_updated_patch_content(str(config_res.path), patch_content)

            self._add_patch_file(config_res, patch_content, svn_add_paths, added_paths)
            self._check_patch_content(patch_content)

        for p in svn_add_paths:
            svn.Svn.add(p, parents=True)
        if not self.Parameters.debug_mode:
            self._commit_changes(added_paths)

    @staticmethod
    def _check_patch_content(patch_content):
        # Check that resources managed as SANDBOX_RESOURCE contain resource id RMDEV-613
        for res in patch_content["resources"]:
            if res["manageType"] == "SANDBOX_RESOURCE":
                eh.verify(
                    "sandboxResourceId" in res,
                    "Failed to procces patch, resource with "
                    "mangeType=SANDBOX_RESOURCE must have sandboxResourceId. Resource: {}".format(res)
                )

    def _create_patch_resource(self, patch_file):
        patch_path = os.path.join("patches", patch_file)
        config_res = YappyBetaConfig(self, "Yappy config {}".format(patch_file), patch_path)

        return config_res

    def _get_updated_patch_content(self, patch_file_path, new_patch_data):
        eh.verify(
            os.path.exists(patch_file_path), "Can't update beta, config: {} doesn't exist".format(patch_file_path)
        )
        loaded_data = yaml.load(fu.read_file(patch_file_path))
        resources = [res["localPath"] for res in loaded_data.get("resources", [])]

        logger.debug("load data from '%s':\n%s", patch_file_path, loaded_data)
        self._update_config_file(loaded_data, new_patch_data)
        # Check that updated patch data contains all resources from origin patch RMDEV-613
        new_resources = [res["localPath"] for res in loaded_data.get("resources", [])]
        for res in resources:
            eh.verify(res in new_resources, "Updated patch does not contain resource: {}".format(res))

        return loaded_data

    @staticmethod
    def _add_patch_file(config_res, patch_content, svn_add_paths, added_paths):
        """
        Check that config res file exists, add to svn_add_paths if it does not. Add added paths to added_paths
        :param config_res: Resource with created patch
        :param pathch_content: Dict with created patch content
        :param svn_add_paths: List with paths. Used to add new paths to 'svn add' command
        :param added_paths: List with paths. Used to commit new changes to svn.
        :return: None
        """
        output_res_path = str(config_res.path)
        if not os.path.exists(output_res_path):
            svn_add_paths.append(output_res_path)
        output_res_root = os.path.dirname(output_res_path)
        if not os.path.exists(output_res_root):
            os.makedirs(output_res_root)
        with io.open(output_res_path, "w") as yaml_file:
            yaml.dump(patch_content, yaml_file, default_flow_style=False, allow_unicode=True)
        sdk2.ResourceData(config_res).ready()  # RMDEV-143
        added_paths.append(output_res_path)

    def _get_patch_file_name(self, c_info, yappy_api):
        patch_name = None
        if self.Parameters.beta_name_source == "CURRENT":
            # Get info about tag and branch from main release item build task
            branch_n, tag_n = self._get_branch_and_tag_from_params(c_info)
            patch_name = "{}-{}".format(branch_n, tag_n)
        elif self.Parameters.beta_name_source == "STRING":
            # Set patch filename from passed string
            patch_name = self.Parameters.patch_name
        elif self.Parameters.beta_name_source == "LAST_RELEASED":
            # Get info about tag and branch from main release item last release
            sample_beta, branch_n, tag_n = yappy_helper.get_sample_beta(c_info, self, yappy_api)
            self.Context.beta_name = sample_beta
            self.Context.branch_num = branch_n
            self.Context.tag_num = tag_n
            patch_name = "{}-{}".format(self.Context.branch_num, self.Context.tag_num)
        logging.debug("Patch name is: %s", patch_name)
        return patch_name

    @staticmethod
    def _set_patch_option(patch_data, patch_option_name, patch_option):
        if patch_option:
            logging.debug("Set patch option %s=%s", patch_option_name, patch_option)
            patch_data[patch_option_name] = patch_option

    def _process_yappy_patch(self, patch):
        yappy_patch = {}

        # Info about service parent
        self._set_patch_option(yappy_patch, "parentExternalId", patch.parent_service)
        self._set_patch_option(yappy_patch, "ignoreParentInstanceSpec", patch.ignore_instance_spec)
        self._set_patch_option(yappy_patch, "callisto", patch.callisto_options)
        self._set_patch_option(yappy_patch, "copyCoredumpPolicy", patch.copy_coredump_policy)

        res_data = []
        # Fill info about patch resources & environment variables
        for res in patch.resources:
            # get all class attributes to insert additional data to yappy patch
            res_params = {"localPath": res.local_path, "manageType": res.manage_type}
            if isinstance(res, yappy_cfg.YappyParametrizedResource):
                # Add resource id from specified sb param
                resource_id = self.Parameters.component_resources.get(res.param_name, None)
                if resource_id is not None:
                    try:
                        resource = sdk2.Resource[resource_id]
                    except Exception as exc:
                        logging.exception("Couldn't get resource with id %s, exception %s", resource_id, exc)
                        continue
                    res_params.update({"manageType": "SANDBOX_RESOURCE", "sandboxResourceId": str(resource.id)})
                    if res.checkconfig_name:
                        yappy_patch.setdefault("expectedCheckconfig", []).append(
                            "{}={}".format(res.checkconfig_name, resource.md5)
                        )
                else:
                    logger.warning("Empty input parameter: %s", res.param_name)
                    continue
                if res.storage:
                    res_params["storage"] = res.storage

            elif isinstance(res, yappy_cfg.YappyLastReleasedResource):
                resource_id, _ = resource_selectors.by_last_released_task(res.res_type)
                res_params.update({"manageType": "SANDBOX_RESOURCE", "sandboxResourceId": str(resource_id)})
            elif isinstance(res, yappy_cfg.YappyStaticResource):
                if res.manage_type == "STATIC_CONTENT" and res.content:
                    res_params["content"] = res.content
                if res.resource_id:
                    res_params["sandboxResourceId"] = res.resource_id

            res_data.append(res_params)
        if res_data:
            yappy_patch["resources"] = res_data

        if patch.instance_spec:
            instance_spec = {}
            for spec in patch.instance_spec:
                if isinstance(spec, yappy_cfg.YappyEnvVar):
                    if spec.container and spec.name and spec.value:
                        if 'containers' not in instance_spec:
                            instance_spec['containers'] = []
                        containers = instance_spec['containers']
                        exist_cont = None
                        for cont in containers:
                            if spec.container == cont.get('name', ''):
                                exist_cont = cont
                                break
                        if not exist_cont:
                            exist_cont = {'name': spec.container}
                            containers.append(exist_cont)
                        env_list = exist_cont.get('env')
                        if env_list is None:
                            env_list = []
                            exist_cont['env'] = env_list
                        exist_env = None
                        for env in env_list:
                            if spec.name == env.get('name', ''):
                                exist_env = env
                                break
                        if not exist_env:
                            exist_env = {'name': spec.name}
                            env_list.append(exist_env)
                        exist_env['valueFrom'] = {
                            'type': 'LITERAL_ENV',
                            'literalEnv': {
                                'value': spec.value,
                            }
                        }
            if instance_spec:
                yappy_patch["instanceSpec"] = instance_spec

        return yappy_patch

    def get_branch_tag(self):
        try:
            _, branch_n, tag_n = self.Context.beta_name.rsplit("-", 2)
            return int(branch_n), int(tag_n)
        except Exception as exc:
            eh.log_exception("Failed to split beta name. Probably it doesn't have 'name-branch-tag' structure", exc)

    def stop_prev_betas(self, c_info, yappy_api):
        logger.info("Stopping previous betas")
        if self.Parameters.get_beta_names_from_state:
            beta_names_getter = self.prev_betas_from_state
        else:
            beta_names_getter = self.existing_branch_betas
        try:
            branch_tag = self.get_branch_tag()
            if branch_tag is None:
                logging.info("Unable to determine branch and tag for prev betas. Nothing to stop")
                return
            branch_n, tag_n = branch_tag
            if branch_n < c_info.last_branch_num:
                warning_string = (
                    "New beta generation launched on an old branch {} (the latest branch is {}). "
                    "There might be some newer betas started which we could not stop. "
                    "Please, stop them manually if you run into some quota issues."
                ).format(branch_n, c_info.last_branch_num)
                logger.warning(warning_string)
                self.set_info(warning_string)
            curr_branch_prev_betas = list(beta_names_getter(c_info, yappy_api, branch_n, tag_n))
            logger.info("Existing previous betas for current branch: %s", curr_branch_prev_betas)
            prev_branch_prev_betas = list(beta_names_getter(c_info, yappy_api, branch_n - 1))
            logger.info("Existing previous betas for previous branch: %s", prev_branch_prev_betas)

            work_betas_num = c_info.yappy_cfg.get_working_betas_limit(self.Parameters.beta_conf_type)
            for prev_beta_name in yappy_helper.get_betas_to_stop(curr_branch_prev_betas, work_betas_num):
                yappy_api.stop_beta(prev_beta_name)
            for prev_beta_name in yappy_helper.get_betas_to_stop(prev_branch_prev_betas, 2):
                yappy_api.stop_beta(prev_beta_name)
            stop_betas_gap = c_info.yappy_cfg.get_stop_betas_gap(self.Parameters.beta_conf_type)
            for i in range(stop_betas_gap, stop_betas_gap + 2):
                old_prev_betas = list(beta_names_getter(c_info, yappy_api, branch_n - i))
                logger.info("Existing previous betas for previous branch (stop all of them): %s", old_prev_betas)
                for prev_beta_name in old_prev_betas:
                    yappy_api.stop_beta(prev_beta_name)
        except Exception as e:
            eh.log_exception("Failed to stop prev betas", e)
        self.test_prev_betas_from_state(c_info, yappy_api)

    def test_prev_betas_from_state(self, c_info, yappy_api):
        """
        The purpose of this method is to test `prev_betas_from_state`.
        It will be removed after several runs.
        """
        try:
            branch_tag = self.get_branch_tag()
            if branch_tag is None:
                logging.info("Unable to determine branch and tag for prev betas. Nothing to stop")
                return
            branch_n, tag_n = branch_tag
            branches_back_count = 4
            for i in xrange(branches_back_count):
                target_branch_num = branch_n - i
                logger.debug(
                    "The following betas found for branch %s: %s",
                    target_branch_num,
                    list(self.prev_betas_from_state(c_info, yappy_api, target_branch_num)),
                )
        except Exception as e:
            eh.log_exception("Searching for betas in state FAILED", e)

    def existing_branch_betas(self, c_info, yappy_api, branch_n, tag_n=None):
        """
        Retrieves all existing beta names for branch `branch_n` from tag 1 to `tag_n - 1`
        if `tag_n` is not `None` or to the last tag in branch otherwise.

        Consider the following table:

                    |    tag 1      tag 2       tag 3       tag 4       tag 5
        ------------+------------------------------------------------------------
        stable-100  | beta-100-1  beta-100-2       -      beta-100-4  beta-100-5
        stable-101  | beta-101-1       -      beta-101-3

        For the given table of betas the following results can be obtained:

        * `self.existing_branch_betas(c_info, api, 100)`    => `'beta-100-1', 'beta-100-2', 'beta-100-4', 'beta-100-5'`
        * `self.existing_branch_betas(c_info, api, 100, 5)` => `'beta-100-1', 'beta-100-2', 'beta-100-4'`
        * `self.existing_branch_betas(c_info, api, 101, 2)` => `'beta-101-1'`
        * `self.existing_branch_betas(c_info, api, 101)`    => `'beta-101-1', 'beta-101-3'`
        """
        if branch_n <= 0:
            return
        last_tag_num = tag_n or (c_info.last_tag_num(branch_n) + 1)
        for tag_n in xrange(1, last_tag_num):
            beta_name = c_info.yappy_cfg.get_beta_name(
                self.Parameters.beta_conf_type, patch_name="{}-{}".format(branch_n, tag_n)
            )
            logger.debug("Checking beta '%s'", beta_name)
            if yappy_api.beta_exists(beta_name):
                yield beta_name
            else:
                logger.debug("Beta '%s' doesn't exist, continue checking", beta_name)

    def prev_betas_from_state(self, c_info, yappy_api, branch_n, tag_n=None):
        """
        Searches for previously started betas by inspecting the component's state.
        Yields each of the beta name it comes over.

        :param c_info: component info instance
        :param yappy_api: yappy client instance
        :param branch_n: branch number
        :param tag_n: tag number; if `None` then the last tag of the given branch is considered
        """
        logger.debug("Getting prev beta names from RM state")
        if branch_n <= 0:
            return
        if not self.job_name:
            return
        client = rm_client.RMClient()
        last_tag_num = tag_n or (c_info.last_tag_num(branch_n) + 1)
        for tag_n in xrange(1, last_tag_num):
            key = '/br{branch}/t{tag}/job/${job}'.format(  # Going to be improved in RMDEV-921
                branch=branch_n,
                tag=tag_n,
                job=self.job_name,
            )
            logger.debug("Get state with key %s", key)
            state_entries = client.get_state(c_info.name, key=key)
            if not state_entries:
                logger.debug("No entries")
                continue
            state_value = json.loads(state_entries[0]['value']).get('event_info', {})
            if RM_STATE_BETA_NAME_KEY not in state_value:
                logger.debug("%s not found in state value", RM_STATE_BETA_NAME_KEY)
                continue
            beta_name = state_value[RM_STATE_BETA_NAME_KEY]
            if yappy_api.beta_exists(beta_name):
                yield beta_name
            else:
                logger.debug("Beta '%s' doesn't exist, continue checking", beta_name)

    @staticmethod
    def get_local_yappy_path(beta_prefix):
        return os.path.abspath(beta_prefix)

    def _commit_changes(self, added_paths):
        ssh_key = rm_svn.get_ssh_key(self, rm_const.COMMON_TOKEN_OWNER, rm_const.ARC_ACCESS_TOKEN_NAME)
        msg = "Add new configuration for beta {beta_name}. SB Task: {sb_task} SKIP_CHECK".format(
            beta_name=self.Context.beta_name, sb_task=lb.task_link(self.id, plain=True),
        )
        with ssh_key:
            commit_info = svn.Arcadia.commit(added_paths, msg, user=rm_const.ROBOT_RELEASER_USER_NAME)
            logger.debug("Commit info: %s", commit_info)
            committed_rev_list = re.findall(r"Committed revision (\d+)", commit_info)
            self.Context.beta_conf_revision = int(committed_rev_list[0]) if committed_rev_list else None

    @staticmethod
    def _update_config_file(beta_data, upd_data):
        logging.debug("Beta data:\n%s", beta_data)
        logging.debug("Update data:\n%s", upd_data)
        new_cc = {k.split("=")[0]: k.split("=")[1] for k in upd_data.get("expectedCheckconfig", [])}
        old_check_config = beta_data.get("expectedCheckconfig", [])
        for check_config in old_check_config:
            cc_key = check_config.split("=")[0]
            if cc_key in new_cc:
                old_check_config[old_check_config.index(check_config)] = "{}={}".format(cc_key, new_cc[cc_key])
        beta_resources = beta_data.get("resources", [])
        for res in upd_data.get("resources", []):
            insert = False
            for beta_res in beta_resources:
                if res["localPath"] == beta_res["localPath"]:
                    beta_resources[beta_resources.index(beta_res)] = res
                    insert = True
            if not insert:
                beta_resources.append(res)
        logging.debug("Updated data:\n%s", beta_data)

    def _front_url(self, suffix):
        return "((http://{beta_name}.hamster.yandex.{suffix} {suffix}))".format(
            beta_name=self.Context.beta_name, suffix=suffix
        )

    def _get_branch_and_tag_from_params(self, c_info):
        release_item = c_info.releases_cfg__resources_info[0]
        comp_res_id = self.Parameters.component_resources.get(release_item.resource_name, None)
        binary_res = None
        if not comp_res_id:
            for comp_res_key, comp_res_id in self.Parameters.component_resources.items():
                try:
                    binary_res = sdk2.Resource[comp_res_id]
                except Exception as exc:
                    logging.exception(
                        "Couldn't get resource %s with id %s, exception %s", comp_res_key, comp_res_id, exc,
                    )
                    continue
                break
        else:
            try:
                binary_res = sdk2.Resource[comp_res_id]
            except Exception as exc:
                logging.exception(
                    "Couldn't get resource %s with id %s, exception %s", release_item.resource_name, comp_res_id, exc,
                )
        eh.verify(binary_res, "Can't find any resource with tag and branch in ComponentNameSubs")
        branch_n, tag_n = c_info.get_tag_info_from_build_task(binary_res.task_id, release_item.build_ctx_key)
        self.Context.branch_num = branch_n
        self.Context.tag_num = tag_n
        return branch_n, tag_n

    def _beta_link(self):
        return lb.yappy_beta_link(self.Context.beta_name)

    @property
    def footer(self):
        if self.Context.beta_name and not self.Parameters.debug_mode:
            return [{
                "helperName": "",
                "content": self._beta_link(),
            }]

    def on_break(self, prev_status, status):
        try:
            self.send_rm_proto_event(status)
        except Exception:
            logging.exception("Unable to send event")

        # Temporary fix for exception statuses
        self.Context.beta_name = None
        super(GenerateYappyBeta, self).on_break(prev_status, status)

    def on_timeout(self, prev_status):
        super(GenerateYappyBeta, self).on_timeout(prev_status)
        if self.Parameters.debug_mode:
            return

        self.on_failure_st_comment(
            rmc.get_component(self.Parameters.component_name),
            sh.STHelper(rm_sec.get_rm_token(self)),
            status='TIMED OUT',
            force_fail=False,
        )

    def _get_rm_proto_event_specific_data(self, rm_proto_events, event_time_utc_iso, status=None):

        if self.Parameters.debug_mode:
            return

        svn_info = rm_const.GSID_SVN_RE.search(self.Context.__GSID)

        try:
            beta_name = self.beta_name
        except Exception:
            logging.exception("Unable to get beta name")
            beta_name = ""

        if not self.Context.start_event_created:

            self.Context.start_event_created = True
            self.Context.save()

            return {
                "new_beta_generation_started_data": rm_proto_events.NewBetaGenerationStartedData(
                    beta_name=beta_name,
                    beta_mode=self.Parameters.update_beta_mode,
                    scope_number=str(self.Parameters.release_number),
                    revision=svn_info.group("svn_revision") if svn_info else "",
                    job_name=self.job_name,
                )
            }

        return {
            "new_beta_generation_data": rm_proto_events.NewBetaGenerationData(
                beta_name=beta_name,
                beta_mode=self.Parameters.update_beta_mode,
                scope_number=str(self.Parameters.release_number),
                revision=svn_info.group("svn_revision") if svn_info else "",
                job_name=self.job_name,
            ),
        }
