# -*- coding: utf-8 -*-
import six
import json
import logging
import re
from collections import defaultdict

from concurrent.futures import as_completed, ThreadPoolExecutor

import sandbox.projects.release_machine.components.all as rmc
import sandbox.projects.release_machine.input_params2 as rm_params
import sandbox.projects.release_machine.tasks.base_task as rm_bt
import sandbox.projects.release_machine.helpers.svn_helper as rm_svn
import sandbox.projects.release_machine.core.task_env as task_env
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.common import binary_task
from sandbox.projects.common.testenv_client import TEClient
from sandbox.projects.release_machine.tasks.GetLastGoodRevision.problems import (
    CheckTestIsNotOK,
    UncheckedInterval,
    UnresolvedProblem,
    IgnoredInterval,
)
from sandbox.projects.release_machine.tasks.GetLastGoodRevision.report import (
    create_revisions_html_report,
    create_problems_html_report,
)

logger = logging.getLogger(__name__)


class NoGoodRevisionFound(Exception):
    pass


def parse_human_readable(td_str):
    pattern = re.compile(r'(?P<hours>\d+):(?P<minutes>\d+):(?P<seconds>\d+)')
    m = pattern.match(td_str)
    if m is None:
        raise TypeError("String '{}' has insupported format".format(td_str))
    m_dict = m.groupdict()
    return int(m_dict['hours']) * 60 * 60 + int(m_dict['minutes']) * 60 + int(m_dict['seconds'])


class TimeDelta(sdk2.parameters.String):
    @classmethod
    def cast(cls, value):
        if isinstance(value, six.string_types):
            try:
                return int(value)
            except Exception:
                return parse_human_readable(value)
        elif isinstance(value, (int, float)):
            return value
        raise TypeError("Unable to parse {} {} as timedelta".format(type(value), value))


class GetLastGoodRevision(rm_bt.BaseReleaseMachineTask):
    """**Release-machine** Get last green revision from db
    """

    __component_info = None

    class Requirements(task_env.TinyRequirements):
        ram = 4096  # 4GiB or less
        disk_space = 2048  # 2 Gb

    class Parameters(rm_params.ComponentName2):
        _lbrp = binary_task.binary_release_parameters(stable=True)

        te_db_name = sdk2.parameters.String("Testenv DB name, overrides component's own testenv_cfg__trunk_db")

        with sdk2.parameters.Group("Revision selection settings") as revision_params:
            start_revision = sdk2.parameters.Integer("Start revision")
            with start_revision.value[0]:
                relative_to_stable_release = sdk2.parameters.Bool(
                    "Automatically detect start revision relative to last stable release",
                    default=True,
                )
            ignore_marked_revisions = sdk2.parameters.Bool(
                "Ignore revisions marked in TestEnv database as bad",
                description="TESTENV-2744",
                default=False,
            )

        with sdk2.parameters.Group("Check tests settings") as check_params:
            jobs_list = sdk2.parameters.List(
                "Jobs list",
                description="Following tests will be checked during revision search. "
                            "If empty, all tests in database will be checked"
            )
            exclude_check_tests = sdk2.parameters.List(
                "Exclude tests",
                description="Check failures in following tests won't affect green revision search"
            )
            hide_not_checked = sdk2.parameters.Bool("Look up for only explicitly checked revisions", default=True)
            hide_filtered = sdk2.parameters.Bool("Exclude filtered revisions", default=True)

        with sdk2.parameters.Group("Diff tests settings") as diff_params:
            require_resolved_diffs = sdk2.parameters.Bool("Require resloved diffs", default=False)
            allow_running_binary_searches = sdk2.parameters.Bool(
                "Allow running binary searches",
                default_value=True,
            )
            include_diff_tests = sdk2.parameters.List(
                "Include tests",
                description="Unresolved diffs in following tests will be checked during revision search. "
                            "If empty, all tests in database will be checked"
            )
            exclude_diff_tests = sdk2.parameters.List(
                "Exclude tests",
                description="Unresolved diffs in following tests won't affect green revision search"
            )

        with sdk2.parameters.Group("Report settings") as report_settings:
            revisions_limit = sdk2.parameters.Integer(
                "Limit report revisions",
                description="Best revisions will be reported to task info",
                default=10,
            )

        with sdk2.parameters.Group("Misc") as misc:
            allow_async = sdk2.parameters.Bool("Allow parallel requests execution", default=True)
            raise_exception = sdk2.parameters.Bool(
                "Raise EXCEPTION instead of FAILURE in case of empty good revisions list",
                description="Useful for TE retry policy",
                default=False,
            )

        with sdk2.parameters.Output:
            good_revision = sdk2.parameters.Integer("Good revision")

    @staticmethod
    def get_unresolved_problems(database_name, diff_tests):
        unresolved_problems = TEClient.get_te_problems(database_name, unresolved_only=True)["rows"]
        logger.debug("Unresolved problems: %s", unresolved_problems)
        unresolved_diffs = defaultdict(list)
        for problem in unresolved_problems:
            if problem["test_name"] in diff_tests:
                unresolved_diffs[problem["revision"]].append(problem)

        logger.debug("Unresolved problems by revision: %s", unresolved_diffs)
        return unresolved_diffs

    @staticmethod
    def get_unchecked_intervals(database_name, diff_tests, start_revision, allow_async=True):
        unchecked_intervals = []

        if allow_async:
            with ThreadPoolExecutor(max_workers=16) as executor:
                results = {
                    executor.submit(
                        TEClient.get_unchecked_intervals,
                        database_name, test_name, start_revision,
                    ): test_name
                    for test_name in diff_tests
                }
            for future in as_completed(results):
                test_name = results[future]
                unchecked_intervals += list(future.result())

        else:
            for test_name in diff_tests:
                unchecked_intervals += list(TEClient.get_unchecked_intervals(database_name, test_name, start_revision))

        logger.debug("Unchecked intervals: %s", unchecked_intervals)
        return unchecked_intervals

    @staticmethod
    def get_ignored_by_markers_intervals(database_name, start_revision, component_name):
        ignored_intervals = []
        marked_intervals = TEClient.get_commit_markers(database_name, start_revision)
        for interval in marked_intervals:
            if interval["component"] == component_name:
                ignored_intervals.append(interval)
        return ignored_intervals

    @staticmethod
    def get_ok_check_tests(
        database_name, check_tests, start_revision, hide_not_checked, hide_filtered, allow_async=True
    ):
        tests_by_revisions = defaultdict(list)
        if allow_async:
            with ThreadPoolExecutor(max_workers=16) as executor:
                results = {
                    executor.submit(
                        TEClient.get_ok_test_revisions,
                        database_name, test_name, start_revision,
                        hide_not_checked=hide_not_checked,
                        hide_filtered=hide_filtered,
                    ): test_name
                    for test_name in check_tests
                }
            for future in as_completed(results):
                revisions = future.result()
                test_name = results[future]
                for rev in revisions:
                    tests_by_revisions[rev].append(test_name)
                logger.debug("%d candidates for %s: %s", len(revisions), test_name, revisions)

        else:
            for test_name in check_tests:
                revisions = TEClient.get_ok_test_revisions(
                    database_name, test_name, start_revision,
                    hide_not_checked=hide_not_checked, hide_filtered=hide_filtered
                )
                for rev in revisions:
                    tests_by_revisions[rev].append(test_name)
                logger.debug("%d candidates for %s: %s", len(revisions), test_name, revisions)

        return tests_by_revisions

    def get_custom_problems(self, revision_candidates):
        """Override this method to use additional criteria to filter revisions
        :rtype: dict
        :returns: {revision: [Problem1, Problem2, ...], ...}
        """
        return {}

    @property
    def component_info(self):
        if self.__component_info is None:
            self.__component_info = rmc.get_component(self.Parameters.component_name)
        return self.__component_info

    @property
    def start_revision(self):
        if self.Parameters.start_revision:
            logger.info("Using explicitly provided start_revision")
            return self.Parameters.start_revision

        if self.Parameters.relative_to_stable_release:
            logger.info("Automatically detecting start_revision relative to last stable release")
            return self.component_info.first_rev

        logger.info("Automatically detecting start_revision relative to last created release scope")
        last_scope_num = self.component_info.last_scope_num
        release_path = self.component_info.full_scope_path(last_scope_num)
        trunk_rev_of_branching = rm_svn.SvnHelper.get_last_trunk_revision_before_copy(
            release_path,
            repo_base_url=self.component_info.svn_cfg__repo_base_url,
            trunk_url=self.component_info.svn_cfg__trunk_url,
        )
        logger.info('Detected last branch trunk revision %s by last scope %s', trunk_rev_of_branching, last_scope_num)
        return trunk_rev_of_branching + 1

    def report_problems(self, revision_problems):
        limit = min(len(revision_problems), self.Parameters.revisions_limit)
        top_revs = sorted(revision_problems.items(), key=lambda item: (len(item[1]), -int(item[0])))[:limit]

        last_rev_problems = max(revision_problems.items(), key=lambda x: x[0])[1]

        logger.info("Revisions info: %s", json.dumps(
            dict([
                (revision, map(six.text_type, problems))
                for revision, problems in revision_problems.items()
            ]),
            indent=2))

        self.set_info(
            create_revisions_html_report(top_revs, limit),
            do_escape=False
        )

        self.set_info(
            create_problems_html_report(last_rev_problems),
            do_escape=False
        )

    def get_revision_problems(self):
        database_name = self.Parameters.te_db_name or self.component_info.testenv_cfg__trunk_db

        database_job_names = [job["name"] for job in TEClient.get_jobs(database_name)]
        invalid_job_names = []
        if not self.Parameters.jobs_list:
            check_tests = [
                job_name for job_name in database_job_names
                if job_name not in self.Parameters.exclude_check_tests
            ]
        else:
            check_tests = list(set(self.Parameters.jobs_list) - set(self.Parameters.exclude_check_tests))
        invalid_job_names.extend(list(set(check_tests) - set(database_job_names)))

        if not self.Parameters.include_diff_tests:
            diff_tests = [
                job_name for job_name in database_job_names
                if job_name not in self.Parameters.exclude_diff_tests
            ]
        else:
            diff_tests = list(set(self.Parameters.include_diff_tests) - set(self.Parameters.exclude_diff_tests))
        invalid_job_names.extend(list(set(diff_tests) - set(database_job_names)))

        if invalid_job_names:
            error_message = "There is no such jobs as {} in TE database {}".format(invalid_job_names, database_name)
            logger.error(error_message)
            raise TaskFailure(error_message)

        ok_tests_by_revisions = self.get_ok_check_tests(
            database_name,
            check_tests,
            self.start_revision,
            self.Parameters.hide_not_checked,
            self.Parameters.hide_filtered,
            allow_async=self.Parameters.allow_async,
        )
        unresolved_diffs = (
            self.get_unresolved_problems(database_name, diff_tests)
            if self.Parameters.require_resolved_diffs else {}
        )
        unchecked_intervals = self.get_unchecked_intervals(
            database_name,
            diff_tests,
            self.start_revision,
            allow_async=self.Parameters.allow_async,
        ) if self.Parameters.require_resolved_diffs or not self.Parameters.allow_running_binary_searches else []
        ignored_intervals = self.get_ignored_by_markers_intervals(
            database_name,
            self.start_revision,
            self.Parameters.component_name,
        ) if self.Parameters.ignore_marked_revisions else []

        revision_problems = {
            r: [CheckTestIsNotOK(r, test_name) for test_name in list(set(check_tests) - set(green_tests))]
            for r, green_tests in ok_tests_by_revisions.items()
        }

        if unresolved_diffs:
            logger.info("Unresolved problems: %s", unresolved_diffs)
            logger.info("First unresolved diff at r%d", min(unresolved_diffs.keys()))
            for r in revision_problems:
                for unresolved_rev, problems in unresolved_diffs.items():
                    if r >= unresolved_rev:
                        revision_problems[r].extend(
                            [
                                UnresolvedProblem(
                                    problem["test_diff/revision1"],
                                    problem["test_diff/revision2"],
                                    owner=problem["owner"],
                                    test_name=problem["test_name"],
                                    database=database_name,
                                ) for problem in problems
                            ]
                        )

        if unchecked_intervals:
            logger.info("Running binary searches at the intervals: %s", unchecked_intervals)
            logger.info(
                "First unchecked interval: %s",
                min(unchecked_intervals, key=lambda interval: interval["first_revision"]),
            )
            for r in revision_problems:
                for unchecked_interval in unchecked_intervals:
                    if r > unchecked_interval["first_revision"]:
                        revision_problems[r].append(
                            UncheckedInterval(
                                unchecked_interval["first_revision"],
                                unchecked_interval["last_revision"],
                                test_name=unchecked_interval["test_name"],
                            )
                        )

        if ignored_intervals:
            logger.info("There are some ignored intervals: %s", ignored_intervals)
            for r in revision_problems:
                for ignored_interval in ignored_intervals:
                    if ignored_interval["revision1"] <= r < ignored_interval["revision2"]:
                        revision_problems[r].append(
                            IgnoredInterval(
                                ignored_interval["revision1"],
                                ignored_interval["revision2"],
                                comment=ignored_interval["comment"],
                            )
                        )

        custom_revision_problems = self.get_custom_problems(revision_problems.keys())
        for rev, problems in custom_revision_problems.items():
            revision_problems.setdefault(rev, []).extend(problems)

        return revision_problems

    def get_last_good_revision(self, revision_problems):
        suitable_revisions = [rev for rev, problems in revision_problems.items() if len(problems) == 0]
        revision = max(suitable_revisions) if suitable_revisions else None

        return revision

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)

        revision_problems = self.get_revision_problems()
        revision = self.get_last_good_revision(revision_problems)

        try:
            self.report_problems(revision_problems)
        except Exception as exc:
            logger.error(exc)

        if revision is None:
            if self.Parameters.raise_exception:
                raise NoGoodRevisionFound("No good revision found, see logs for detailed report")
            else:
                raise TaskFailure("No suitable revision found")

        self.Parameters.good_revision = revision
        self.Context.good_revision = revision
