import six
import textwrap
from collections import defaultdict

from jinja2 import Environment

from sandbox.projects.release_machine.tasks.GetLastGoodRevision.problems import (
    CheckTestIsNotOK,
)

STYLE = '''
    .table {
        white-space: normal;
    }
    .table th {
        padding: 0px 8px;
        text-align: left;
        vertical-align: top;
        background-color: #f2f2f2;
    }
    .table tr {
        padding: 0px 8px;
        text-align: left;
        vertical-align: top;
    }
    .table tr:nth-child(even) {
        background-color: #f2f2f2;
    }
'''

REPORT_TEMPLATE = textwrap.dedent('''
<style>
{{ style }}
</style>
<div class="table">
<h3>Top {{ limit }} revisions</h3>
<table>
<theader>
{% for col in header %}
    <th>{{ col }}</th>
{% endfor %}
</theader>
<tbody>
{% for row in rows %}
    <tr>
        <td>
            <span style="color: {% if row.problems %}red{% else %}green{% endif %};">
            {{ row.Revision }}
            </span>
        </td>
        {% for col in header[1:] %}
        <td>
            {% if row.get(col) %}
            {{ row.get(col, [])|join('<br/>') }}
            {% else %}
            <span style="color: green;">Ok</span>
            {% endif %}
        </td>
        {% endfor %}
    </tr>
{% endfor %}
</tbody>
</table>
</div>
''')


PROBLEMS_REPORT_TEMPLATE = '''
<style>
{{ style }}
</style>
<div class="table">
<h3>Problems with the most recent revision</h3>
<table>
    <theader>
        <th>Problem type</th>
        <th>Description</th>
    </theader>
<tbody>
{% for problem_type, problem_descriptions in problems.items() %}
    <tr>
        <td>
            {{ problem_type }}
        </td>
        <td>
            {{ problem_descriptions|join('<br/>') }}
        </td>
    </tr>
{% endfor %}
</tbody>
</table>
</div>
'''


def create_revisions_html_report(revisions, limit):
    env = Environment()
    template = env.from_string(REPORT_TEMPLATE)

    rows = []
    header = ["Revision"]
    for revision, problems in sorted(revisions, key=lambda r_p: -int(r_p[0])):
        row = {
            "Revision": revision,
            "problems": len(problems)
        }
        row_problems = defaultdict(list)
        for problem in problems:
            if problem.__class__.__name__ not in header:
                header.append(problem.__class__.__name__)
            row_problems[problem.__class__.__name__].append(problem.snippet)

        row.update(row_problems)
        rows.append(row)

    return template.render(header=header, rows=rows, limit=limit, style=STYLE)


def create_problems_html_report(problems):
    env = Environment()
    template = env.from_string(PROBLEMS_REPORT_TEMPLATE)
    formatted_problems = defaultdict(list)

    for problem in problems:
        if isinstance(problem, CheckTestIsNotOK):
            formatted_problems["CheckTestIsNotOK"].append(problem.test_name)
        else:
            formatted_problems[problem.__class__.__name__].append(six.text_type(problem))

    return template.render(problems=formatted_problems, style=STYLE)
