# -*- coding: utf-8 -*-
import logging
import re

import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.input_params2 as rm_params
import sandbox.projects.release_machine.security as rm_sec
import sandbox.projects.release_machine.tasks.base_task as rm_bt
import sandbox.sdk2 as sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.release_machine.helpers import staff_helper
from sandbox.projects.common.telegram_api import notify
from sandbox.projects.messenger.bot import QBot
from sandbox.projects.release_machine.client import RMClient
from sandbox.projects.common.decorators import memoized_property as cached_property


CHAT_ID_REQUEST = "/get_chat_id @release_machine_bot"
SUPERGROUP_UPDATE = "Bad Request: group chat was upgraded to a supergroup chat"
TELEGRAM = "telegram"
Q_MESSENGER = "q_messenger"
CHAT_NAME_ARG = "chat_name="
CHAT_NAME_REGEX = re.compile("([a-z_]+)$")
COMP_NAME_ATTR_REGEX = re.compile("component_name=([a-z_]+)")


class GetTelegramAndQChatId(rm_bt.BaseReleaseMachineTask):
    """
    Get telegram and q chat ids by parsing chat messages.

    Docs: https://wiki.yandex-team.ru/releasemachine/notifications/#nastrojjkauvedomlenijjvtelegram
    """

    class Requirements(task_env.TinyRequirements):
        disk_space = 2 * 1024  # 2 Gb

    class Parameters(rm_params.ComponentName2):
        kill_timeout = 6 * 60 * 60  # 6 hours
        get_q_messenger = sdk2.parameters.Bool("Get also chat_id for q messenger", default=False)
        _lbrp = binary_task.binary_release_parameters(stable=True)

    class Context(rm_bt.BaseReleaseMachineTask.Context):
        chats_found = []

    @cached_property
    def rm_client(self):
        return RMClient()

    @staticmethod
    def _get_staff_login(telegram_login):
        staff_api = staff_helper.StaffApi(rm_sec.get_rm_token(None))
        staff_login = staff_api.get_person_by_telegram_login(telegram_login)
        return staff_login

    @staticmethod
    def _send_message_to_chat(bot, chat_id, message):
        response = bot.send_message(
            chat_id=chat_id,
            text=message,
        )
        if "error_code" in response:
            logging.error(
                "Got error while sending message to chat_id `%s`, exception: %s",
                chat_id,
                str(response),
            )

    def _send_chat_id_to_rm(self, chat_name, messenger_type, chat_id, comp_name):
        response = self.rm_client.add_component_chat(comp_name, chat_name, chat_id, messenger_type)
        self.Context.chats_found.append((chat_name, chat_id))
        self.Context.save()
        return response

    @staticmethod
    def _append_messenger_type(chat_name, messenger_type):
        if chat_name:
            chat_name += "::" + messenger_type
        return chat_name

    def _get_login_and_staff_login(self, message, chat_id, messenger_type):
        if messenger_type == Q_MESSENGER:
            # Need because of https://st.yandex-team.ru/MSSNGRBACKEND-1571#5e6793952990d1400ed37cfd
            login = message["from"].get("login", "") or message["from"].get("username", "")
            staff_login = login
        else:
            login = message["from"]["username"]
            logging.debug("Got message from private chat, telegram_login %s, chat_id %s", login, chat_id)
            staff_login = self._get_staff_login(login)
        return login, staff_login

    def _get_chat_name_from_bot_command(self, message_text):
        return message_text[message_text.index(CHAT_NAME_ARG):].strip().split(" ", 1)[0].split("=", 1)[1]

    def _add_chat_id_for_component(self, message, chat_id, messenger_type, comp_name, bot):
        chat_type = message["chat"]["type"]

        logging.debug("Chat type is %s", chat_type)

        if chat_type == "private":
            messenger_login, staff_login = self._get_login_and_staff_login(message, chat_id, messenger_type)

            if not staff_login:
                error_msg = "Can't get staff_login for tg_login `{login}`".format(login=messenger_login)
                logging.info(error_msg)
                self._send_message_to_chat(bot, chat_id, error_msg)
                return

            chat_name = self._append_messenger_type(staff_login, messenger_type)

            logging.info(
                "Got staff login `%s`, RM chat name `%s` messenger login `%s` and chat_id `%s`",
                staff_login,
                chat_name,
                messenger_login,
                chat_id,
            )

        else:

            if CHAT_NAME_ARG not in message["text"]:
                error_msg = (
                    "Chat name cannot be omitted in case of groups or supergroups. Please make sure to use the "
                    "following command format: "
                    "`/get_chat_id @release_machine_bot component_name=<component_name> chat_name=<chat_name>`"
                )
                self._send_message_to_chat(bot, chat_id, error_msg)
                return

            chat_name = self._get_chat_name_from_bot_command(message["text"])

            # fixme
            # if not CHAT_NAME_REGEX.fullmatch(chat_name):
            # ^^^ use this after python 2->3 migration and alter regex appropriately
            if not CHAT_NAME_REGEX.match(chat_name):
                error_msg = "Chat name {} does not match pattern {}".format(chat_name, CHAT_NAME_REGEX.pattern)

                logging.error(error_msg)

                self._send_message_to_chat(bot, chat_id, error_msg)

            chat_name = self._append_messenger_type(chat_name, messenger_type)

        logging.debug("Chat name is %s", chat_name)

        response = self._send_chat_id_to_rm(chat_name, messenger_type, chat_id, comp_name)

        if not response:
            error_msg = "unknown error"
        else:
            error_msg = response.get("error", "")

        if error_msg:
            logging.error("Unable to save chat")
            self._send_message_to_chat(
                bot,
                chat_id,
                "Cannot save your chat: {}".format(error_msg),
            )

    def _process_message_with_chat_id_request(self, message, bot, messenger_type):

        logging.debug("Processing message %s from %s ", message.get("text"), message.get("chat"))

        chat_id = message["chat"]["id"]

        if "component_name=" not in message["text"]:

            error_msg = (
                "Cannot parse command `{}`. The correct input format is: "
                "`/get_chat_id @release_machine_bot component_name=<your_component_name> [chat_name=<chat_name>]`"
            ).format(message["text"])
            self._send_message_to_chat(bot, chat_id, error_msg)

        else:

            component_name = COMP_NAME_ATTR_REGEX.findall(message["text"])[0]
            response = self.rm_client.get_component(component_name)

            if not response:

                logging.error(
                    "Got response status code %s for component_name %s",
                    self.rm_client.status_code, component_name,
                )
                error_msg = "Can't get component with name {component_name}".format(
                    component_name=component_name,
                )
                self._send_message_to_chat(bot, chat_id, error_msg)

            else:

                self._add_chat_id_for_component(message, chat_id, messenger_type, component_name, bot)

        response = bot.send_message(
            chat_id=chat_id,
            text="This chat_id is {chat_id}".format(chat_id=chat_id),
        )
        if "error_code" in response:
            logging.error(
                "Got error while sending message to chat_id `%s`, exception: %s",
                chat_id,
                str(response),
            )
            if response["description"] == SUPERGROUP_UPDATE:
                chat_id = response["parameters"]["migrate_to_chat_id"]
                bot.send_message(
                    chat_id=chat_id,
                    text="This chat_id is {chat_id}".format(chat_id=chat_id),
                )

    @staticmethod
    def _prepare_updates(bot, messenger_type, offset=None):

        updates = bot.get_updates(offset)
        if messenger_type == TELEGRAM:
            updates = updates["result"]
        return updates

    def on_execute(self):
        super(GetTelegramAndQChatId, self).on_execute()
        telegram_bot = notify.create_bot(self, rm_const.TELEGRAM_TOKEN_NAME, rm_const.COMMON_TOKEN_OWNER)
        if telegram_bot is None:
            logging.error("Unable to get telegram bot. Skip telegram notifications")
            return
        q_token = rm_sec.get_rm_token(self, rm_const.Q_MESSENGER_TOKEN_NAME)
        q_bot = QBot(q_token)

        bot_messenger_pairs = [(telegram_bot, TELEGRAM)]

        if self.Parameters.get_q_messenger:
            bot_messenger_pairs.append((q_bot, Q_MESSENGER))

        with self.memoize_stage.get_updates(30):
            for bot, messenger_type in bot_messenger_pairs:
                updates = self._prepare_updates(bot, messenger_type)
                while updates:
                    logging.debug("Got %d messages in updates", len(updates))
                    for msg in updates:
                        if (
                            "message" in msg and
                            "text" in msg["message"] and
                            "from" in msg["message"]
                        ):
                            message = msg["message"]
                            if message["text"].startswith(CHAT_ID_REQUEST):
                                self._process_message_with_chat_id_request(
                                    message, bot, messenger_type
                                )
                    updates = self._prepare_updates(bot, messenger_type, offset=updates[-1]["update_id"] + 1)

            if self.Context.chats_found:
                self.set_info(
                    "Chat ids and titles are: {}".format(self.Context.chats_found),
                )
            else:
                self.set_info(
                    "There are no messages like \'/get_chat_id @release_machine_bot\' in bot's updates. "
                )

            raise sdk2.WaitTime(10 * 60)  # 10 minutes
