# -*- coding: utf-8 -*-

import six

import logging
import datetime as dt
import json
import os
import random
import re
import requests
import time
from collections import defaultdict
if six.PY2:
    import urlparse
else:
    from urllib import parse as urlparse

import sandbox.common.errors as err
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.projects.release_machine.rm_notify as rm_notify
import sandbox.projects.release_machine.components.all as rmc
import sandbox.projects.release_machine.components.components_info as rm_comp
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.helpers.responsibility_helper as rm_responsibility
import sandbox.projects.release_machine.helpers.svn_helper as rm_svn
import sandbox.projects.release_machine.tasks.base_task as rm_bt

from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.projects.common import binary_task
from sandbox.projects.common import decorators
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import metrics_launch_manager as mlm
from sandbox.projects.common.metrics_launch_manager import mlm_analysis as mlma
from sandbox.projects.common import string
from sandbox.projects.common import time_utils
from sandbox.projects.common import utils2
from sandbox.projects.common import templates
from sandbox.projects.common import kosher_release
from sandbox.projects.common.betas.beta_api import BetaApi
from sandbox.projects.common.search import findurl
from sandbox.projects.common.search import bugbanner2 as bb2

from sandbox.projects.release_machine.helpers import wiki_helper
from sandbox.projects.release_machine import yappy as yappy_helper
from sandbox.projects.release_machine import resources as rm_resources
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.release_machine.helpers.metrics_helper import get_aggregated_launchset_detalization
from sandbox.projects.release_machine.tasks.LaunchMetrics import parameters as lm_params
from sandbox.projects.release_machine.tasks.LaunchMetrics import const as lm_const
from sandbox.projects.release_machine.tasks.LaunchMetrics import ignored_metrics
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.sandboxsdk import paths


@rm_notify.notify2()
class LaunchMetrics(rm_bt.BaseReleaseMachineTask):
    """
        **Release-machine**

        Задача для запуска тестов Metrics в sandbox (SEARCH-1389) с использованием MLM-шаблонов:
        - Запускает тесты Metrics для какой-либо компоненты Release Machine.
        - Сравнивает состояние бет до и после скачки. Пишет warning, если состояния отличаются.
        - Пишет в релизный тикет о результатах

        Maintainer/Responsible: ilyaturuntaev@.
    """

    # This flag determines whether or not should we stop MLM launch when task changes its status into one of the
    # break-statuses. We do NOT want to stop MLM launch when experiencing problems reaching MLM - RMDEV-2021.
    # In other cases (e.g., task stop forced by user) we DO stop the launch
    _stop_launch_on_break = True

    class Requirements(task_env.StartrekRequirements):
        disk_space = 2 * 1024  # 2 Gb

    class Parameters(lm_params.LaunchMetricsParameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)

    class Context(rm_bt.BaseReleaseMachineTask.Context):
        sample_beta_name = None
        svn_revision = 0
        job_name = ""
        beta_statuses = []
        launch_info = {}
        mlm_launch_info = {}
        final_status = mlma.LaunchFinalStatus.UNKNOWN
        adjusted_final_status = mlma.LaunchFinalStatus.UNKNOWN  # RMDEV-3170

    def form_launch_info_header(self):
        launch_info = self.Context.launch_info
        launch_statuses = []
        for launch in launch_info["launches"]:
            statuses = (launch["name"], launch["status"], launch["metrics_status"])
            launch_statuses.append(statuses)
        return {
            "helperName": "<br> Launch info </br>",
            "content": {
                u"<a href={}>{}</a>: {}".format(
                    mlm.get_mlm_link(launch_info["launch_id"]),
                    launch_info["name"],
                    launch_info["status"],
                ): {
                    "header": [
                        {"key": "region", "title": "Region"},
                        {"key": "launch_status", "title": "Launch status"},
                        {"key": "metrics_status", "title": "Metrics status"},
                    ],
                    "body": {
                        "region": [i[0] for i in launch_statuses],
                        "launch_status": [i[1] for i in launch_statuses],
                        "metrics_status": [i[2] for i in launch_statuses],
                    },
                },
            }
        }

    def form_beta_statuses_header(self):
        beta_statuses = self.Context.beta_statuses
        status_header = [{"key": "name", "title": "Beta name"}]
        status_header.extend(
            {
                "key": lm_const.CHECK_NUM_TEMPLATE.format(check_num),
                "title": str(betas[lm_const.DATETIME_KEY]) if lm_const.DATETIME_KEY in betas
                else lm_const.CHECK_NUM_TEMPLATE.format(check_num)
            } for check_num, betas in enumerate(beta_statuses, 1)
        )
        halfhour_delta = dt.timedelta(minutes=30)
        timestamp_created = int(time_utils.datetime_to_timestamp(self.created - halfhour_delta) * 1000)
        timestamp_updated = int(time_utils.datetime_to_timestamp(self.updated + halfhour_delta) * 1000)
        status_body = {"name": [
            lb.yappy_beta_link(beta_name) + lb.yappy_beta_status_chart_link(
                beta_name=beta_name,
                link_name=' (chart)',
                time_from=timestamp_created,
                time_to=timestamp_updated,
            ) for beta_name in beta_statuses[0] if beta_name != lm_const.DATETIME_KEY
        ]}
        for check_num, betas in enumerate(beta_statuses, 1):
            check_num_key = lm_const.CHECK_NUM_TEMPLATE.format(check_num)
            status_body[check_num_key] = []
            for beta_name in betas:
                if beta_name == lm_const.DATETIME_KEY:
                    continue
                if betas[beta_name]:
                    status_body[check_num_key].append("<span style='color:green'>CONSISTENT<span>")
                else:
                    status_body[check_num_key].append("<span style='color:red'>INCONSISTENT<span>")
        return {
            "helperName": "<br> Beta statuses </br>",
            "content": {
                "Beta Statuses": {
                    "header": status_header,
                    "body": status_body,
                },
            }
        }

    @sdk2.header()
    def header(self):
        headers = []
        if self.Context.launch_info:
            headers.append(self.form_launch_info_header())
        if self.Context.beta_statuses:
            headers.append(self.form_beta_statuses_header())
        return headers

    @property
    def is_running_in_release_machine_mode(self):
        return self.Parameters.metrics_mode_type == "release_machine"

    @property
    def checked_beta_host_without_tld(self):
        """E.g. 'report-web-1.hamster.yandex'."""
        return self._get_beta_host_without_tld(self.Parameters.checked_beta)

    @property
    def sample_beta_host_without_tld(self):
        """E.g. 'hamster.yandex'."""
        return self._get_beta_host_without_tld(self.Context.sample_beta_host)

    @decorators.memoized_property
    def token(self):
        return rm_sec.get_rm_token(self)

    @decorators.memoized_property
    def yappy_api_token(self):
        return sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.YAPPY_TOKEN_NAME)

    @decorators.memoized_property
    def yappy_api_client(self):
        return BetaApi.fromurl(token=self.yappy_api_token)

    @decorators.memoized_property
    def metrics_launcher(self):
        return mlm.MetricsLauncher(oauth_token=self.token)

    @decorators.memoized_property
    def c_info(self):

        if not self.is_running_in_release_machine_mode:
            return None

        logging.info("Building component info object (c_info) for %s", self.Parameters.component_name)
        c_info = rmc.get_component(self.Parameters.component_name)
        logging.info("Component name: %s", c_info)

        return c_info

    @decorators.memoized_property
    def template_dir(self):
        # type: ('LaunchMetrics') -> str
        return str(self.Parameters.template_dir.format(search_subtype=self.Parameters.search_subtype))

    @property
    def template_name(self):
        # type: ('LaunchMetrics') -> str
        return self.Parameters.custom_template_name

    @property
    def is_manual_run(self):
        """
        Is this a manual run

        If 'testenv_database' context field is not empty while 'testenv_auxilary_database' context field is empty
        then the task is launched by TestEnv automation
        """
        return not self.Context.testenv_database or self.Context.testenv_auxiliary_database is not ctm.NotExists

    @decorators.memoized_property
    def has_yappy_cfg(self):
        return hasattr(self.c_info, "yappy_cfg") and getattr(self.c_info.yappy_cfg, "betas", False)

    def _template_provided(self):
        """
        Check if custom_template_name input parameter is provided. If the parameter value is empty
        then consider it a special case of AB experiment launch that does not require actual tests launch (RMDEV-1214).
        In that case set appropriate task data and return `False`. Otherwise return `True`

        :return: `False` if custom_template_name is empty, `True` otherwise
        """

        custom_template_name_present = bool(self.Parameters.custom_template_name)
        resource_source_required = self.Parameters.template_source == lm_const.TEMPLATE_SOURCE_RESOURCE
        template_resource_present = bool(self.Parameters.template_resource)

        if custom_template_name_present and (not resource_source_required or template_resource_present):
            logging.info("Template source is OK. Proceeding")
            return True

        # RMDEV-1214
        # special case for ab experiments: do not run test if template name not specified
        # mark test as SUCCESS to allow other tests to run
        self.set_info(
            "Either template name or template source not specified! "
            "This means, than there is no reason to run test",
        )
        self.Context.launch_info = {
            lm_const.LAUNCH_INFO_STATUS_KEY: 'VALID',
            lm_const.LAUNCH_INFO_FINAL_STATUS_KEY: 'VALID',
        }
        self._save_report([("not_launched", "txt")])
        self.Context.adjusted_final_status = 'VALID'
        return False

    def _check_input_params(self):
        logging.info("Check sample and checked betas extra cgi params")
        if self.Parameters.sample_extra_params:
            try:
                urlparse.parse_qsl(str(self.Parameters.sample_extra_params).strip("&"), strict_parsing=True)
            except ValueError:
                eh.check_failed("Bad sample beta extra params.\n{}".format(eh.shifted_traceback()))
        if self.Parameters.checked_extra_params:
            try:
                urlparse.parse_qsl(str(self.Parameters.checked_extra_params).strip("&"), strict_parsing=True)
            except ValueError:
                eh.check_failed("Bad checked beta extra params.\n{}".format(eh.shifted_traceback()))

    def _get_beta_host_without_tld(self, beta_host):
        return beta_host if beta_host.endswith('.yandex') else "{}.yandex".format(beta_host)

    def _process_release_machine_sla_and_event_data(
        self,
        table_report,
        metrics_json_response,
    ):
        if not self.is_running_in_release_machine_mode:
            return

        if not self.Parameters.silent_mode:

            sla_response = self.metrics_launcher.get_launch_sla_info(self.Context.launch_info["launch_id"])
            sla_tickets = sla_response.get("slaIssues", [])
            sla_ticket_link = None

            if len(sla_tickets) > 0:
                sla_ticket_link = sla_tickets[0].get("link")

            title, message = self.c_info.launch_result_on_end(
                self, table_report, metrics_json_response, sla_ticket_link
            )

            self.send_st_messsage(title, message)
            self.send_rm_proto_event()

        self.yappy_beta_state(self.Context.sample_beta_name, self.Context.checked_beta_name, "after")

    def on_enqueue(self):
        resource = resource_types.METRICS_LAUNCH_DIFF(self, "metrics launch diff", "diff")
        self.Context.out_resource_id = resource.id

    def on_break(self, prev_status, status):
        super(LaunchMetrics, self).on_break(prev_status, status)

        self.send_rm_proto_event(status=status)

        logging.info("Trying to stop launch on break")
        if not self.Context.launch_info or status != ctt.Status.STOPPED or not self._stop_launch_on_break:
            logging.info("Next status is: %s", status)
            return

        self.metrics_launcher.stop_launch(self.Context.launch_info["launch_id"])

    def on_terminate(self):
        super(LaunchMetrics, self).on_terminate()

        self.send_rm_proto_event()

        logging.info("Trying to stop launch on terminate")
        if self.Context.launch_info and self.Context.launch_info.get('launch_id'):
            self.metrics_launcher.stop_launch(self.Context.launch_info["launch_id"])

    def on_execute(self):
        super(LaunchMetrics, self).on_execute()
        self.add_bugbanner(bb2.Banners.UPS)
        self.fill_context()

        self.store_rm_proto_event(status=self.status)

        if not self._template_provided():
            return

        self._check_input_params()

        self.set_beta_names()

        try:

            if not self.Context.launch_info:
                self.setup_and_start_betas()
                self.launch_template()

            mlm_launch_info = self._wait_launch()
            self.Context.mlm_launch_info = mlm_launch_info
            _, table_report, launch_fails = self.generate_reports(mlm_launch_info, self.Context.launch_info["status"])
            self.check_diff_2_serps_query_5(mlm_launch_info)

            self._process_release_machine_sla_and_event_data(table_report, mlm_launch_info)
            self._check_final_status(mlm_launch_info, launch_fails)

        except mlm.MLMError:

            self.set_info(
                "Stopping the task because MLM is not responding. Please make sure MLM is fine and re-run the task"
            )

            self._stop_launch_on_break = False

            raise err.TaskStop

        if table_report:
            setattr(self.Context, rm_const.ACCEPTANCE_FAIL_KEY, True)

    def setup_and_start_betas(self):
        if not self.is_running_in_release_machine_mode:
            return
        if self.has_yappy_cfg:
            self.start_yappy_betas([self.Context.sample_beta_name, self.Context.checked_beta_name])
        self.Context.checked_beta_version = (
            self._get_beta_version() or "Acceptance"
        )
        self.Context.sample_tag = self.Context.sample_beta_name.replace("{}-".format(self.c_info.name), "")

    def launch_template(self):
        self.set_info("Launch started")
        self.Context.launch_start = int(time.time())
        self.yappy_beta_state(self.Context.sample_beta_name, self.Context.checked_beta_name, "before")
        launch_template = self.get_launch_template(c_info=self.c_info)
        mlm_launch_info = self.metrics_launcher.launch_template(launch_template)
        logging.debug("MLM response is %s", mlm_launch_info)
        self._update_launch_info(mlm_launch_info)
        if self.c_info is not None and not self.Parameters.silent_mode:
            title, message = self.c_info.launch_result_on_start(self)
            self.send_st_messsage(title, message)
            self.send_rm_proto_event()
        logging.info("Check launch info:\n%s", self.Context.launch_info)

    def fill_context(self):
        try:

            self.Context.job_name = self.job_name

            if not self.is_ci_launch:
                svn_info, te_info, _ = self.Context.__GSID.split(" ", 2)
                self.Context.svn_revision = svn_info.split(":", 1)[1]  # SVN:<svn_revision>
            else:
                self.Context.svn_revision = str(self.ci_context.target_revision.number)

        except (ValueError, IndexError):
            # Sandbox launch, cannot get SVN launch revision
            logging.warning("Cannot get info from __GSID.")

    def _get_beta_version(self):

        major_release_number = self.Parameters.release_number
        minor_release_number = self.Parameters.minor_release_number

        if not major_release_number:
            return None

        if self.c_info.is_ci:
            return "{}-{}".format(major_release_number, minor_release_number)

        if not isinstance(self.c_info, rm_comp.Branched):
            return major_release_number

        # Process branched component and get tag of launched task
        try:
            full_branch_path = self.c_info.full_branch_path(major_release_number)
        except Exception as ex:
            eh.log_exception("Failed to get branch path", ex)
            return None
        revs = list(map(int, rm_svn.SvnHelper.revisions_merged(full_branch_path)))
        logging.debug("Got merged revisions: %s", str(revs))
        if not self.Context.svn_revision:
            return "b" + str(major_release_number) if major_release_number else None
        rev = int(self.Context.svn_revision)
        tag_id = revs.index(rev) + 1 if rev in revs else None
        return "{}-{}".format(major_release_number, tag_id)

    def start_yappy_betas(self, beta_names):
        logging.debug("beta_statuses are %s", self.Context.beta_statuses)
        beta_statuses = {}
        current_datetime = dt.datetime.now().strftime("%b %d %Y %H:%M:%S")
        beta_statuses[lm_const.DATETIME_KEY] = current_datetime

        for beta_name in beta_names:

            if not self.yappy_api_client.beta_exists(beta_name):
                msg = "Beta '{}' doesn't exist".format(beta_name)
                self.set_info(msg)
                raise err.TaskStop(msg)

            beta_statuses[beta_name] = self.yappy_api_client.get_beta_state(beta_name).get("status", "STOPPED")

            if beta_statuses[beta_name] not in ["CONSISTENT", "INCONSISTENT"]:
                self.yappy_api_client.start_beta(beta_name)

            beta_statuses[beta_name] = True if beta_statuses[beta_name] == "CONSISTENT" else False

        self.Context.beta_statuses.append(beta_statuses)
        logging.debug("Beta statuses are %s", self.Context.beta_statuses)

        if any(
            not beta_statuses[beta_name] for beta_name in beta_statuses
            if beta_name != lm_const.DATETIME_KEY
        ) and self.Parameters.wait_consistency:
            wait_status, msg = yappy_helper.wait_consistency(self, self.yappy_api_client, beta_names, "beta_statuses")
            if not wait_status:
                self.set_info(msg, do_escape=False)
                eh.check_failed('Waiting betas consistency timed out')

    def yappy_beta_state(self, sample_beta_name, checked_beta_name, state):
        if self.has_yappy_cfg:
            if not self.yappy_api_client.yappy.beta_exists(sample_beta_name):
                self._save_beta_state(sample_beta_name, checked_beta_name, state)
                if state == "after":
                    self._check_beta_state(sample_beta_name)

    def _check_beta_state(self, beta_name):
        info_diff_res_id = yappy_helper.cmp_beta_infos(
            getattr(self.Context, "descr_before")[beta_name]["configuration"]["sourceConfigs"],
            getattr(self.Context, "descr_after")[beta_name]["configuration"]["sourceConfigs"]
        )
        if info_diff_res_id:
            self.set_info(
                "Beta '{}' state has changed during this launch. Results are not relevant!\n"
                "Diff is in resource: {}".format(beta_name, lb.resource_link(info_diff_res_id)),
                do_escape=False,
            )
        else:
            self.set_info("Beta '{}' state has not changed during this launch!".format(beta_name))

    def _save_beta_state(self, sample_beta_name, checked_beta_name, time_when):
        setattr(self.Context, "descr_{}".format(time_when), {
            sample_beta_name: self.yappy_api_client.get_beta_info(sample_beta_name),
            checked_beta_name: self.yappy_api_client.get_beta_info(checked_beta_name),
        })
        logging.debug(
            "Sample beta info %s launch:\n%s",
            time_when, json.dumps(getattr(self.Context, "descr_{}".format(time_when))[sample_beta_name], indent=2)
        )
        logging.debug(
            "Checked beta info %s launch:\n%s",
            time_when, json.dumps(getattr(self.Context, "descr_{}".format(time_when))[checked_beta_name], indent=2)
        )

    def set_beta_names(self):
        """
        Set sample beta name as the last accessable beta before the last stable release for RM component
        """

        checked_beta = self.Parameters.checked_beta
        self.Context.checked_beta_name = checked_beta.split('.')[0]

        if self.Parameters.sample_beta:

            self.Context.sample_beta_host = self.Parameters.sample_beta
            self.Context.sample_beta_name = self.Context.sample_beta_host.split('.')[0]

        elif self.has_yappy_cfg:

            logging.info("Sample beta host is not set. Retrieving from c_info")

            sample_beta, _, _ = yappy_helper.get_sample_beta(self.c_info, self, self.yappy_api_client)
            sample_beta_host = self.yappy_api_client.yappy.get_beta_domain_name_without_tld(sample_beta)
            sample_beta_host = sample_beta_host.replace(".yandex", "")

            self.Context.sample_beta_name = sample_beta
            self.Context.sample_beta_host = sample_beta_host

            logging.info("Default sample beta host found: '%s'", sample_beta_host)

        else:

            self.set_info(
                "Sample beta not provided neither via input parameters, nor via {}'s config. "
                "No sample beta found".format(self.c_info.name)
            )

        self.Context.save()

    def generate_unanswers_table(self, batch_number, session, reason=''):
        try:
            result = '<{{ {} {}\n((https://scraper.yandex-team.ru/batch/{} Scraper))'.format(
                batch_number, reason, batch_number
            )
            unanswers = session.get(
                'https://scraper.yandex-team.ru/api/ui-facade/batch/{}/unanswers'.format(batch_number),
                verify=False,
            ).json()['by-sources']
            if unanswers:
                result += '\n**Unanswers by sources:**\n#|\n|| Source| Count||\n{}|#'.format(
                    '\n'.join(
                        '|| {}| {}||'.format(source, count) for source, count in sorted(unanswers.iteritems())
                    )
                )
            result += '\n}>'
            return result
        except Exception:
            return "\nFailed to get unanswers table for batch {}\n".format(batch_number)

    @decorators.retries(max_tries=3, delay=5, backoff=5)
    def get_response_json(self, batch_number, session):
        r = session.get(
            "https://scraper.yandex-team.ru/api/scraper/batch/{}/status".format(batch_number),
            verify=False
        )
        try:
            return r.json()
        except Exception as e:
            logging.debug(
                "Failed to get JSON from scraper for batch #%s. Response text: '%s'",
                batch_number,
                r.text
            )
            eh.log_exception("Failed to parse json with error", e)
            return {}

    def check_scraper_problems(self, c_info, metrics_json_response, checked_beta_name):
        try:
            session = requests.Session()
            session.headers["Authorization"] = "OAuth {}".format(self.token)
            session.headers["Accept"] = "application/json"
            session.headers["Content-Type"] = "application/json"

            failed_batches = []
            failed_batches_short = []
            failed_batches_visited = set([])
            all_batches = 0

            for launch in metrics_json_response["launches"]:
                for query_group in launch["queryGroups"]:
                    batch_number = query_group["scraperLink"].split("/")[-1]
                    logging.info("Getting data on %s scraper batch" % batch_number)
                    # todo: use projects.ScrapeRequests.scraper._ScraperProxy#is_batch_completed
                    response_json = self.get_response_json(batch_number, session)
                    logging.info("Json response from scraper: %s" % response_json)
                    status = response_json.get("status", "")
                    if status != "COMPLETE" and batch_number not in failed_batches_visited:
                        failed_batches_visited.add(batch_number)
                        failed_batches.append(self.generate_unanswers_table(
                            batch_number, session, status
                        ))
                        failed_batches_short.append("https://scraper.yandex-team.ru/batch/{}".format(batch_number))
                        all_batches += 1
                        continue
                    failed_serps = response_json["failed-serps"]
                    requested_serps = response_json["requested-serps"]
                    failed_percentage = 100.0 * failed_serps / requested_serps
                    threshold = 1
                    if failed_percentage > threshold - 0.01 and batch_number not in failed_batches_visited:
                        failed_batches_visited.add(batch_number)
                        failed_batches.append(self.generate_unanswers_table(
                            batch_number, session, 'Failed {0:.2f}%'.format(failed_percentage)
                        ))
                        failed_batches_short.append("https://scraper.yandex-team.ru/batch/{}".format(batch_number))
                    all_batches += 1
            launch_info = self.Context.launch_info
            if failed_batches:
                message = "@{}, please take a look at <{{problems with scraper batches:\n".format(
                    c_info.get_responsible_for_release()
                ) + ' '.join(failed_batches) + '}>'
                logging.info("There are problems on scraper, going to post to st:\n%s" % message)
                self.Context.tm_message = (
                    "Warning! Only {} of {} batches downloaded OK from {}. "
                    "Check MLM: <a href='{}'>{}</a> Task {}".format(
                        all_batches - len(failed_batches),
                        all_batches,
                        checked_beta_name,
                        mlm.get_mlm_link(launch_info["launch_id"]),
                        launch_info["name"],
                        lb.task_link(self.id)
                    ),
                )

                if c_info.name in [
                    'middle',
                    'upper',
                    'begemot',
                ]:
                    return None

                return message

            message = "Downloaded {} batches from {}. Link: <a href='{}'>{}</a> Task {}".format(
                all_batches,
                checked_beta_name,
                mlm.get_mlm_link(launch_info["launch_id"]),
                launch_info["name"],
                lb.task_link(self.id)
            )
            self.Context.tm_message = message

        except Exception as e:
            eh.log_exception('Exception while trying to notify about unanswers', e)
        return None

    def get_ticket_key(self, c_info):
        if self.Context.ticket_key is not ctm.NotExists:
            return self.Context.ticket_key
        st_helper = STHelper(self.token)
        issue = st_helper.find_ticket_by_release_number(self.Parameters.release_number, c_info, fail=False)
        ticket_key = issue.key if issue else None
        self.Context.ticket_key = ticket_key
        return ticket_key

    def run_findurl(self, c_info, metrics_json_response):
        if not self.Parameters.run_findurl:
            return None
        try:
            return findurl.FindUrl().generate_result_site(
                self,
                findurl.FindUrl.compare_searches(
                    metrics_json_response,
                    self.Parameters.max_missing_docs_per_basket,
                    self.Parameters.max_queries_per_basket,
                    run_bisect=c_info.metrics_cfg__run_bisect,
                    current_sandbox_task=self
                ),
                self.get_ticket_key(c_info)
            )
        except Exception as e:
            eh.log_exception("Failed miserably to check lost documents with FindUrl", e)

    def send_st_messsage(self, title, message, unanswer_problems=None):
        if not self.c_info.notify_cfg__use_startrek:
            logging.debug("use_startrek = False so not posting to Startrek")
            return
        logging.info("Trying to send startrek comment with title: '%s' and message: '%s'", title, message)
        st_helper = STHelper(self.token)
        # sleep for random amount of time to prevent creating 2 tickets by metrics tasks, started at the same time
        time.sleep(random.uniform(0, 5))
        release_num = self.Parameters.release_number
        metrics_test = rm_const.TicketGroups.MetricsTest
        comment, _ = st_helper.find_comment(release_num, metrics_test, self.c_info)
        found_metrics_launch = self._update_comment_text(comment, title, message)
        logging.info("Tried to find launch in startrek comment and found_metrics_launch=%s", found_metrics_launch)
        if not found_metrics_launch:
            st_helper.write_grouped_comment(
                metrics_test, title, message, release_num, self.c_info
            )
        if unanswer_problems:
            comment, _ = st_helper.find_comment(
                release_num, metrics_test, self.c_info
            )
            if not comment:
                logging.error("Unable to find comment, skip sending message")
                return None

            # clean up summonees, then add back, to send new e-mail
            comment.update(
                text=comment.text,
                summonees=[],
            )
            comment.update(
                text=comment.text,
                summonees=[self.c_info.notify_cfg__st__assignee_after_acceptance or self.c_info.st_assignee],
            )

    def _update_comment_text(self, comment, title, message):
        if not comment:
            return False
        comment_text = comment.text
        comment_parts = re.findall(r"(<{[^<]*Sandbox task:[^0-9]*([0-9]+)[^>]*}>)", comment_text)
        for text_part, task_id in comment_parts:
            if int(task_id) != self.id:
                continue
            comment_text = comment_text.replace(
                text_part,
                "<{{{title}\n{content}}}>\n".format(
                    title=string.all_to_str(title), content=string.all_to_str(message)
                ).decode("utf-8")
            )
            comment.update(
                text=comment_text,
            )
            return True
        return False

    @staticmethod
    def _get_table_column_metric(metric_value, color="red"):
        return '**!!({color}){value}!!**'.format(color=color, value=metric_value)

    @staticmethod
    def _prepare_metric_value(metric_value, check_zero_value=False, metric_name=None):
        try:
            numeric_value = float(metric_value)
        except ValueError:
            # metric value is not numeric and has unicode type
            return string.all_to_str(metric_value)
        if check_zero_value:
            # Report about bad test case if metric value is equal to zero
            if str(numeric_value) == "0.0":
                if metric_name and metric_name.startswith("has-wizard"):
                    # Report about broken wizard check (UPS-12)
                    return lm_const.TEST_WARNING_MESSAGE
                # Report about high or low metric value
                return "Zero diff. Pay attention to the metric value!"

        return "{:+.5f}".format(numeric_value)

    def _generate_wiki_table(self, launch_fails, launch_status):
        if not launch_fails:
            if launch_status != "FAILED":
                return "All metrics are **!!(green)OK!!**\n"
            return ""
        table = ["", "Test name", "Metric name", "Diff value", "Diff percent"]
        t_body = []
        for region, tests in launch_fails.items():
            insert_region = '**{}**'.format(string.all_to_str(region))
            for test in tests:
                tq_params = test['url'].split('?')[-1] if test['url'] else None

                insert_test = string.all_to_str(test['name'])
                if test['url'] and ("view/tests?" in test['url'] or "mc/compare?" in test['url']):
                    insert_test = "(({} {}))".format(test["url"], insert_test)

                for metric in test['metrics']:
                    metric_name = self._get_metrics_name(metric, test, tq_params)
                    # Do not post warns, it cause lots of red zero diffs in issue (RMDEV-521)
                    if metric["status"] == "CRITICAL":
                        diff_value = self._prepare_metric_value(
                            metric.get("diffValue", "Unknown"),
                            check_zero_value=True,
                            metric_name=metric['metricName'],
                        )
                        diff_per_cent = self._prepare_metric_value(metric.get("diffPercent", "Unknown"))
                        t_body.append([
                            insert_region,
                            insert_test,
                            metric_name,
                            self._get_table_column_metric(diff_value, "red"),
                            self._get_table_column_metric(diff_per_cent, "red"),
                        ])
                    else:
                        continue
                    insert_test = ''
                    insert_region = ''

        return wiki_helper.format_table(table, t_body)

    @staticmethod
    def _get_metrics_name(metric, test, tq_params):
        if test['url'] and "view/tests?" in test['url']:
            metric_name = metric['metricName']
        elif tq_params:
            metric_name = '(({0}mc/queries?{1}&metric={2} {3}))'.format(
                rm_const.Urls.METRICS,
                tq_params,
                metric['metricName'],
                string.all_to_str(metric['metricName'])
            )
        else:
            metric_name = '**!!(red)Metrics failed or not finished!!!**'
        return metric_name

    @staticmethod
    def _check_metrics_for_reg(i, reg_eval, changed_regs, unchanged_regs):
        name, metrics = i.get("name"), i.get("metrics", [])
        max_diff = 0
        for metric in metrics:
            if metric["metricId"] == "diff-2-serps-query-5":
                max_diff = max(metric.get("value", 0), metric.get("baselineValue", 0))
                break
        if name:
            if max_diff:
                changed_regs[reg_eval].append((name, max_diff))
            elif reg_eval not in unchanged_regs:
                unchanged_regs.add(reg_eval)
        else:
            logging.debug("Strange info: %s", json.dumps(i, indent=1))

    def check_diff_2_serps_query_5(self, response):
        """SEARCH-3379"""
        changed_regs = defaultdict(list)
        unchanged_regs = set()
        for region in response["launches"]:
            reg = region["regional"]
            evaluation = region.get("evaluation", "")
            reg_eval = "{}_{}".format(reg, evaluation)
            logging.info("Try to show diff-2-serps-query-5 metrics for %s", reg_eval)
            for i in region["diffQueryGroups"]:  # serps
                logging.info("Check diffQueryGroups")
                self._check_metrics_for_reg(i, reg_eval, changed_regs, unchanged_regs)
            for i in region["diffAcceptanceTests"]:  # tests
                logging.info("Check diffAcceptanceTests")
                self._check_metrics_for_reg(i, reg_eval, changed_regs, unchanged_regs)
        logging.info("Changed regions = %s", changed_regs)
        logging.info("Unchanged regions = %s", unchanged_regs)

    def _get_row_template(self, metric, style):
        row_template = "<tr><td>{metric_name}</td><td {style}>{diff_value}</td><td {style}>{diff_percent}%</td></tr>"
        return row_template.format(
            metric_name=self._prepare_metric_value(metric["metricName"]),
            diff_value=self._prepare_metric_value(
                metric["diffValue"],
                check_zero_value=True,
                metric_name=metric["metricName"],
            ),
            diff_percent=self._prepare_metric_value(metric["diffPercent"]),
            style=style,
        )

    def _generate_html(self, launch_fails):
        if not launch_fails:
            return "All metrics valid"
        html_info = [templates.get_html_template("metrics_launch_header.html")]
        for region, tests in launch_fails.items():
            html_info.append("<h3>{}</h3>\n<table>".format(string.all_to_str(region)))
            for test in tests:
                if test["url"]:
                    title = "<a href={}> {} </a>".format(
                        string.all_to_str(test["url"]), string.all_to_str(test["name"])
                    )
                else:
                    title = string.all_to_str(test["name"])
                test_title = '<tr><td colspan="3" class="title"> {} </td></tr>'.format(string.all_to_str(title))
                html_info.append(test_title)
                rows = ""
                for metric in test["metrics"]:
                    if metric["status"] == "WARN":
                        rows += self._get_row_template(metric, style='class="value-warning"')
                    elif metric["status"] == "CRITICAL":
                        rows += self._get_row_template(metric, style='class="value-critical"')
                    elif metric["status"] == "UNKNOWN":
                        rows += (
                            '<tr><td colspan="2">{}</td><td class="value-warning">unknown</td></tr>'.format(
                                string.all_to_str(metric["metricName"])
                            )
                        )
                html_info.append(string.all_to_str(rows))
            html_info.append("</table>")
        html_info.append("</body>")
        return "\n".join(html_info)

    def generate_reports(self, mlm_launch_info, launch_status):
        logging.info("Generating reports")
        launch_fails = mlma.get_launch_fails(mlm_launch_info)
        html_report = self._generate_html(launch_fails)
        table_report = self._generate_wiki_table(launch_fails, launch_status)

        self._save_report([(html_report, "html"), (table_report, "txt")])
        return html_report, table_report, launch_fails

    def _save_report(self, reports):
        report_resource = sdk2.Resource[self.Context.out_resource_id]
        diff_path = str(report_resource.path)
        paths.make_folder(diff_path)
        for report, extension in reports:
            report = string.all_to_str(report)
            fu.write_file(os.path.join(diff_path, self.metrics_output_file_name(extension)), report)
        sdk2.ResourceData(report_resource).ready()

    def _check_final_status(self, mlm_launch_info, launch_fails):
        final_status = mlma.get_final_status(mlm_launch_info)
        self.Context.adjusted_final_status = final_status

        launch_status = mlm_launch_info["status"]

        if self.Parameters.test_mode and launch_status == 'COMPLETED':
            self.set_info("Status '{}' is OK in test mode".format(final_status))
            return

        if final_status not in ["VALID", "WARN"]:
            self.set_info(
                utils2.resource_redirect_link(self.Context.out_resource_id, lm_const.OUTPUT_NAME),
                do_escape=False,
            )

            adjusted_final_status = mlma.calculate_adjusted_final_status(
                final_status, launch_fails, ignored_metrics.CRITICAL_METRICS_IGNORE_LIST
            )
            logging.info("Adjusted final status: %s", adjusted_final_status)
            self.Context.adjusted_final_status = adjusted_final_status

            eh.ensure(
                adjusted_final_status in ["VALID", "WARN"],
                "Launch adjusted final status is: {}".format(adjusted_final_status)
            )

    def _wait_launch(self):
        """
        @return mlm_launch_info
        """
        mlm_launch_info = self.metrics_launcher.get_launch_info(self.Context.launch_info["launch_id"])
        self._update_launch_info(mlm_launch_info)
        if mlm_launch_info["status"] == u'RUNNING':
            current_launch_time_sec = int(time.time()) - self.Context.launch_start
            # Notification sending below should be changed or removed
            # after RMDEV-811 (RMDEV-181)
            if self.c_info:
                notify_error = rm_notify.notify_if_acceptance_delayed(self.c_info, self, current_launch_time_sec)
                if notify_error:
                    logging.error(
                        "Acceptance delay notification FAILED for {component_name}. "
                        "Launch duration:  {launch_duration} s. "
                        "Error: {notify_error}. "
                        "Launch info: {mlm_launch_info}".format(
                            component_name=self.c_info.name,
                            launch_duration=current_launch_time_sec,
                            mlm_launch_info=mlm_launch_info,
                            notify_error=notify_error,
                        )
                    )
            if current_launch_time_sec > int(self.Parameters.fail_threshold):
                self.metrics_launcher.stop_launch(self.Context.launch_info["launch_id"])
                eh.check_failed("Launch running more than {} hours. Stopped it.".format(
                    int(self.Parameters.fail_threshold) / 60 / 60
                ))
            current_sleep_time = self.Context.sleep_time
            logging.info("Current sleep time: %s", current_sleep_time)
            if current_sleep_time is ctm.NotExists:
                current_sleep_time = self.Parameters.optimistic_expected_launch_time_sec
                next_sleep_time = lm_const.INITIAL_SLEEP_TIME
            else:
                next_sleep_time = int(min(current_sleep_time * lm_const.BACKOFF_FACTOR, lm_const.MAX_SLEEP_TIME))
            self.Context.sleep_time = next_sleep_time
            logging.info("Still running, sleep %s seconds", current_sleep_time)
            logging.debug("Current MLM launch info:\n%s\n\n", json.dumps(mlm_launch_info, indent=2))
            if self.c_info is not None and not self.Parameters.silent_mode:
                self.post_preliminary_status(mlm_launch_info, self.c_info)
            raise sdk2.WaitTime(current_sleep_time)
        self.set_info("Launch done with status: {}. Time spent: {}".format(
            mlm_launch_info["status"], dt.timedelta(seconds=int(time.time()) - self.Context.launch_start)
        ))
        logging.debug("Launch info json:\n%s\n\n", json.dumps(mlm_launch_info, indent=2))
        return mlm_launch_info

    def post_preliminary_status(self, mlm_launch_info, c_info):
        try:
            success = True
            no_unknown = True
            for launch in mlm_launch_info["launches"]:
                for diff_query_group in launch['diffQueryGroups']:
                    # if diff_query_group['status'] != 'COMPLETED':
                    logging.info("%s %s", diff_query_group['status'], diff_query_group['name'])
                    for metric in diff_query_group['metrics']:
                        if metric['metricId'] == 'diff-2-serps':
                            logging.info('  %s', metric)
                            if metric['status'] != 'VALID':
                                success = False
                            if metric['status'] == 'UNKNOWN':
                                no_unknown = False
            logging.info("All diff-2-serps are VALID" if success else "Not all diff-2-serps are VALID")
            if not self.Context.preliminary_status_posted:
                if no_unknown:
                    logging.info("There are no 'UNKNOWN' diff-2-serps metrics.")
                    launch_fails = mlma.get_launch_fails(mlm_launch_info)
                    report = self._generate_wiki_table(launch_fails, "PRELIMINARY RESULTS")
                    title, message = c_info.launch_result_in_middle(self, report)
                    self.send_st_messsage(title, message)
                    self.Context.preliminary_status_posted = True
                else:
                    logging.info("There are still some 'UNKNOWN' diff-2-serps metrics.")
            else:
                logging.info("Preliminary status already posted")
        except Exception as e:
            eh.log_exception("Failed to log debug info", e)

    @staticmethod
    def _get_metrics_status(tests):
        """ :return Metrics statuses for region, sorted form valid to critical """
        metrics = [m for test in tests for m in test["metrics"]]
        levels = ["VALID", "UNKNOWN", "WARN", "CRITICAL"]
        defined_metrics = [m for m in metrics if m["status"] != "UNDEFINED"]
        if not defined_metrics:
            # если все метрики имеют статус "UNDEFINED", возвращаем статус "UNDEFINED",
            # это значит, что никакие пороги не были заданы (общий статус прохождения метрик неопределен)
            return "UNDEFINED"
        else:
            # для списка статусов метрик с заданными порогами
            # возвращаем статус с максимальным индексом в списке уровней критичности levels
            statuses = [metric["status"] for metric in defined_metrics]
            return max(statuses, key=lambda s: levels.index(s))

    def get_new_launches(self, mlm_launch_info, qex_launch_info):
        new_launches = []

        # Do not mix QEX launch info with MLM launch info response.
        # They have slightly different structures, sure we need ask Metrics team why
        logging.debug("QEX LAUNCH INFO:\n%s\n\n", json.dumps(qex_launch_info, indent=4))

        qex_info_dict = {launch['id']: launch for launch in qex_launch_info}

        logging.debug('Ready Statuses: %s', {lid: qex_info_dict[lid].get('readyStatus') for lid in qex_info_dict})

        for launch in mlm_launch_info.get("launches", []):
            new_launch = {
                "name": launch["name"],
                "status": launch["status"]
            }
            if launch["status"] not in ["COMPLETED", "CANCELED"]:
                new_launch["metrics_status"] = "UNKNOWN"
            else:
                new_launch["serps"] = [
                    {
                        "name": u"{}.{}".format(serp["name"], serp["filterName"]),
                        "url": serp["detailUrl"],
                        "metrics": serp["metrics"]
                    } for serp in launch["diffQueryGroups"]
                ]
                new_launch["tests"] = [
                    {
                        "name": test["typeName"],
                        "url": test["detailUrl"],
                        "metrics": test["metrics"]
                    } for test in launch["diffAcceptanceTests"]
                ]
                new_launch["metrics_status"] = qex_info_dict.get(launch['id'], {}).get(
                    "readyStatus",
                    self._get_metrics_status(new_launch["serps"] + new_launch["tests"]),
                )
            new_launches.append(new_launch)
        return new_launches

    def _update_launch_info(self, mlm_launch_info):

        if self.Context.launch_info and self.Context.launch_info.get('launch_id'):
            qex_launch_info = self.metrics_launcher.get_qex_launch_info(self.Context.launch_info["launch_id"])
        else:
            qex_launch_info = []

        self.Context.launch_info = {
            "launch_id": mlm_launch_info["id"],
            "name": mlm_launch_info["name"],
            lm_const.LAUNCH_INFO_STATUS_KEY: mlm_launch_info["status"],
            "launches": self.get_new_launches(mlm_launch_info, qex_launch_info),
            lm_const.LAUNCH_INFO_FINAL_STATUS_KEY: mlma.get_final_status(mlm_launch_info),
        }
        self.Context.save()

    def get_template(self):
        # type: ('LaunchMetrics') -> dict
        """
        Get template for the given launch.
        The result and the loading method depend on the value of template_source input parameter.

        :return: Metrics launch template as a dict
        """

        if self.Parameters.template_source == lm_const.TEMPLATE_SOURCE_ARCADIA:
            return self._get_template_from_arcadia()

        if self.Parameters.template_source == lm_const.TEMPLATE_SOURCE_RESOURCE:
            return self._get_template_from_provided_resource()

        if self.Parameters.template_source == lm_const.TEMPLATE_SOURCE_LAST_RELEASED:
            return self._get_template_from_last_released_resource()

        raise ValueError(
            "Unexpected value for input parameter 'template_source': {} (expected one of {})".format(
                self.Parameters.template_source,
                ", ".join([lm_const.TEMPLATE_SOURCE_RESOURCE, lm_const.TEMPLATE_SOURCE_ARCADIA]),
            )
        )

    def _get_template_from_last_released_resource(self):
        # type: ('LaunchMetrics') -> dict
        """Find kosher release of METRICS_TEMPLATE_RESOURCE to the given stage and load template from it"""

        logging.info(
            "Going to find last %s release of %s and load template from it",
            self.Parameters.template_resource_release_stage,
            str(rm_resources.METRICS_TEMPLATE_RESOURCE),
        )

        resource_id = kosher_release.find_release(
            resource_type=str(rm_resources.METRICS_TEMPLATE_RESOURCE),
            stage=self.Parameters.template_resource_release_stage,
            sb_rest_client=self.server,
        )

        if not resource_id:
            raise Exception(
                "Cannot find any resource of type {} released to {}".format(
                    str(rm_resources.METRICS_TEMPLATE_RESOURCE),
                    self.Parameters.template_resource_release_stage,
                ),
            )

        logging.info("%s resource found: %s", str(rm_resources.METRICS_TEMPLATE_RESOURCE), resource_id)

        return self._get_template_from_resource(rm_resources.METRICS_TEMPLATE_RESOURCE[resource_id])

    def _get_template_from_provided_resource(self):
        # type: ('LaunchMetrics') -> dict
        """Load template from resource specified in the `template_resource` input parameter"""
        logging.info("Going to get template from resource %s", self.Parameters.template_resource)
        return self._get_template_from_resource(self.Parameters.template_resource)

    def _get_template_from_arcadia(self):
        # type: ('LaunchMetrics') -> dict
        """Load template file from Arcadia, apply patch if needed and return the result template as a dict."""

        logging.info("Going to get template from Arcadia: %s", self.template_dir)

        checkout_url = sdk2.svn.Arcadia.append(self.Parameters.checkout_arcadia_from_url, self.template_dir)
        sdk2.svn.Arcadia.checkout(checkout_url, self.template_dir)

        eh.verify(
            os.path.exists(os.path.join(self.template_dir, self.template_name)),
            "Template '{}' doesn't exist in directory '{}'".format(self.template_name, self.template_dir)
        )

        if self.Parameters.arcadia_patch:
            sdk2.svn.Arcadia.apply_patch(self.path(), self.Parameters.arcadia_patch, self.path())

        return self.load_template_from_local_path(os.path.join(self.template_dir, self.template_name))

    def _get_template_from_resource(self, resource_obj):
        # type: ('LaunchMetrics') -> dict
        """Load template from resource :param resource_obj:"""

        template_resource_path = str(sdk2.ResourceData(resource_obj).path)
        template_full_path = os.path.join(template_resource_path, self.template_dir, self.template_name)

        logging.info("Template full path: %s", template_full_path)

        eh.verify(
            os.path.exists(template_full_path),
            "Template '{}' does not exist in '{}'".format(self.template_name, self.template_dir),
        )

        return self.load_template_from_local_path(template_full_path)

    @classmethod
    def load_template_from_local_path(cls, template_full_path):
        template = fu.json_load(template_full_path)
        logging.debug("Template loaded:\n%s", json.dumps(template, indent=2))
        return template

    def _make_server(self, template_element, host, region, server_index, enable_auto_clicker):
        template_server = template_element["servers"][server_index]

        name = host.split('.')[0]
        if self.Parameters.search_subtype == rm_const.SearchSubtypes.MUSIC and not host.startswith("http://"):
            host = "http://{}".format(host)

        template_server.update({
            "name": name,
            "baseline": (server_index == 0),
            "host": "{}.{}".format(host, region),
        })

        if "config" not in template_server or not template_server["config"]:
            # absent/null values should be overwritten
            template_server["config"] = {}

        return template_element

    @staticmethod
    def _parse_cgi(cgi_str):
        if not cgi_str:
            return {}

        splitted_params = cgi_str.strip("&").split("&")
        cgi = {}
        for s in splitted_params:
            if s:
                cgi_key, cgi_value = s.split("=", 1)
                if cgi_key not in cgi:
                    cgi[cgi_key] = []
                cgi[cgi_key].append(cgi_value)

        return cgi

    def _fix_cgi(self, template, extra_params, server_index):
        if not extra_params:
            return

        new_cgi = self._parse_cgi(extra_params)
        old_cgi = self._parse_cgi(template["servers"][server_index].get("cgi", ""))

        old_cgi.update(new_cgi)

        # translate {key: [a, b, c]} into key=a&key=b&key=c
        template["servers"][server_index]["cgi"] = "&".join([
            "=".join([k, vi]) for k, v in old_cgi.iteritems()
            for vi in v
        ])

    def _set_limit(self, template_element):
        if not self.Parameters.serp_download_limit:
            return
        for query_group in template_element.get("queryGroups", []):
            if not query_group.get("limitMs"):
                query_group["limitMs"] = int(self.Parameters.serp_download_limit) * 1000

    @staticmethod
    def _set_template_config(template, key, subkey, value):
        for i in [0, 1]:
            if subkey is None:
                template["servers"][i]["config"][key] = value
            else:
                template["servers"][i]["config"].setdefault(key, {})[subkey] = value

    def _change_template(self, template, host_1, host_2, extra_params_1, extra_params_2, auto_clicker_options):
        region = self._get_template_region(template)

        enable_autoclicker = auto_clicker_options.get("enable")
        if self.Parameters.search_subtype != rm_const.SearchSubtypes.GEO:
            self._make_server(template, host_1, region, 0, enable_autoclicker)
            self._make_server(template, host_2, region, 1, enable_autoclicker)

        experiment_config = template.get("experimentConfig")
        # Check that nodiff download required:
        if experiment_config and experiment_config.get("batchedMiscParameters"):
            experiment_config.setdefault('scraperOverYtParameters', {})['pool'] = self.Parameters.scraper_over_yt_pool
        else:
            self._set_template_config(template, 'scraperOverYt', None, True)
            if self.Parameters.scraper_over_yt_pool:
                self._set_template_config(template, 'scraperOverYtPool', None, self.Parameters.scraper_over_yt_pool)
                self._set_template_config(
                    template, 'scraperOverYtParameters', 'scraper-over-yt-pool', self.Parameters.scraper_over_yt_pool
                )

        if enable_autoclicker:
            eh.verify(
                "retries" in auto_clicker_options
                and "metric" in auto_clicker_options,
                "Bad Auto Clicker parameter"
            )
            template["reloadOptions"] = {
                "retryCount": int(auto_clicker_options.get("retries")),
                "diffThreshold": 0,
                "metric": auto_clicker_options.get("metric"),
                "filter": auto_clicker_options.get("filter")
            }

        self._fix_cgi(template, extra_params_1, 0)
        self._fix_cgi(template, extra_params_2, 1)
        self._set_limit(template)
        return template

    def _get_template_region(self, template):

        # music uses host like `<beta_name>.music-web.music.qable.yandex.net`
        if self.Parameters.search_subtype == rm_const.SearchSubtypes.MUSIC:
            return 'net'
        regional = template.get("regional")
        eh.ensure(regional, "Can't get regional for launch template with id {}. ".format(template.get("id")))
        region = mlm.REGIONS.get(regional)
        eh.ensure(region, "Unable to get region by regional '{}'".format(regional))
        return region

    def get_launch_template(self, c_info=None):
        auto_clicker_options = {
            "enable": self.Parameters.enable_autoclicker,
            "retries": self.Parameters.autoclicker_retry_count,
            "metric": self.Parameters.autoclicker_metric_name,
            "filter": self.Parameters.autoclicker_filter
        }

        launch_data = self.get_template()
        self.patch_template_description(launch_data, c_info)
        self.patch_responsibles(launch_data, c_info)
        self.patch_task_project_id(launch_data, c_info)
        if self.Parameters.test_mode:
            self.patch_template_external_id(launch_data)
            logging.debug("Patched template (externalId):\n%s", json.dumps(launch_data, indent=2))
        self.patch_template_sla_project(launch_data)
        if self.is_manual_run:
            launch_data = self.patch_enrichment_options(launch_data)
        old_templates = launch_data.get("launchTemplates")
        eh.ensure(old_templates, "Can't receive old launch template data")

        new_launch_templates = []
        logging.info(
            "Setting hosts: '%s' and '%s'",
            self.sample_beta_host_without_tld,
            self.checked_beta_host_without_tld,
        )
        for old_launch_element in old_templates:
            template_element = self._change_template(
                old_launch_element,
                str(self.sample_beta_host_without_tld),
                str(self.checked_beta_host_without_tld),
                str(self.Parameters.sample_extra_params),
                str(self.Parameters.checked_extra_params),
                auto_clicker_options,
            )
            launch_quota = self.Parameters.launch_template_quota
            if launch_quota:
                for server in template_element.get("servers", []):
                    config = server.get("config")
                    if config:
                        config.setdefault("quota", launch_quota)
            new_launch_templates.append(template_element)

        launch_data["launchTemplates"] = new_launch_templates
        return launch_data

    def patch_responsibles(self, template, c_info=None):
        """
        Add important people to MLM template (see RMDEV-387)
        """
        responsibles = set(template["responsibleUsers"])
        responsibles.add(self.author)  # task author should be able to restart launch (maybe robot)
        if c_info:
            responsible_for_release = c_info.get_responsible_for_release()
            if responsible_for_release:
                responsibles.add(responsible_for_release)
            if c_info.notify_cfg__st:
                if c_info.notify_cfg__st__assignee_after_acceptance:
                    responsibles.add(
                        rm_responsibility.get_responsible_user_login(
                            c_info.notify_cfg__st__assignee_after_acceptance,
                        ),
                    )
                if c_info.notify_cfg__st__assignee:
                    responsibles.add(c_info.st_assignee)
                # these people typically don't need to, but...
                for st_follower in c_info.get_followers(self.token):
                    responsibles.add(st_follower)

        template["responsibleUsers"] = list(responsibles)

    def patch_template_description(self, template, c_info=None):
        descr = [
            self.Parameters.custom_template_descr or
            "'{checked_beta} ({checked_beta_version})' vs '{sample_beta}'".format(
                checked_beta=self.Parameters.checked_beta,
                checked_beta_version=self.Context.checked_beta_version,
                sample_beta=self.Context.sample_beta_name,
            ),
            "MLM launch by {author} from Sandbox task {task}".format(
                author=lb.staff_link(self.author, link_type=lb.LinkType.wiki),
                task=lb.task_wiki_link(self.id, self.id),
            ),
        ]

        if self.template_name:
            # RMDEV-1665
            descr.append("Based on `{}`".format(self.template_name))

        if c_info and not self.Parameters.silent_mode:
            descr.append("Responsible for acceptance: {}".format(
                lb.staff_link(c_info.get_responsible_for_release(), link_type=lb.LinkType.wiki)
            ))
            ticket = self.get_ticket_key(c_info)
            if ticket:
                descr.append("Acceptance ticket: {}".format(lb.st_wiki_link(ticket)))
            descr.append(
                "Please report all problems to "
                "((https://nda.ya.ru/t/Rejj-ct33Vy8Sh United Priemka Support)) aka UPS"
            )

        template["description"] = ". ".join(descr)

    def patch_task_project_id(self, template, c_info):
        if c_info:
            # RMDEV-670
            template["taskProjectId"] = self.get_ticket_key(c_info)

    def patch_template_external_id(self, template):
        for launch_template in template.get("launchTemplates", []):
            for group in launch_template.get("queryGroups", []):
                group['externalId'] = self.Parameters.external_id
                for metric in group["metrics"]:
                    # Do not add enrichment for test launches (OFFLINEACL-2310)
                    metric["enrichment"] = False

    def patch_template_sla_project(self, template):
        if self.Parameters.sla_project:
            sla_project = self.Parameters.sla_project
            template['slaProject'] = sla_project
            logging.debug("Patched slaProject (to {})".format(sla_project))
        else:
            logging.debug("`sla_project` not provided. Leaving original values")

    @staticmethod
    def metrics_output_file_name(extension):
        return "{}.{}".format(lm_const.OUTPUT_NAME, extension)

    def patch_enrichment_options(self, launch_data):
        logging.debug("Try to patch enrichment options")
        # Try to patch enrichment options for manual runs
        new_enrichment_options = lm_const.MANUAL_RUNS_ENRICHMENT_QUOTA.get(self.Parameters.search_subtype)

        if self.Parameters.ang_quota:
            # Use passed via params angQuota (RMDEV-350)
            new_enrichment_options = {
                "angQuota": self.Parameters.ang_quota,
                "labels": self.Parameters.ang_quota.replace(" ", "_")
            }

        if not new_enrichment_options:
            return launch_data

        # Try to change angQuota on the top template level
        old_enrichment_options = launch_data.get("enrichmentOptions")
        if old_enrichment_options and old_enrichment_options.get("options"):
            logging.debug("Insert new enrichment options on top level: %s", new_enrichment_options)
            launch_data["enrichmentOptions"]["options"].update(new_enrichment_options)
            return launch_data

        # TODO: check enrichment options on launchTemplate and Metrics level
        return launch_data

    def parse_mlm_details(self, launch_info):
        launch_id = launch_info.get("launch_id")
        if not launch_id:
            return {}
        details = get_aggregated_launchset_detalization(self.metrics_launcher.get_launchset_tasks(launch_id))
        return {
            key: {
                'started_at': details[key]['started_at'].isoformat(),
                'finished_at': details[key]['finished_at'].isoformat(),
                'duration': (details[key]['finished_at'] - details[key]['started_at']).total_seconds(),
            } for key in details
        }

    def _get_rm_proto_event_specific_data(self, rm_proto_events, event_time_utc_iso, status=None):

        status = status or self.status

        logging.info("Building AcceptanceTestData for task status %s", status)

        from release_machine.common_proto import test_results_pb2 as rm_test_results

        launch_info = self.Context.launch_info

        launch_info_status_map = {
            'VALID': rm_test_results.TestResult.TestStatus.OK,
            'WARN': rm_test_results.TestResult.TestStatus.WARN,
            'UNKNOWN': (
                rm_test_results.TestResult.TestStatus.UB if status in ('SUCCESS', 'FAILURE', 'TIMEOUT', 'EXCEPTION')
                else rm_test_results.TestResult.TestStatus.ONGOING
            ),
            'FATAL': rm_test_results.TestResult.TestStatus.CRIT,
            'CRITICAL': rm_test_results.TestResult.TestStatus.CRIT,
        }

        metrics_status_style_class_map = {
            'VALID': 'table-cell-success',
            'WARN': 'table-cell-warning',
            'CRITICAL': 'table-cell-fail',
        }

        mlm_details = self.parse_mlm_details(launch_info)

        logging.info("Launch info status is %s", launch_info.get(lm_const.LAUNCH_INFO_STATUS_KEY))
        logging.info("Launch info final status is %s", launch_info.get(lm_const.LAUNCH_INFO_FINAL_STATUS_KEY))

        first_failed_launch_id = mlma.get_first_failed_launch_id(self.Context.mlm_launch_info)

        test_result_status = launch_info_status_map.get(
            self.Context.adjusted_final_status,  # RMDEV-3170
            rm_test_results.TestResult.TestStatus.ONGOING,
        )

        return {
            'acceptance_test_data': rm_proto_events.AcceptanceTestData(
                acceptance_type=rm_proto_events.AcceptanceTestData.AcceptanceType.METRICS,
                job_name=self.Context.job_name,
                revision=six.text_type(self.Context.svn_revision),
                scope_number=six.text_type(self.Parameters.release_number),
                experiment_id=self.Parameters.ab_experiment_id,
                details=[
                    rm_proto_events.AcceptanceTestPartDetails(
                        key=key,
                        started_at=mlm_details[key]['started_at'],
                        finished_at=mlm_details[key]['finished_at'],
                        duration_sec=mlm_details[key]['duration'],
                    ) for key in mlm_details
                ],
                test_result=rm_test_results.TestResult(
                    status=test_result_status,
                    report_link=mlm.get_mlm_link(
                        template_id=launch_info.get("launch_id"),
                        launch_id=first_failed_launch_id,
                        critical_only=(test_result_status == rm_test_results.TestResult.TestStatus.CRIT),
                    ),
                    report_message='',
                    task_link=lb.task_link(self.id, plain=True),
                    report_table=rm_test_results.Table(
                        header=rm_test_results.TableRow(
                            cells=[
                                rm_test_results.TableCell(
                                    content=u"MLM Launch Block",
                                ),
                                rm_test_results.TableCell(
                                    content=u"Launch Status",
                                ),
                                rm_test_results.TableCell(
                                    content=u"Metrics Status",
                                ),
                            ]
                        ),
                        rows=[
                            rm_test_results.TableRow(
                                cells=[
                                    rm_test_results.TableCell(
                                        content=launch['name'],
                                    ),
                                    rm_test_results.TableCell(
                                        content=launch[lm_const.LAUNCH_INFO_STATUS_KEY],
                                    ),
                                    rm_test_results.TableCell(
                                        content=launch['metrics_status'],
                                        style_class=metrics_status_style_class_map.get(launch['metrics_status'], '')
                                    ),
                                ]
                            ) for launch in launch_info.get("launches", [])
                        ],
                    ),
                ),
            )
        }
