# -*- coding: utf-8 -*-

import logging

import sandbox.common.types.task as ctt
import sandbox.projects.common.link_builder as lb
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.helpers.merge_helper as rm_merge
import sandbox.projects.release_machine.helpers.startrek_helper as rm_st
import sandbox.projects.release_machine.input_params2 as rm_params
import sandbox.projects.release_machine.rm_notify as rm_notify
import sandbox.projects.release_machine.tasks.base_task as rm_bt
from sandbox import sdk2
from sandbox.projects.common import decorators
from sandbox.projects.common import binary_task
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import string
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.helpers import svn_helper
from sandbox.projects.release_machine.helpers.arcanum_helper import ArcanumApi
from sandbox.sdk2.vcs import svn


@rm_notify.notify2()
class LogMergesInStartrek(rm_bt.BaseReleaseMachineTask):
    """
        **Release-machine**

        Таск для логирования информации о мерджах в стабильные ветки (SEARCH-1431).
        Используется в рамках релизного цикла в Релизной Машине, SDK2.
    """

    class Requirements(task_env.StartrekRequirements):
        pass

    class Parameters(rm_params.ComponentName2):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        kill_timeout = 600  # 10 min
        release_number = sdk2.parameters.Integer("Release number", default=0, required=True)
        merge_revision = sdk2.parameters.Integer("Merge revision", default=None, required=True)

    def on_enqueue(self):
        rm_bt.BaseReleaseMachineTask.on_enqueue(self)
        self.Requirements.semaphores = ctt.Semaphores(
            acquires=[ctt.Semaphores.Acquire(name="Log_merges_{}_{}".format(
                self.Parameters.component_name,
                self.Parameters.release_number,
            ), capacity=1)],
            release=(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH),
        )

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)
        if not self.Parameters.release_number or not self.Parameters.merge_revision:
            eh.check_failed("Some input parameters are undefined!")
        c_info = rmc.COMPONENTS[self.Parameters.component_name]()
        c_info.check_testenv_db(self)
        rev_info = self._get_rev_info(self.Parameters.merge_revision)
        logging.info("Got revision info:\n%s", rev_info)
        if "[branch:" in rev_info["message"]:
            self.set_info("This is the revision of branch creation, so don't comment to ticket")
            return
        try:
            self._log_merge_history_in_startrek(c_info)
        except Exception as e:
            eh.log_exception("Can't log merge in startrek", e)
            raise

    def _log_merge_history_in_startrek(self, c_info):
        major_release_number = self.Parameters.release_number
        if c_info.notify_cfg__st__write_merges_in_table:
            issue = self._write_merges_to_table(c_info, major_release_number)
        else:
            issue = self._write_comment(c_info, major_release_number)
        if issue:
            self.set_info("Log merges added in {}".format(lb.st_link(issue.key)), do_escape=False)

    def _write_merges_to_table(self, c_info, major_release_number):
        reverse = c_info.notify_cfg__st__merged_revisions_order == rm_const.SortOrder.DESCENDING
        iter_tags = svn_helper.SvnHelper.iter_tags(
            c_info.full_branch_path(major_release_number),
            self.Parameters.merge_revision,
            reverse,
        )
        rows = []
        for i, tag_and_rev in enumerate(iter_tags):
            tag_num, rev = tag_and_rev
            tag_wiki_name = c_info.tag_wiki_page_name(branch_num=major_release_number, tag_num=tag_num)
            rev_info = self._get_rev_info(rev)
            message, revs_to_merge = rm_merge.update_message(rev_info["message"], rev)
            tag_full_path = string.left_strip(
                c_info.full_tag_path(tag_num=tag_num, branch_num=major_release_number),
                "arcadia:/",
            )

            if "rollback" in message.lower():
                action = "rollback"
            else:
                action = "merge"

            rows.append(
                [
                    lb.wiki_link_from_arcadia_path(path=tag_full_path, name=tag_wiki_name),
                    ", ".join(lb.revision_link(r, link_type=lb.LinkType.wiki) for r in revs_to_merge),
                    lb.revision_link(rev, link_type=lb.LinkType.wiki),
                    message,
                    self._obtain_st_tickets(revs_to_merge),
                    action,
                ]
            )
        return self._st_helper.replace_grouped_table(
            rm_const.TicketGroups.MergedRevisions,
            rows,
            major_release_number,
            c_info
        )

    def _write_comment(self, c_info, major_release_number):
        merged_revisions = svn_helper.SvnHelper.iter_tags(
            c_info.full_branch_path(major_release_number),
            self.Parameters.merge_revision,
            True,  # Reverse revisions to get last
        )
        last_tag_num, last_revision = next(merged_revisions)
        rev_info = self._get_rev_info(last_revision)
        last_message, revs_to_merge = rm_merge.update_message(rev_info["message"], last_revision)
        return self._st_helper.comment(
            major_release_number,
            "{}\nLink to the tag: {}\n{}\n".format(
                last_message,
                self._get_tag_link(major_release_number, tag_num=last_tag_num),
                self._obtain_st_tickets(revs_to_merge),
            ),
            c_info,
        )

    def _get_tag_link(self, release_num, tag_num):
        c_info = rmc.COMPONENTS[self.Parameters.component_name]()
        path = string.left_strip(
            c_info.full_tag_path(tag_num=tag_num, branch_num=release_num),
            "arcadia:/",
        )
        tag_name = path.split('/')[-1]
        tag_link = lb.wiki_link_from_arcadia_path(path, name=tag_name)
        return tag_link

    @decorators.memoized_property
    def _rm_token(self):
        return rm_sec.get_rm_token(self)

    @decorators.memoized_property
    def _arcanum_api(self):
        return ArcanumApi(token=self._rm_token)

    @decorators.memoized_property
    def _st_helper(self):
        return rm_st.STHelper(self._rm_token)

    def _obtain_st_tickets(self, revs):
        logging.info("Try to get tickets found for revisions %s", revs)
        try:
            st_tickets = set([])
            for rev in revs:
                st_tickets |= set(self._arcanum_api.get_st_tickets_for_commits(rev))
            logging.info("Tickets to link: %s", st_tickets)
            return "\n".join(map(str, st_tickets))
        except Exception:
            logging.exception("Unable to get tickets")
        return ""

    @staticmethod
    def _get_rev_info(rev):
        info = {
            "author": "",  # todo: use it or remove
            "message": "",
        }
        info_list = svn.Arcadia.log(svn_helper.BRANCH_PATH, rev, revision_to=rev, limit=1)
        if info_list:
            rev_info = info_list[0]
            info["author"] = "staff:{}".format(string.all_to_str(rev_info["author"]))
            info["message"] = string.all_to_str(rev_info["msg"])
        else:
            logging.info("WARNING! Nothing found for revision: %s", rev)

        return info
