import hashlib
import logging
import re
import textwrap
import collections

from six import iteritems

import sandbox.common.types.misc as ctm
import sandbox.projects.common.file_utils as fu
import sandbox.projects.common.search.bugbanner2 as bb2
import sandbox.projects.release_machine.components.all as rmc
import sandbox.projects.release_machine.core as rm_core
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.resources as rm_res
import sandbox.projects.release_machine.rm_notify as rm_notify
import sandbox.projects.release_machine.tasks.base_task as rm_bt
import sandbox.sdk2 as sdk2
import sandbox.projects.common.vcs.arc as arc_client
import sandbox.projects.common.time_utils as tu
from sandbox.projects.common import constants
from sandbox.projects.common import decorators
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import string
from sandbox.projects.common import utils
from sandbox.projects.release_machine import input_params2 as rm_params
from sandbox.projects.release_machine import rm_utils
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.components.configs import all as rm_configs
from sandbox.projects.release_machine import arc_helper
from sandbox.projects.release_machine.helpers import arcanum_helper
from sandbox.projects.release_machine.helpers import commit
from sandbox.projects.release_machine.helpers import events_helper
from sandbox.projects.release_machine.helpers import merge_helper
from sandbox.projects.release_machine.helpers import review_helper
from sandbox.projects.release_machine.helpers import svn_helper
from sandbox.sandboxsdk.svn import Arcadia

COMPONENTS_AND_LAST_BRANCHES_KEY = 'components_and_last_branches'


ExtractedCommitsInfo = collections.namedtuple("ExtractedCommitsInfo", ["revs", "authors", "messages"])


class MergeRollbackBaseTask(rm_bt.BaseReleaseMachineTask):
    """
        Base task for MergeToStable and RollbackCommit tasks. Includes whole common logic from both of them.
    """

    _extracted_commit_messages = None
    _extracted_commit_authors = None

    class Requirements(task_env.StartrekRequirements):
        disk_space = 60000  # 60 Gb for checkouts

    class Parameters(rm_params.BaseReleaseMachineParameters):
        kill_timeout = 40 * 60  # 40 min

        do_commit = sdk2.parameters.Bool("Do commit", default_value=True)

        developers_options = sdk2.parameters.Bool("Enable developer's options", default=False, do_not_copy=True)
        with developers_options.value[True]:
            commit_user = sdk2.parameters.String('Use this username as committer')
            need_check = sdk2.parameters.Bool("Don't skip check for commit", default=False)
            never_merge_to_released_branches = sdk2.parameters.Bool(
                "Merge only into not released branches (RMDEV-283)",
                default=False,
            )
            commit_message = sdk2.parameters.String(
                "Commit message that contains mergeto markers "
                "(leave this field blank unless you understand what are you doing)",
                description="Field passed via Testenv, for task manual runs left this field blank"
            )

            use_new_checkouts = sdk2.parameters.Bool("Use ram for checkouts", default=True)

            with sdk2.parameters.Group("Service Parameters") as service_params:
                publish_task_to_st = sdk2.parameters.Bool("Show task in release ticket", default=False)
                # RMDEV-228 and RMDEV-252 workarounds
                force_merge = sdk2.parameters.Bool(
                    "Include --ignore-ancestry in `svn merge` command. "
                    "Don't use it unless you REALLY understand what it does. "
                    "See RMDEV-228 for details",
                    default_value=False,
                )
                mark_as_beta = sdk2.parameters.Bool(
                    "Commit needs Release Machine testing (SEARCH-2560)",
                    default=False,
                )
                skip_arc_branch = sdk2.parameters.Bool(
                    "For arc-components, merge/rollback only from svn branch, not arc."
                    "Warning: using this option invalidates your branch for further arc-svn synchronization until you "
                    "use conflict resolver",
                    default_value=False,
                )
                skip_svn_branch = sdk2.parameters.Bool(
                    "Do NOT perform ARC->SVN sync. For ARC components only.",
                    default_value=False,
                )
                send_notifications = sdk2.parameters.Bool(
                    "Send events and notifications (RMDEV-1992)",
                    default=True,
                )
                use_sync_conflict_resolver = sdk2.parameters.Bool(
                    "Resolve conflict between svn and arc automatically",
                    default=False,
                )

        with sdk2.parameters.Output():
            merge_conflicts = sdk2.parameters.Resource(
                "Directory with merge conflicts",
                resource_type=rm_res.MergeConflicts,
            )

    def on_enqueue(self):
        rm_bt.BaseReleaseMachineTask.on_enqueue(self)
        if self.Parameters.use_new_checkouts:
            self.Requirements.ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 50000, None)  # 50 Gb for new checkouts
        with self.memoize_stage.merge_conflicts_dir:
            self.Parameters.merge_conflicts = rm_res.MergeConflicts(
                task=self,
                description="Directory with merge conflicts",
                path="merge_conflicts",
            )

    def _check_input_params(self):
        raise NotImplementedError

    def _prepare_components_and_branch_num(self, revs):
        raise NotImplementedError

    def _on_execute(self, revs, commit_info, action_items, components_and_last_branches):
        raise NotImplementedError

    @decorators.memoized_property
    def commit_user(self):
        return self.Parameters.commit_user or self.author

    class Context(rm_bt.BaseReleaseMachineTask.Context):
        extracted_revs = []
        merged_revs = []
        success_merges = []
        failed_merges = []
        reviews = []
        reviews_check_attempt = 0

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)
        self._arc = arc_client.Arc(secret_name=rm_const.COMMON_TOKEN_NAME, secret_owner=rm_const.COMMON_TOKEN_OWNER)
        do_commit = self.Parameters.do_commit
        if not do_commit:
            # enable additional bugbanners in dry-run mode (for RMINCIDENTS-378)
            self.add_bugbanner(bb2.Banners.ReleaseMachineTest)
        if self.Context.reviews:
            # We need wait for tests in reviews to complete
            with self.memoize_stage.wait_for_tests(max_runs=100):
                # We already have send all revisions to review, now we need to check their statuses
                review_helper.wait_for_tests(task=self)

        with self.memoize_stage.check_input_params:
            # Base check for input parameters
            check_input = self._check_input_params()
            self.verify_input_params(check_input)

        c_info = None
        branch_number = None
        if (
            self.action_type == rm_const.ActionType.ROLLBACK and
            self.action_mode == rm_const.RollbackMode.RELEASE_MACHINE_MODE
        ):
            c_info = rmc.COMPONENTS[self.Parameters.component_name]()
            branch_number = self.Parameters.branch_num
        revs = arc_helper.extract_revisions(
            self.Parameters.revs,
            arc_client=self._arc,
            reverse=(self.action_type == rm_const.ActionType.ROLLBACK),
            c_info=c_info,
            branch_number=branch_number,
        ).result

        self.Context.extracted_revs = revs
        self.Context.save()

        logging.info("Revisions to %s: %s", self.action_type, revs)
        logging.info('%s mode is %s', self.action_type.capitalize(), self.action_mode)

        action_items, comp_and_last_branches, commit_info = self._prepare_components_and_branch_num(revs)

        self._post_started_events()

        check_branch_existence = self._check_merge_types(action_items)
        self.verify_input_params(check_branch_existence)

        self._on_execute(revs, commit_info, action_items, comp_and_last_branches)

    def check_review_creation_possibility(self, action_items):
        if self.Parameters.skip_arc_branch:
            logging.info("Got `skip_arc_branch` option enabled, create review only in svn")
            return
        if self.action_mode == rm_const.RollbackMode.TRUNK_AND_MERGE_MODE:
            c_info = rmc.COMPONENTS[self.Parameters.component_name_to_merge]()
            if c_info.svn_cfg__use_arc:
                eh.check_failed(
                    "We could not create review with precommit checks for arc release branch, failing"
                )

        if self.action_mode == rm_const.RollbackMode.RELEASE_MACHINE_MODE:
            for action_item in action_items:
                if action_item.c_info and action_item.c_info.svn_cfg__use_arc:
                    eh.check_failed(
                        "We could not create review with precommit checks for arc release branch, failing"
                    )

    def verify_input_params(self, check):
        if not check.ok:
            try:
                self._notify_on_errors(check.result)
            except Exception as exc:
                eh.log_exception("Got exception while posting events", exc)
                error_msg = "Got exception while posting events: {exc}".format(exc=exc)
                rm_notify.send_tm_message(self, error_msg, [rm_core.rm_const.RM_USERS["rm_maintainers"].telegram])
            eh.check_failed(check.result)

    def add_info_for_conflicts(self):
        if self.Parameters.merge_conflicts.path.exists():
            self.set_info(
                lb.resource_link(
                    resource_id=self.Parameters.merge_conflicts.id,
                    link_name="Directory with merge conflicts",
                ),
                do_escape=False,
            )

    @property
    def action_mode(self):
        raise NotImplementedError

    @property
    def is_release_machine_mode(self):
        return self.action_mode == rm_const.CommitMode.RELEASE_MACHINE_MODE

    def header(self):
        header_parts = ["{} results:".format(self.action_type.capitalize())]
        succes_merges = self.Context.success_merges
        if succes_merges:
            header_parts.append(("<span class='status status_success'>Success</span>: " + ", ".join(succes_merges)))
        failed_merges = self.Context.failed_merges
        if failed_merges:
            header_parts.append(("<span class='status status_exception'>Fail</span>: " + ", ".join(failed_merges)))
        header_parts.append("Revisions to {}:".format(self.action_type))
        revs = merge_helper.split_and_check_revs_correctness(
            self.Parameters.revs,
        )
        if not revs.ok:
            header_parts.append("You've entered wrong revision's format, got error {}".format(revs.result))
        else:
            for revision in self.Context.extracted_revs:
                header_parts.append(lb.revision_link(revision))
        return header_parts

    def _check_results_list(self, results):
        """
        If there are any Errors in results list, then join them and return Error object with joined string as value.
        If there are no Errors, just Ok's or nothing, then return Ok object
        :param results: list with Error or Ok objects
        :return: Ok or Error object
        """
        errors = [x.result for x in results if not x.ok]
        logging.debug("Got errors: %s", errors)
        if errors:
            # Here we join all errors from results in one string
            return rm_core.Error("\n".join(errors))
        return rm_core.Ok()

    def _check_merge_types(self, action_items):
        """
        Check whether are there any branches, which don't exist
        :param action_items: list with objects ActionItem as values
        :return: Send tm message and fail if there are any bad branches, else return None
        """
        branch_existence_results = []
        for action_item in action_items:
            # We need just merge_types for check
            for merge_type in action_item.merge_types:
                branch_existence_results.append(
                    merge_helper.check_branch_existence(
                        comp_name=action_item.name,
                        branch=merge_type.full,
                        rev_to_checkout=self.Parameters.revision_to_checkout,
                    )
                )
        return self._check_results_list(branch_existence_results)

    def _check_component_name(self, component_name):
        if not component_name:
            return rm_core.Error("Please, choose component_name!")
        if component_name not in rm_configs.get_all_names():
            return rm_core.Error("You've entered component_name, which doesn't exist: {}".format(component_name))
        return rm_core.Ok()

    def _notify_on_errors(self, errors_string, recipients=None):
        """
        Sends notifies on errors
        :param errors_string: string with all input errors
        """
        if not recipients:
            recipients = [self.commit_user]
        recipients = list(set(recipients))
        message = textwrap.dedent(
            """
            {task_type_capitalize} {task_status}

            Dear {recipients}, there are some errors in task {task_type} {task} because of bad task input parameters.

            Errors: {errors_string}
            """.format(
                task_type_capitalize=self.action_type.capitalize(),
                task_status="failure",
                recipients=', '.join(recipients),
                task_type=self.action_type,
                task=lb.task_link(self.id),
                errors_string=errors_string,
            )
        )

        self.send_notifications(recipients, message, subject="Bad input parameters in task {id}".format(id=self.id))

    def send_notifications(self, recipients, message, subject=None):
        from release_machine.common_proto import events_pb2, notifications_pb2
        try:
            event_time_utc_iso = tu.datetime_utc_iso()
            notify_list = self._prepare_notify_list(recipients)
            logging.debug("Got notify_list: %s", notify_list)

            event = events_pb2.EventData(
                general_data=events_pb2.EventGeneralData(
                    hash=hashlib.md5(u"{}{}{}".format(
                        self.gsid,
                        rm_const.EVENT_HASH_SEPARATOR,
                        event_time_utc_iso,
                    )).hexdigest(),
                    component_name=getattr(self.Parameters, "component_name", "release_machine") or "release_machine",
                    referrer="sandbox_task:{}".format(self.id),
                ),
                task_data=self._get_rm_proto_event_task_data(event_time_utc_iso, self.status),
                custom_message_data=events_pb2.CustomMessageData(
                    message=message,
                    condition_tag=rm_const.RELEASE_MACHINE_ERROR_REPORT,
                ),
                custom_notifications=self._get_custom_notifications(notify_list, message, notifications_pb2, subject),
            )
            events_helper.EventHelper(use_testing_client=False).post_grpc_events([event])
        except Exception as exc:
            eh.log_exception("Got exception while posting event", exc)

    def _get_custom_notifications(self, notify_list, message, notifications_pb2, subject=None):
        custom_notifications = []
        subject = subject or "Notification from SB task {id}".format(id=self.id)

        for transport_type, chat_id in notify_list:
            if transport_type == rm_notify.TransportType.EMAIL:
                notification_bundle = notifications_pb2.NotificationBundle(
                    email_data=notifications_pb2.EmailEventNotificationData(
                        addr=chat_id,
                        subject=subject,
                        body=message,
                    )
                )
            elif transport_type == rm_notify.TransportType.TELEGRAM:
                notification_bundle = notifications_pb2.NotificationBundle(
                    telegram_data=notifications_pb2.TelegramEventNotificationData(
                        chat_id=str(chat_id),
                        message=message,
                    )
                )
            else:
                notification_bundle = notifications_pb2.NotificationBundle(
                    q_messenger_data=notifications_pb2.QMessengerEventNotificationData(
                        chat_id=chat_id,
                        message=message,
                    )
                )
            custom_notifications.append(notification_bundle)
        return custom_notifications

    def _prepare_notify_list(self, recipients):
        """
        Prepare list with pairs (transport_type, chat_id) for each recipient,
        where transport_type - one of (email, tg, q).
        :param recipients: list with people logins
        :return: list with pairs (transport_type, chat_id)
        """
        notify_list = []
        for recipient in recipients:
            notify_list.append((rm_notify.TransportType.EMAIL, recipient))
            if recipient in rm_core.rm_const.RM_USERS:
                notify_list.append((rm_notify.TransportType.TELEGRAM, rm_core.rm_const.RM_USERS[recipient].telegram))
                notify_list.append(
                    (rm_notify.TransportType.Q_MESSENGER, rm_core.rm_const.RM_USERS[recipient].q_messenger)
                )
        return notify_list

    def _get_extracted_commits_info(self):

        result = ExtractedCommitsInfo(revs=[], authors=[], messages=[])

        if not self.Context.extracted_revs:
            return result

        if self._extracted_commit_messages and self._extracted_commit_authors:

            result.revs.extend(self.Context.extracted_revs)
            result.authors.extend(self._extracted_commit_authors)
            result.messages.extend(self._extracted_commit_messages)

            return result

        for rev in self.Context.extracted_revs:

            result.revs.append(rev)

            rev_log = Arcadia.log(Arcadia.ARCADIA_BASE_URL, revision_from=rev, revision_to=rev, limit=1)

            if not rev_log:
                result.authors.append("")
                result.messages.append("")
                continue

            result.authors.append(rev_log[0]["author"])

            commit_message = string.all_to_unicode(rev_log[0]["msg"])
            # get rid of boilerplate part of messages
            commit_message = commit_message.replace(
                u'Note: mandatory check (NEED_CHECK) was skipped', u''
            ).strip()

            result.messages.append(commit_message)

        self._extracted_commit_authors = result.authors
        self._extracted_commit_messages = result.messages

        return result

    def _get_commit_messages_for_custom_notification(self):

        extracted_commits_info = self._get_extracted_commits_info()

        result = []

        for revision, message in zip(extracted_commits_info.revs, extracted_commits_info.messages):

            if len(message) >= rm_notify.COMMIT_MESSAGE_MAX_CHARS:
                message = message[:rm_notify.COMMIT_MESSAGE_MAX_CHARS] + "..."

            result.append(
                u"r{rev_link}: {rev_message}\n========================".format(
                    rev_link=lb.revision_link(revision),
                    rev_message=message,
                )
            )

        return result

    def _send_notifications_for_succeed_action(self, action_items, revs, msg_part, commit_info):
        try:
            action_names = ", ".join([action_item.name for action_item in action_items])
            logging.info("Send notification for succeeded actions: %s", action_names)
            message_parts = [
                u"[{action_names}]".format(action_names=action_names),
                u"SB: {task}, revs: {revs}".format(
                    task=lb.task_link(self.id),
                    revs=", ".join(map(lb.revision_link, revs)),
                ),
                msg_part,
            ]

            messages = self._get_commit_messages_for_custom_notification()

            message_parts.extend(messages)

            setattr(self.Context, rm_notify.TM_MESSAGE, "\n".join(message_parts))
            self.send_tm_notify()

        except Exception as exc:
            eh.log_exception("Telegram message was not sent", exc)

    def _get_commit_info_for_path(self, revs, path, action_type):
        commit_info = svn_helper.CommitInfo()
        try:
            commit_info = svn_helper.SvnHelper.extract_commit_info(revs, path)
        except Exception as exc:
            recipients = commit_info.diff_resolvers + commit_info.committers + [self.commit_user]
            self._notify_on_errors(
                errors_string="No commit info for path '{}'. ".format(path),
                recipients=recipients,
            )
            self.set_info(merge_helper.CHECK_SYNTAX)
            eh.check_failed(
                "Invalid {action_type}: {exc}\nTraceback:\n{traceback}\nContext: {context}".format(
                    action_type=action_type,
                    exc=str(exc),
                    traceback=eh.shifted_traceback(),
                    context="No commit info for path '{}'. ".format(path),
                )
            )
        return commit_info

    def _get_last_branches(self, c_info):
        """
        Get last number_last_branches_to_merge branch numbers for c_info
        :param c_info: Component info
        :return: branch list
        """
        branch_nums = []
        for i in range(c_info.merges_cfg__number_last_branches_to_merge):
            if c_info.last_branch_num - i <= 0:
                break
            branch_nums.append(c_info.last_branch_num - i)
        return branch_nums

    def _get_branch_nums(self, c_info):
        """
        Processes component info and returns suitable
        list of branch numbers to merge to.
        :param c_info: Component info
        :return: branch list to merge to and marker need_to_cut
        """
        if self.Parameters.branch_nums_to_merge:
            # branch nums are explicitly specified, use them
            return self.Parameters.branch_nums_to_merge, False
        # by default, use two last branches:
        return self._get_last_branches(c_info), True

    def _check_reviews_results(self, reviews_results):
        """
        Returns whether all reviews were created.
        :param reviews_results: list of tuples (merge_type, review_result).
        :return: True if all review_results are OK, else False.
        """
        for merge_type, review_res in reviews_results:
            if review_res.status == rm_const.CommitStatus.failed:
                return False
        return True

    def _handle_commit_result(self, merge_type, status):
        """
        Fills Context.failed_merges and Context.success_merges with merge_type
        :param merge_type: CommitPath
        :param status: CommitStatus class element
        :return: None
        """
        if rm_const.CommitMode.RELEASE_MACHINE_MODE == self.action_mode:
            short_result = "{component_name}-{branch_num}".format(
                component_name=merge_type.c_info.name if merge_type.c_info else None,
                branch_num=merge_type.name,
            )
        else:
            short_result = "Path: {path}".format(path=merge_type.name)
        if status == rm_const.CommitStatus.failed:
            self.Context.failed_merges.append(short_result)
        elif status == rm_const.CommitStatus.success:
            self.Context.success_merges.append(short_result)
        elif status == rm_const.CommitStatus.changeset_is_empty:
            logging.debug("Nothing was committed because of empty changeset")

    def _present_reviews_results(self, reviews_results):
        """
        Add info in task for reviews_results, save review ids in Context,
        :param reviews_results: list with tuples (merge_type, review_result)
        :return: None
        """
        review_ids = []
        for merge_type, review_res in reviews_results:
            if review_res.status == rm_const.CommitStatus.success:
                review_ids.append(review_res.review_id)
                self.set_info(
                    "Review for {path} was created: {link}".format(
                        path=merge_type.name,
                        link=lb.review_link(review=review_res.review_id),
                    ),
                    do_escape=False,
                )
            else:
                if review_res.status == rm_const.CommitStatus.changeset_is_empty:
                    review_ids.append(None)
                # If we have `changeset_is_empty` or `failed` status
                self.set_info("Review for {path} was not created".format(path=merge_type.name))
                self._handle_commit_result(merge_type, review_res.status)
            self.Context.reviews = review_ids
            self.Context.save()

    def _check_all_reviews(self, action_items, do_commit, revs, commit_info):
        reviews = self.Context.reviews
        component_and_merge_type_pairs = []
        for action_item in action_items:
            for merge_type in action_item.merge_types:
                component_and_merge_type_pairs.append((action_item.c_info, merge_type))
        arcanum_api = arcanum_helper.ArcanumApi(token=rm_sec.get_rm_token(self))
        merge_result = svn_helper.MergeResult()
        logging.info("Start the final tests check")
        for i, review_id in enumerate(reviews):
            if not review_id:
                # If we have `changeset_is_empty` while merging
                tests_status = rm_const.CiTestResult.success
            else:
                review_info = arcanum_api.get_review_tests_statuses(review_id)['result']
                tests_status = commit.check_review_tests_statuses(review_info)
            if do_commit and tests_status in frozenset([rm_const.CiTestResult.failed, rm_const.CiTestResult.running]):
                merge_result.update(
                    commit_status=rm_const.CommitStatus.failed,
                    merge_path=component_and_merge_type_pairs[i][1],
                )
                self._handle_commit_result(
                    merge_type=component_and_merge_type_pairs[i][1],
                    status=rm_const.CommitStatus.failed,
                )
                if tests_status == rm_const.CiTestResult.failed:
                    self.Context.failed_merges.append(
                        "Tests in review {review_id} failed".format(review_id=review_id)
                    )
                else:
                    self.Context.failed_merges.append(
                        "Tests in review {review_id} are still running".format(review_id=review_id)
                    )
                commit_result = rm_const.CommitResult(rm_const.CommitStatus.failed, None)
            elif tests_status == rm_const.CiTestResult.success:
                logging.info("Tests in review {review_id} have succeed".format(review_id=review_id))
                commit_result = commit.wait_review_for_merge(review_id, arcanum_api)
                merge_result.update(
                    commit_status=commit_result.status,
                    merge_path=component_and_merge_type_pairs[i][1],
                    revision=commit_result.revision,
                )
                self._handle_commit_result(
                    merge_type=component_and_merge_type_pairs[i][1],
                    status=commit_result.status,
                )
                if do_commit and commit_result == rm_const.CommitStatus.success:
                    try:
                        self._on_success_commit(component_and_merge_type_pairs[i][1], commit_result.revision, revs)
                    except Exception as exc:
                        eh.log_exception("Error with _on_success_commit", exc)
            if do_commit:
                commit.post_commit_actions(
                    task=self,
                    merge_path=component_and_merge_type_pairs[i][1],
                    commit_info=commit_info,
                    commit_failed=commit_result.status == rm_const.CommitStatus.failed,
                    revs=revs,
                    committed_revision=commit_result.revision,
                    c_info=component_and_merge_type_pairs[i][0],
                )
        if do_commit:
            commit.post_to_arcanum(self, commit_info.rb_review_ids, merge_result)
        return merge_result

    def _set_components_and_last_branches(self, component_name, cut_branch_nums, need_to_cut):
        max_cut_branch_nums = max(cut_branch_nums)
        lbn_dict = getattr(self.Context, COMPONENTS_AND_LAST_BRANCHES_KEY) or {}
        lbn_dict.update({
            component_name: max_cut_branch_nums
        })
        setattr(self.Context, COMPONENTS_AND_LAST_BRANCHES_KEY, lbn_dict)
        self.Context.save()
        if not need_to_cut:
            return {}
        return lbn_dict

    def get_components_and_branch_nums(self, commit_info, component_name):
        action_items = []
        components_and_last_branches = {}
        if component_name:
            # We have already checked that component_name is valid (see _check_input_params)
            c_info = rmc.COMPONENTS[component_name]()
            branch_nums, need_to_cut = self._get_branch_nums(c_info)
            cut_branch_nums = self._cut_branch_nums(c_info, branch_nums, need_to_cut)
            logging.debug("Got branches to merge to %s for component %s", cut_branch_nums, component_name)

            components_and_last_branches.update(
                self._set_components_and_last_branches(c_info.name, cut_branch_nums, need_to_cut)
            )
            merge_types = []
            for branch_num in cut_branch_nums:
                merge_types.append(
                    svn_helper.MergeReleaseMachinePath(branch_num, c_info, getattr(self.ramdrive, "path", None)),
                )
            action_items.append(
                rm_const.ActionItem(mode=self.action_mode, c_info=c_info, merge_types=merge_types)
            )
        else:
            # We don't have any component_name in task parameters,
            # so it should be started by `mergeto` hook in commit_message.
            return self.get_components_and_branch_nums_from_mergeto(commit_info)
        return action_items, components_and_last_branches

    def get_components_and_branch_nums_from_mergeto(self, commit_info):
        components_and_last_branches = {}
        action_items = []
        allowed_components = rmc.get_component_names()
        try:
            # RMDEV-1106: Method `parse_mergeto` can throw, so we'll loose email notification in this case
            # (e.g.ill-formed component name specification). So we need to parse message under same
            # try-catch block.
            mergeto_info = merge_helper.parse_mergeto(self.Parameters.commit_message, allowed_components)
            if not mergeto_info:
                # User hasn't entered component_name in task Parameters and there is no mergeto in commit message.
                raise Exception('No component name specified in task parameters or in `mergeto` marker commit message.')

            logging.debug('mergeto_info: %s', mergeto_info)

            for component_name, branch_nums in mergeto_info.items():
                rmc.check_component_name(component_name)
                c_info = rmc.get_component(component_name)
                need_to_cut = False

                if len(branch_nums) == 0:
                    # by default, use two last branches:
                    branch_nums_split = self._get_last_branches(c_info)
                    need_to_cut = not c_info.svn_cfg__merge_to_old_branches
                    logging.debug(
                        "Need_to_cut is %s, because merge_to_old_branches in component config is %s",
                        need_to_cut, c_info.svn_cfg__merge_to_old_branches,
                    )
                elif len(branch_nums) > 0 and branch_nums[-1] == '+':
                    first_branch = int(branch_nums[0])
                    branch_nums_split = []
                    for branch_num in range(first_branch, c_info.last_branch_num + 1):
                        branch_nums_split.append(branch_num)
                else:
                    branch_nums_split = branch_nums
                cut_branch_nums = self._cut_branch_nums(c_info, branch_nums_split, need_to_cut)
                components_and_last_branches.update(
                    self._set_components_and_last_branches(c_info.name, cut_branch_nums, need_to_cut)
                )
                merge_types = []
                for branch_num in cut_branch_nums:
                    merge_types.append(
                        svn_helper.MergeReleaseMachinePath(branch_num, c_info, getattr(self.ramdrive, "path", None))
                    )
                action_items.append(
                    rm_const.ActionItem(mode=self.action_mode, c_info=c_info, merge_types=merge_types)
                )
        except Exception as exc:
            recipients = commit_info.diff_resolvers + commit_info.committers + [self.commit_user]
            self._notify_on_errors(
                errors_string=exc,
                recipients=recipients,
            )
            self.set_info(merge_helper.CHECK_SYNTAX)
            eh.check_failed(
                "Invalid {action_type}: {exc}\nTraceback:\n{traceback}\nContext: {context}".format(
                    action_type=self.action_type,
                    exc=str(exc),
                    traceback=eh.shifted_traceback(),
                    context=self.Parameters.commit_message,
                )
            )
        return action_items, components_and_last_branches

    def _cut_branch_nums(self, c_info, branch_nums, need_to_cut=False):
        """
        SEARCH-2257: Do not merge into branches older than current production
        """
        if not need_to_cut:
            return branch_nums
        try:
            if not self.Parameters.merge_to_old_branches:
                logging.info("Try to cut branch numbers: %s", branch_nums)
                last_released_res = next(c_info.get_last_release())
                prod_branch_num_match = re.findall(
                    r'.*/arc/tags/.*{}/arcadia.*'.format(c_info.svn_cfg__tag_folder_pattern),
                    rm_utils.get_ctx_field(last_released_res.build_task_id, constants.ARCADIA_URL_KEY, self.server),
                )
                if prod_branch_num_match:
                    prod_branch_num = int(utils.flatten(prod_branch_num_match).next())
                    if (
                        (self.Parameters.developers_options and self.Parameters.never_merge_to_released_branches) or
                        c_info.merges_cfg__never_merge_to_released_branches
                    ):
                        prod_branch_num += 1
                    branch_nums = [branch_num for branch_num in branch_nums if int(branch_num) >= prod_branch_num]
                    self.set_info(
                        "Cut all branch numbers less than {} (it is last released branch), left only: {}".format(
                            prod_branch_num,
                            branch_nums,
                        )
                    )
                else:
                    logging.warning("Can't get production branch number, merge to all branches")
        except Exception:
            logging.error("Can't cut branch numbers:\n%s", eh.shifted_traceback())
        if not branch_nums:
            tm_message = (
                "[{comp_name}] Can't get any branches to merge to. "
                "Probably you have `never_merge_to_released_branches` flag in your config and last branch "
                "is already released to stable.".format(comp_name=c_info.name)
            )
            self._notify_on_errors(tm_message)
            eh.check_failed(tm_message)
        return branch_nums

    def _do_action(self, action_items, revs, commit_info, common_path):
        do_commit = self.Parameters.do_commit
        merge_result = svn_helper.MergeResult()

        for action_item in action_items:
            c_info = action_item.c_info
            # We check commit_user, when task is launched from RM UI
            granted = (
                c_info.check_merge_permissions(self, self.commit_user)
                if (c_info and self.Parameters.do_commit) else True
            )
            if not granted:
                self.set_info(
                    c_info.merges_cfg__not_granted_message(self.commit_user, c_info.get_responsible_for_component())
                )

            for merge_type in action_item.merge_types:
                if c_info and c_info.svn_cfg__use_arc and not self.Parameters.skip_arc_branch:
                    commit_result = arc_helper.commit_in_arc(
                        self,
                        arc_client=self._arc,
                        revs=revs,
                        descr=self.Parameters.description,
                        commit_info=commit_info,
                        merge_path=merge_type,
                        do_commit=do_commit,
                        c_info=c_info,
                        commit_user=self.Parameters.commit_user,
                    )
                else:
                    if granted:
                        commit_result = commit.perform_action(
                            self,
                            common_path=common_path,
                            revs=revs,
                            descr=self.Parameters.description,
                            commit_info=commit_info,
                            merge_path=merge_type,
                            do_commit=do_commit,
                            ignore_ancestry=self.Parameters.force_merge,
                            checkout_rev=self.Parameters.revision_to_checkout,
                            c_info=c_info,
                            commit_user=self.Parameters.commit_user,
                        )
                    else:
                        commit_result = rm_const.CommitResult(rm_const.CommitStatus.failed)

                if not do_commit:
                    logging.info("Dry run mode for %s, do nothing...", merge_type)

                merge_result.update(
                    commit_status=commit_result.status,
                    merge_path=merge_type,
                    revision=commit_result.revision,
                )

                self._handle_commit_result(merge_type=merge_type, status=commit_result.status)

                merge_result = self._handle_successful_commit_and_sync_branches(
                    c_info=c_info,
                    do_commit=do_commit,
                    commit_result=commit_result,
                    merge_type=merge_type,
                    revs=revs,
                    merge_result=merge_result,
                )

        if do_commit:
            commit.post_to_arcanum(self, commit_info.rb_review_ids, merge_result)

        return merge_result

    def _handle_successful_commit_and_sync_branches(
        self,
        c_info,
        do_commit,
        commit_result,
        merge_type,
        revs,
        merge_result,
    ):

        logging.info("_handle_successful_commit_and_sync_branches ...")

        if not do_commit:
            logging.info("do_commit == False - skipping")
            return merge_result

        if commit_result.status not in {rm_const.CommitStatus.success, rm_const.CommitStatus.changeset_is_empty}:
            logging.info("commit_result.status is %s - skipping", commit_result.status)
            return merge_result

        try:
            self._on_success_commit(merge_type, commit_result.revision or "", revs)
        except Exception as exc:
            eh.log_exception("Error with _on_success_commit", exc)

        logging.info("Going to sync ARC -> SVN ...")

        if self.Parameters.skip_svn_branch:
            logging.info("skip_svn_branch == True - skipping")
            return merge_result

        if not c_info:
            logging.info("c_info is %s - skipping", c_info)
            return merge_result

        if not c_info.svn_cfg__use_arc:
            logging.info("Component %s does not use arc - skipping",  c_info.name)
            return merge_result

        if self.Parameters.skip_arc_branch:
            logging.info("skip_arc_branch == True - skipping")
            return merge_result

        if not (
            isinstance(merge_type, svn_helper.RollbackReleaseMachinePath) or
            isinstance(merge_type, svn_helper.MergeReleaseMachinePath)
        ):
            logging.info("merge type is %s - skipping")
            return merge_result

        sync_status = arc_helper.sync_branch_from_arc(
            self,
            self._arc,
            c_info,
            merge_type.name,
        )

        merge_result.update(
            commit_status=sync_status.status,
            merge_path=merge_type,
            revision=sync_status.revision,
        )

        return merge_result

    def on_finish(self, prev_status, status):
        self._save_merge_conflicts()
        super(MergeRollbackBaseTask, self).on_finish(prev_status, status)

    def on_break(self, prev_status, status):
        self._save_merge_conflicts()
        super(MergeRollbackBaseTask, self).on_break(prev_status, status)

    def _save_merge_conflicts(self):
        if not self.Parameters.merge_conflicts.path.exists():
            fu.write_file(self.Parameters.merge_conflicts.path, "There are no conflicts here")
        sdk2.ResourceData(self.Parameters.merge_conflicts).ready()

    def _get_started_event_data(self, rm_proto_events, scope_number):
        return {}

    def _get_started_event_for_component(self, rm_proto_events, component_name, scope_number):

        if not scope_number:
            self.set_info("Unable to get scope_number for component %s", component_name)

        try:

            from release_machine.common_proto import events_pb2 as rm_proto_events

            event_time_utc_iso = tu.datetime_utc_iso()

            started_event_data = self._get_started_event_data(rm_proto_events, scope_number)

            if not started_event_data:
                logging.info(
                    "Got empty started event data for component %s scope number %s - skipping",
                    component_name,
                    scope_number,
                )
                return

            return rm_proto_events.EventData(
                general_data=rm_proto_events.EventGeneralData(
                    hash=hashlib.md5(u"{}{}{}".format(
                        self.gsid,
                        rm_const.EVENT_HASH_SEPARATOR,
                        event_time_utc_iso,
                    )).hexdigest(),
                    component_name=component_name,
                    referrer="sandbox_task:{}".format(self.id),
                ),
                task_data=self._get_rm_proto_event_task_data(event_time_utc_iso, self.status),
                **started_event_data  # noqa C815
            )

        except Exception as e:

            eh.log_exception("Unable to build proto event", e)

    def _get_started_events(self):

        if not self.is_release_machine_mode:
            logging.info("The task is not running in release machine mode, so no events are going to be sent")
            return []

        from release_machine.common_proto import events_pb2 as rm_proto_events

        events = []

        component_branches = getattr(self.Context, COMPONENTS_AND_LAST_BRANCHES_KEY) or []

        if not component_branches:
            self.set_info("Unable to build started event: component scopes cannot be detected")
            return events

        for component_name, scope_number in iteritems(component_branches):

            event = self._get_started_event_for_component(rm_proto_events, component_name, scope_number)

            if not event:
                logging.warning("Got empty event for component %s scope number %s", component_name, scope_number)
                continue

            events.append(event)

        return events

    def _post_started_events(self):
        if not self.Parameters.send_notifications:
            logging.debug("Send_notifications flag disabled, don't send events")
            return

        action_started_events = self._get_started_events()
        if action_started_events:
            events_helper.EventHelper().post_grpc_events(action_started_events)
