# -*- coding: utf-8 -*-
import collections
import six
import logging

import sandbox.projects.release_machine.core as rm_core
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.helpers.events_helper as events_helper
import sandbox.projects.release_machine.rm_notify as rm_notify
import sandbox.projects.release_machine.tasks.MergeRollbackBaseTask as rm_merge_rollback_task
import sandbox.sdk2 as sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import time_utils as tu
from sandbox.projects.release_machine.components.configs.all import get_all_branched_names
from sandbox.projects.release_machine import arc_helper
from sandbox.projects.release_machine.helpers import commit
from sandbox.projects.release_machine.helpers import merge_helper
from sandbox.projects.release_machine.helpers import review_helper
from sandbox.projects.release_machine.helpers import svn_helper
from sandbox.projects.release_machine.input_params2 import set_choices

MergeMode = rm_const.CommitMode
MERGE_RESULTS_CONTEXT_KEY = 'component_2_results'


class _MergeResultStatus(object):
    SUCCESS = 'success'
    FAIL = 'failure'


@rm_notify.notify2()
class MergeToStable(rm_merge_rollback_task.MergeRollbackBaseTask):
    """
        **Release-machine**

        Implementation MergeToStable sandbox task using SDK2
        Merge specified revisions from trunk to branches. Task description will be added to merge commit message.
        It has various modes:
        - Custom mode is to merge specified revisions to specified branch.
        - Release Machine component mode is to merge specified revisions to branches created by Release Machine.
        Console mode of this task is also available as arcadia/devtools/search-env/merge_to_stable.py
        Docs: https://wiki.yandex-team.ru/ReleaseMachine/merge-scripts/
    """

    class Parameters(rm_merge_rollback_task.MergeRollbackBaseTask.Parameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        with sdk2.parameters.String("Merge mode") as merge_mode_type:
            merge_mode_type.values[MergeMode.RELEASE_MACHINE_MODE] = (
                merge_mode_type.Value("Release machine", default=True)
            )
            merge_mode_type.values[MergeMode.CUSTOM_MODE] = merge_mode_type.Value("Custom")

        with merge_mode_type.value[MergeMode.RELEASE_MACHINE_MODE]:
            branch_nums_to_merge = sdk2.parameters.List(
                "Branch numbers to merge to. " +
                "If omitted, commit will be merged to 2 last branches of the given component",
                sdk2.parameters.Integer,
                default=[],
            )
            with sdk2.parameters.String("Component name", default="", required=False) as component_name:
                set_choices(component_name, get_all_branched_names())

            merge_to_old_branches = sdk2.parameters.Bool(
                "Merge to old branches (SEARCH-2257). "
                "By default, merge to branches older than released stable is NOT performed",
                default_value=False,
            )
            mark_as_beta = sdk2.parameters.Bool(
                "Commit needs Release Machine testing (SEARCH-2560)",
                default_value=False,
                description="Add markers to merged commit to launch heavy test in branch Testenv databases"
            )

        with merge_mode_type.value[MergeMode.CUSTOM_MODE]:
            branch_path = sdk2.parameters.String("Branch path", required=True)
            revision_to_checkout = sdk2.parameters.Integer(
                "Revision to checkout. Please, leave it blank",
                default_value=None,
            )

        revs = sdk2.parameters.String("Revisions to merge (comma-separated)")

    class Context(rm_merge_rollback_task.MergeRollbackBaseTask.Context):
        type = "MERGE_TO_STABLE"

    action_type = rm_const.ActionType.MERGE

    @property
    def action_mode(self):
        return self.Parameters.merge_mode_type

    def _check_input_params(self):
        results = []
        results.append(arc_helper.extract_revisions(self.Parameters.revs, self._arc))
        if self.action_mode == MergeMode.RELEASE_MACHINE_MODE:
            if self.Parameters.branch_nums_to_merge:
                results.append(merge_helper.check_branch_numbers_correctness(self.Parameters.branch_nums_to_merge))
            if self.Parameters.component_name:
                results.append(self._check_component_name(self.Parameters.component_name))
        elif self.action_mode == MergeMode.CUSTOM_MODE:
            if self.Parameters.revision_to_checkout:
                results.append(merge_helper.check_revs_correctness([str(self.Parameters.revision_to_checkout)]))
        else:
            results.append(rm_core.Error("You should choose at least one mode!"))
        return self._check_results_list(results)

    def _prepare_components_and_branch_num(self, revs):
        commit_info = self._get_commit_info_for_path(
            revs, svn_helper.TRUNK_PATH, rm_const.ActionType.MERGE
        )
        merge_action_items = []
        components_and_last_branches = {}

        if self.action_mode == MergeMode.RELEASE_MACHINE_MODE:
            merge_action_items, components_and_last_branches = self.get_components_and_branch_nums(
                commit_info,
                self.Parameters.component_name,
            )

            # TODO: Replace it with normal notifications to many components
            action_names = " ".join([action_item.name for action_item in merge_action_items])
            setattr(self.Context, rm_notify.COMPONENT_NAMES, action_names)
            self.Context.save()
        else:
            c_info = None
            custom_path = commit.check_or_complete_custom_path(self.Parameters.branch_path)
            merge_types = [svn_helper.MergeCustomPath(custom_path, getattr(self.ramdrive, "path", None))]
            merge_action_items.append(
                rm_const.ActionItem(MergeMode.CUSTOM_MODE, c_info, merge_types)
            )
        return merge_action_items, components_and_last_branches, commit_info

    def _fill_merge_results_ctx(self, merge_result):  # type: ('MergeToStable', svn_helper.MergeResult) -> None

        setattr(self.Context, MERGE_RESULTS_CONTEXT_KEY, {})
        component_2_results = collections.defaultdict(lambda: {_MergeResultStatus.SUCCESS: [], _MergeResultStatus.FAIL: []})

        for success in merge_result.successes:

            if not success.c_info:
                logging.warning("No c_info in merge_result success item %s", success)
                continue

            component_2_results[success.c_info.name][_MergeResultStatus.SUCCESS].append(success.branch_num)

        for failure in merge_result.failures:

            if not failure.c_info:
                logging.warning("No c_info in merge_result failure item %s", failure)
                continue

            component_2_results[failure.c_info.name][_MergeResultStatus.FAIL].append(failure.branch_num)

        setattr(self.Context, MERGE_RESULTS_CONTEXT_KEY, component_2_results)
        self.Context.save()

    def _on_execute(self, revs, commit_info, merge_action_items, components_and_last_branches):
        if not self.Parameters.need_check:
            # We don't need to run precommit checks before merge
            common_path = svn_helper.get_common_path(revs, sdk2.svn.Arcadia.ARCADIA_BASE_URL, merge_from_trunk=True)
            merge_result = self._do_action(merge_action_items, revs, commit_info, common_path)
        else:
            self.check_review_creation_possibility(action_items=merge_action_items)
            merge_result = self._create_and_wait_reviews(merge_action_items, revs, commit_info)

        self._fill_merge_results_ctx(merge_result)

        self.Context.merged_revs = merge_result.result_revs
        if merge_result.successes and self.Parameters.do_commit:
            msg_part = u"merged OK into branches: {}\n".format(
                ", ".join([path.short for path in merge_result.successes])
            )
            if not merge_result.is_ok():
                msg_part += u"Merge failed on branches: {}\n".format(
                    ", ".join([path.short for path in merge_result.failures]),
                )
            self._send_notifications_for_succeed_action(merge_action_items, revs, msg_part, commit_info)
        self.add_info_for_conflicts()
        if not components_and_last_branches:
            eh.ensure(merge_result.is_ok(), "Merge to some branches failed, see logs for details")
        else:
            # Check that commits are merged correctly in components with certain branches
            for failure in merge_result.failures:
                if failure.c_info.name not in components_and_last_branches.keys():
                    eh.check_failed("Merge to some branches failed, see logs for details")
            # Check that commits are merged correctly in last branches of components with uncertain branches
            for component_name, last_branch_num in components_and_last_branches.items():
                for failure in merge_result.failures:
                    if failure.c_info.name == component_name and failure.branch_num == last_branch_num:
                        eh.check_failed("Merge to some branches failed, see logs for details")

    def _create_and_wait_reviews(self, merge_action_items, revs, commit_info):
        with self.memoize_stage.review_creation(commit_on_entrance=False):
            # We should create reviews firstly
            reviews_results = list(review_helper.create_reviews(
                task=self,
                action_items=merge_action_items,
                revs=revs,
                commit_info=commit_info,
                base_url=sdk2.svn.Arcadia.ARCADIA_BASE_URL,
                do_commit=self.Parameters.do_commit,
                merge_from_trunk=True,
            ))
            self._present_reviews_results(reviews_results=reviews_results)
            if self._check_reviews_results(reviews_results=reviews_results):
                self.Context.save()
                # Sleep for 30 minutes to wait tests statuses
                raise sdk2.WaitTime(30 * 60)
            else:
                eh.check_failed("Reviews for some merges haven't been created")
        with self.memoize_stage.review_checking(commit_on_entrance=False):
            # We have waited for all tests to finish, now we need to check their statuses
            merge_result = self._check_all_reviews(
                action_items=merge_action_items,
                do_commit=self.Parameters.do_commit,
                revs=revs,
                commit_info=commit_info,
            )
        return merge_result

    @sdk2.header()
    def header(self):
        header_parts = super(MergeToStable, self).header()
        header_parts.append('Merged revisions:')
        for revision in self.Context.merged_revs:
            if revision:
                header_parts.append(lb.revision_link(revision))
        return "</br>\n".join(header_parts)

    def _on_success_commit(self, merge_type, committed_revision, revs):

        if self.action_mode != MergeMode.RELEASE_MACHINE_MODE:
            return

        try:
            from release_machine.common_proto import events_pb2

            event_time_utc_iso = tu.datetime_utc_iso()

            event_data = events_pb2.EventData(
                general_data=events_pb2.EventGeneralData(
                    hash=six.text_type(hash((
                        merge_type.c_info.name,
                        "MergeCommit",
                        0,  # Fixme (ilyaturuntaev) RMDEV-1949
                        committed_revision or self.id,  # if empty changeset we do not want this to be deduplicated
                        merge_type.name,
                        "TASK_SUCCESS",
                    ))),
                    component_name=merge_type.c_info.name,
                    referrer=u"sandbox_task:{}".format(self.id),
                ),
                task_data=self._get_rm_proto_event_task_data(
                    event_time_utc_iso=event_time_utc_iso,
                    status='SUCCESS',
                ),
                merge_commit_data=self._get_merge_commit_event_data(events_pb2, committed_revision, merge_type.name)
            )

            self.set_proto_event(event_data)

            logging.debug("Constructed event %s", event_data)
            events_helper.post_proto_events((event_data,))
            return
        except Exception as e:
            eh.log_exception("Unable to build proto event", e)

    def _on_merge_failed(self, fail_status='FAILURE'):
        if self.action_mode != MergeMode.RELEASE_MACHINE_MODE:
            return
        component_2_results = getattr(self.Context, MERGE_RESULTS_CONTEXT_KEY, {})
        if not component_2_results:
            self.set_info("Unable to send event: component scope cannot be detected")
        for component_name in component_2_results:

            try:
                from release_machine.common_proto import events_pb2

                event_time_utc_iso = tu.datetime_utc_iso()

                for scope_number in component_2_results[component_name][_MergeResultStatus.FAIL]:

                    event_data = events_pb2.EventData(
                        general_data=events_pb2.EventGeneralData(
                            hash=six.text_type(hash((
                                component_name,
                                "MergeCommit",
                                0,
                                None,
                                scope_number,
                                "TASK_FAILURE",
                            ))),
                            component_name=component_name,
                            referrer=u"sandbox_task:{}".format(self.id),
                        ),
                        task_data=self._get_rm_proto_event_task_data(
                            event_time_utc_iso=event_time_utc_iso,
                            status='FAILURE',
                        ),
                        merge_commit_data=self._get_merge_commit_event_data(events_pb2, 0, scope_number)
                    )

                    self.set_proto_event(event_data)

                    logging.debug("Constructed event %s", event_data)
                    events_helper.post_proto_events((event_data,))

            except Exception as e:
                eh.log_exception("Unable to build proto event", e)

        self.set_info('MergeCommit event with status {} sent'.format(self.status))

    def on_failure(self, prev_status):
        self._on_merge_failed()
        super(MergeToStable, self).on_failure(prev_status)

    def on_break(self, prev_status, status):
        self._on_merge_failed(fail_status=status)
        super(MergeToStable, self).on_break(prev_status, status)

    def _get_started_event_data(self, rm_proto_events, scope_number):

        extracted_commits_info = self._get_extracted_commits_info()

        return {
            "merge_started_data": rm_proto_events.MergeStartedData(
                scope_number=str(scope_number),
                origin_commit_id_list=[str(rev) for rev in extracted_commits_info.revs],
            )
        }

    def _get_merge_commit_event_data(self, rm_proto_events, branch_commit_id, scope_number):

        extracted_commits_info = self._get_extracted_commits_info()

        return rm_proto_events.MergeCommitData(
            branch_commit_id=str(branch_commit_id),
            scope_number=str(scope_number),
            origin_commit_id_list=[str(rev) for rev in extracted_commits_info.revs],
            origin_commit_authors=extracted_commits_info.authors,
            origin_commit_messages=extracted_commits_info.messages,  # Turn on after RM stable-184
        )
