# -*- coding: utf-8 -*-
import logging

import sandbox.projects.release_machine.components.all as rmc
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.input_params2 as rm_params
import sandbox.projects.release_machine.tasks.base_task as rm_bt
from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper

logger = logging.getLogger(__name__)


class MoveStartrekTicket(rm_bt.BaseReleaseMachineTask):
    """
        **Release-machine**

        Move startrek issue to target status
    """

    class Parameters(rm_params.ComponentName2):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        issue_key = sdk2.parameters.String(
            "Startrek issue",
            required=False,
            description="If empty, issue will be automatically found by release num",
        )
        release_num = sdk2.parameters.Integer("Release number", required=False)
        transition = sdk2.parameters.String("Transition to perform")
        with sdk2.parameters.Output:
            success = sdk2.parameters.Bool("Transition succeeded")

    class Requirements(task_env.StartrekRequirements):
        pass

    @staticmethod
    def perform_transition(issue_key, transition, st_client):
        issue = st_client.issues[issue_key]
        available_transitions = [tr.id for tr in issue.transitions.get_all()]
        if transition in available_transitions:
            logging.info("Execute transition '%s'.", transition)
            issue.transitions[transition].execute()
            return True
        else:
            logging.warning(
                "Transition '%s' is not allowed in '%s' status for '%s'",
                transition, issue.status.key, issue_key
            )
            logging.debug("Allowed transitions are %s", available_transitions)
            return False

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)

        c_info = rmc.COMPONENTS[self.Parameters.component_name]()

        token = rm_sec.get_rm_token(self)
        st_helper = STHelper(token)

        if not self.Parameters.issue_key and not self.Parameters.release_num:
            eh.check_failed("Invalid parameters combination, either release_num or issue_key should be specified")

        issue_key = (
            self.Parameters.issue_key
            or st_helper.find_ticket_by_release_number(self.Parameters.release_num, c_info).key
        )
        self.Parameters.success = self.perform_transition(issue_key, self.Parameters.transition, st_helper.st_client)
