import logging

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.release_machine import input_params2 as rm_params
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.tasks import base_task as rm_bt

logger = logging.getLogger(__name__)


PLACE_COMMENT_DESCRIPTION = "ticket_description"
PLACE_COMMENT_SEPARATE_COMMENT = "separate_comment"
UPDATE_MODE_APPEND = "append"
UPDATE_MODE_REPLACE = "replace"


class PostStartrekComment(rm_bt.BaseReleaseMachineTask):
    class Parameters(rm_params.BaseReleaseMachineParameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        issue_key = sdk2.parameters.String("Startrek ticket", required=True)
        notify_ticket_followers = sdk2.parameters.Bool("Notify ticket followers", default_value=False)
        place_comment = sdk2.parameters.String(
            "Place comment into",
            choices=[(s, s) for s in [PLACE_COMMENT_SEPARATE_COMMENT, PLACE_COMMENT_DESCRIPTION]],
            default_value="separate_comment",
        )
        comment_text = sdk2.parameters.String("Text", multiline=True)
        comment_marker = sdk2.parameters.String("Comment marker (used to find and update comment)", required=False)
        update_mode = sdk2.parameters.String(
            "Update comment mode",
            choices=[(s, s) for s in [UPDATE_MODE_APPEND, UPDATE_MODE_REPLACE]],
            default_value=UPDATE_MODE_REPLACE,
        )
        mention = sdk2.parameters.List(
            "Summon people (applicable to '{}' place_comment mode)".format(PLACE_COMMENT_SEPARATE_COMMENT),
            value_type=sdk2.parameters.String,
            default=[],
        )

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)

        if not self.Parameters.issue_key:
            eh.check_failed("Issue key not specified!")

        import startrek_client

        st_client = startrek_client.Startrek(
            token=rm_sec.get_rm_token(self),
            useragent=rm_const.ROBOT_RELEASER_USER_NAME
        )
        issue = st_client.issues[self.Parameters.issue_key]
        if self.Parameters.place_comment == PLACE_COMMENT_DESCRIPTION:
            updated_description = self._get_new_text(issue.description)
            issue.update(
                description=updated_description,
                params={"notify": self.Parameters.notify_ticket_followers}
            )
        elif self.Parameters.place_comment == PLACE_COMMENT_SEPARATE_COMMENT:
            old_comment = self._get_comment(issue)
            if not old_comment:
                issue.comments.create(
                    text=str(self.Parameters.comment_text),
                    params={
                        "notify": self.Parameters.notify_ticket_followers,
                    },
                    summonees=self.Parameters.mention,
                )
            else:
                updated_comment = self._get_new_text(old_comment.text)
                old_comment.update(
                    text=updated_comment,
                    params={
                        "notify": self.Parameters.notify_ticket_followers,
                    },
                    summonees=self.Parameters.mention,
                )
        else:
            eh.check_failed("Unknown place_comment mode: {}".format(self.Parameters.place_comment))

    def _get_new_text(self, old_text):
        old_text = old_text or ""
        if self.Parameters.update_mode == UPDATE_MODE_APPEND:
            logger.info("Appending text ...")
            return "{}\n{}".format(old_text, self.Parameters.comment_text)
        elif self.Parameters.update_mode == UPDATE_MODE_REPLACE:
            logger.info("Replacing text by marker '%s' ...", self.Parameters.comment_marker)
            parts = old_text.strip().split(self.Parameters.comment_marker)
            if len(parts) > 1:
                logger.info("Replacing\n%s\nwith\n%s", parts[1], self.Parameters.comment_text)
                parts[1] = self.Parameters.comment_text
            else:
                logger.info("Marker not found, appending text.")
                parts.extend(["\n", self.Parameters.comment_text])
            return self.Parameters.comment_marker.join(parts)
        else:
            eh.check_failed("Unknown update mode: {}".format(self.Parameters.update_mode))

    def _get_comment(self, issue):
        if not self.Parameters.comment_marker:
            logger.info("Empty comment marker")
            return
        for comment in issue.comments.get_all():
            if self.Parameters.comment_marker in comment.text:
                logger.info("Comment found: %s", comment)
                return comment
        logger.info("Cannot find comment with marker `%s` in issue `%s`", self.Parameters.comment_marker, issue)
