# -*- coding: utf-8 -*-

import logging
import re
import six
import typing  # noqa

from sandbox import sdk2

from sandbox.projects.release_machine.tasks import base_task as rm_bt
from sandbox.projects.release_machine import core as rm_core
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import task_env
from sandbox.projects.release_machine import input_params2 as rm_params
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.components.configs import all as all_cfg

from sandbox.projects.common.vcs import arc
from sandbox.projects.common import binary_task
from sandbox.projects.common import decorators
from sandbox.projects.common import error_handlers
from sandbox.projects.common import link_builder

from sandbox.projects.release_machine.tasks.PrepareCiEnvironment import component_env_processor


class PrepareCiEnvironment(rm_bt.BaseReleaseMachineTask):

    _component_names_to_process = None

    class Requirements(task_env.TinyRequirements):
        pass

    class Context(rm_bt.BaseReleaseMachineTask.Context):
        current_component_name = None
        components_considered = 0
        component_count = 0
        arc_log_latest = None
        results = {}
        changed_count = 0
        changed_components = list()

    class Parameters(rm_params.DefaultReleaseMachineParameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)

        component_name_filter = sdk2.parameters.String(
            "Component name filter",
            description="A regular expression, which, when present, "
                        "limits the components to only those that match it",
        )

        component_name_forced_ignore = sdk2.parameters.List(
            "The components that should be ignored no matter what",
            description="The components listed here will not be processed at all. This parameters is stronger"
                        "than component_name_filter.",
        )

        create_review = sdk2.parameters.Bool(
            "Create review",
            description="Create a review with a.yaml update",
            default_value=False,
        )

    @sdk2.header()
    def header(self):

        if not self.Context.component_count:
            return "<div>Initialization...</div>"

        percent_done = float(self.Context.components_considered) / self.Context.component_count * 100

        if percent_done == 100:
            return "<div>Component processing finished</div>"

        percent_done_int = int(percent_done)
        progress = "▮" * percent_done_int + "▯" * (100 - percent_done_int)

        return """
        <div>
            <div>
                Current component: <a href="{component_link}">{component_name}</a>
            </div>
            <div>
                {progress}
            </div>
        </div>
        """.format(
            component_link=self.current_component_link,
            component_name=self.current_component_name,
            progress=progress,
        )

    @sdk2.report(title="Report")
    def get_report_html(self):

        changed_components_links = ", ".join(
            [
                '<a href="https://rm.z.yandex-team.ru/component/{component_name}/manage2">{component_name}</a>'.format(
                    component_name=component_name,
                )
                for component_name in self.Context.changed_components
            ]
        )

        return """
        <div>
            <div>
                <div>
                    <strong>Components total:</strong>&nbsp;{component_count}
                </div>
            </div>
            <div>
                <div>
                    <h2>Changed Components</h2>
                </div>
                <div>
                    {changed_components_links}
                </div>
            </div>
        </div>
        """.format(
            component_count=self.Context.component_count,
            changed_components_links=changed_components_links,
        )

    @decorators.memoized_property
    def arc_client(self):
        return arc.Arc(arc_oauth_token=self.common_release_token)

    @decorators.memoized_property
    def common_release_token(self):
        return rm_sec.get_rm_token(self)

    @property
    def local_feature_branch_name_prefix(self):
        return "{task_type}-{task_id}".format(
            task_type=self.type,
            task_id=self.id,
        )

    @property
    def upstream_branch_prefix(self):
        return "users/{}/{}".format(
            rm_const.ROBOT_RELEASER_USER_NAME,
            self.local_feature_branch_name_prefix,
        )

    @property
    def current_component_name(self):
        return self.Context.current_component_name

    @current_component_name.setter
    def current_component_name(self, value):
        logging.info("Next component: %s", value)
        self.Context.current_component_name = value
        self.Context.components_considered += 1
        self.Context.save()

    @property
    def arc_log_latest(self):
        return self.Context.arc_log_latest

    @arc_log_latest.setter
    def arc_log_latest(self, value):
        self.Context.arc_log_latest = value
        self.Context.save()

    @property
    def current_component_link(self):
        return link_builder.rm_ui_link(self.current_component_name)

    @property
    def component_names_to_process(self):
        if not self._component_names_to_process:

            name_filter_re = re.compile(self.Parameters.component_name_filter or ".*", re.IGNORECASE)
            self._component_names_to_process = [
                component_name
                for component_name in filter(name_filter_re.match, all_cfg.get_all_ci_names())
                if component_name not in self.Parameters.component_name_forced_ignore
            ]

            self.Context.component_count = len(self._component_names_to_process)
            self.Context.save()

            self.set_info("{} components to process".format(self.Context.component_count))
            logging.info("Going to process the following components: %s", self._component_names_to_process)
            logging.info("Total: %s", self.Context.component_count)

        return self._component_names_to_process

    def get_update_ayaml_commit_message(self, component_name):
        return "Updating Release Machine {component_name} a.yaml [sb:{task_id}] [RM Config] [RMDEV-2534]".format(
            component_name=component_name,
            task_id=self.id,
        )

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)

        results = []

        with self.arc_client.mount_path(None, None, fetch_all=False, extra_params=["--vfs-version", "2"]) as arcadia_mp:

            self._checkout_trunk(arcadia_mp)

            for component_name in self.component_names_to_process:

                logging.info("Consider component %s", component_name)

                try:

                    self.current_component_name = component_name
                    self._process_current_component(arcadia_mp)

                except Exception as e:

                    error_handlers.log_exception(
                        "Unable to process {} correctly".format(component_name),
                        e,
                    )

                    results.append(rm_core.Error("Unable to process {}".format(component_name)))

                else:

                    results.append(rm_core.Ok("{} processed successfully".format(component_name)))

                create_review_result = self._create_review(arcadia_mp, component_name)
                results.append(create_review_result)

            self._handle_results(results)

    @decorators.log_start_and_finish()
    def _process_current_component(self, mount_point):
        # type: (str) -> None

        logging.info("Processing %s", self.current_component_name)

        processor = component_env_processor.ComponentEnvProcessor(
            component_name=self.current_component_name,
            arcadia_root=mount_point,
            token=self.common_release_token,
            dry_run=self.Parameters.debug_mode,
        )

        logging.info("New component processor initialized: %s", processor)

        build_ayaml_result = processor.build_a_yaml()
        save_ayaml_result = processor.save_a_yaml()
        request_roles_result = processor.request_roles()

        role_request_link = request_roles_result.get_context_field("request_link")

        self.set_info(
            """
            <a href="{component_link}">{component_name}</a>:<br/>
            - build a.yaml: {build_ayaml_result}<br/>
            - save a.yaml: {save_ayaml_result}<br/>
            - <a href="{role_request_link}" target="_blank">request IDM roles</a>: {request_roles_result}<br/>
            """.format(
                component_name=self.current_component_name,
                component_link=self.current_component_link,
                build_ayaml_result=build_ayaml_result,
                save_ayaml_result=save_ayaml_result,
                request_roles_result=request_roles_result,
                role_request_link=role_request_link,
            ),
            do_escape=False,
        )

    @decorators.log_start_and_finish()
    def _checkout_trunk(self, mount_point):
        # type: (str) -> None
        """
        arc checkout trunk

        :param mount_point: Arcadia mount point
        """
        self.arc_client.checkout(mount_point, branch="trunk")
        self.arc_log_latest = self.arc_client.log(mount_point, max_count=1, as_dict=True)

    @decorators.log_start_and_finish()
    def _create_review(self, mount_point, component_name):
        # type: (str, str) -> rm_core.Result

        status = self.arc_client.status(mount_point, as_dict=True)

        status = status.get("status", {})

        changed_paths_count = len(status.get("changed", [])) + len(status.get("untracked", []))

        if changed_paths_count == 0:
            return rm_core.Ok("{}: Changeset is empty. No need to commit anything".format(component_name))

        if changed_paths_count > self.Context.changed_count:
            self.Context.changed_count += 1
            self.Context.changed_components.append(component_name)
            self.Context.save()

        if self.Parameters.debug_mode:
            logging.info("The task is running in debug mode => review is not going to be created")
            return rm_core.Ok("debug_mode = True => not committing anything")

        if not self.Parameters.create_review:
            logging.info("Launch parameters demand that we do not create any PR")
            return rm_core.Ok("create_review = False => not committing anything")

        self.arc_client.checkout(
            mount_point,
            branch="{}-{}".format(
                self.local_feature_branch_name_prefix,
                component_name,
            ),
            create_branch=True,
        )

        self.arc_client.add(
            mount_point,
            all_changes=True,
        )

        update_ayaml_commit_message = self.get_update_ayaml_commit_message(component_name)

        self.arc_client.commit(
            mount_point,
            message=update_ayaml_commit_message,
            all_changes=True,
        )

        self.arc_client.push(
            mount_point,
            force=True,
            upstream="{}-{}".format(
                self.upstream_branch_prefix,
                component_name,
            ),
        )

        publish_result = self.arc_client.pr_create(
            mount_point,
            message=update_ayaml_commit_message,
            publish=True,
            auto=True,
        )

        self.arc_client.checkout(
            mount_point,
            branch="trunk",
        )

        self.set_smarter_info(publish_result)

        return rm_core.Ok("Review created successfully")

    def _handle_results(self, results):
        # type: (typing.List[rm_core.Result]) -> None

        logging.info("Results: \n{}".format("\n".join([str(r) for r in results])))

        failures = [r for r in results if not r.ok]

        if failures:
            error_handlers.check_failed(
                "Some errors occurred: \n{}".format(
                    "\n".join([str(f) for f in failures]),
                ),
            )

    def add_result(self, component_name, value):
        self.Context.results[component_name] = value
        self.Context.save()

    def set_smarter_info(self, info):
        info = re.sub(
            r"(http[s]?)://([^' ]+)",
            r'<a href="\1://\2" target="_blank">\1://\2</a>',
            six.text_type(info),
        )

        self.set_info(six.text_type(info), do_escape=False)
