# -*- coding: utf-8 -*-

import logging
import os
import requests
import typing  # noqa

from sandbox.projects.release_machine.components import all as rm_components
from sandbox.projects.release_machine import core as rm_core
from sandbox.projects.common import decorators


class ComponentEnvProcessor:

    def __init__(self, component_name, arcadia_root, token, dry_run=False):
        self._component_name = component_name
        self._arcadia_root = arcadia_root
        self._token = token
        self._c_info = None
        self._role_request_url = None
        self._dry_run = dry_run

        self._builder = None

    @property
    def a_yaml_builder(self):
        if not self._builder:

            from release_machine.config_converter import a_yaml_builder

            self._builder = a_yaml_builder.get_builder(
                c_info=self.c_info,
                arcadia_root=self.arcadia_root,
                token=self._token,
            )
            logging.info("a.yaml builder initialized: %s", self.a_yaml_builder)

        return self._builder

    @property
    def component_name(self):
        return self._component_name

    @property
    def arcadia_root(self):
        return self._arcadia_root

    @property
    def c_info(self):
        if not self._c_info:
            logging.info("Going to build c_info for %s", self.component_name)
            self._c_info = rm_components.get_component(self.component_name)
        return self._c_info

    @decorators.log_start_and_finish(log_level=logging.INFO, log_result=True)
    def build_a_yaml(self):
        try:
            self.a_yaml_builder.build()
        except KeyError:
            logging.exception("Failed to build a.yaml for %s", self.component_name)
            return rm_core.Error("Failed to build a.yaml for {}".format(self.component_name))

        return rm_core.Ok("a.yaml successfully built for {}".format(self.component_name))

    @decorators.log_start_and_finish(log_level=logging.INFO)
    def save_a_yaml(self):
        try:
            self.a_yaml_builder.save()
        except (ValueError, OSError):
            logging.exception("Cannot save a.yaml for %s", self.component_name)
            return rm_core.Error("a.yaml cannot be saved for {}".format(self.component_name))

        return rm_core.Ok("a.yaml successfully saved for {}".format(self.component_name))

    @decorators.log_start_and_finish(log_level=logging.INFO)
    def request_roles(self):

        from release_machine.rm_tool.tools.request_roles import arc_roles, roles

        user = self.c_info.get_responsible_for_component()
        robot = self.c_info.robot
        arc_branch_path = os.path.join(self.c_info.svn_cfg__arc_branch_dir, "")
        arc_tag_path = os.path.join(self.c_info.svn_cfg__arc_tag_dir, "")

        logging.info(
            "Target:\n"
            "  prefix admin:         {user}\n"
            "  robot (create + f-f): {robot}\n"
            "  branch path:          {arc_branch_path}\n"
            "  tag path:             {arc_tag_path}".format(
                user=user,
                robot=robot,
                arc_branch_path=arc_branch_path,
                arc_tag_path=arc_tag_path,
            ),
        )

        simulate_request_list = arc_roles.construct_user_requests(
            user=user,
            robot=robot,
            branch_path=arc_branch_path,
            tag_path=arc_tag_path,
            simulate=True,
        )

        role_client = roles.IdmRoleClient(token=self._token)

        actual_request_list = []

        for role_request in simulate_request_list:

            try:

                role_request_response_json = role_client.request_role(role_request)

            except requests.HTTPError as http_error:

                if http_error.response.status_code == 409:
                    response_json = http_error.response.json()
                    logging.info(" - %s", response_json['message'])
                    continue

                return rm_core.Error("Cannot request IDM roles for {} - http error".format(self.component_name))

            except requests.ConnectionError:
                logging.exception("Unable to request roles due to network errors")
                return rm_core.Error(
                    "Cannot request IDM roles for {} - connection error".format(self.component_name),
                )

            logging.info(" - %s", role_request_response_json)

            actual_request_list.append(role_request._replace(simulate=False))

        logging.info("Going to request the following roles:")

        for role_request in actual_request_list:
            logging.info(
                " - %s - %s - %s",
                role_request.user,
                role_request.path,
                role_request.fields_data['ref-prefix'],
            )

        if self._dry_run:
            logging.info("This is a dry run thus no roles are actually going to be requested")
            return rm_core.Ok("Role request: dry run (no roles requested)")

        role_ids = []

        for role_request in actual_request_list:
            role_request_response_json = role_client.request_role(role_request)
            role_id = role_request_response_json.get('id')
            role_ids.append(role_id)
            logging.info(" - %s", role_id)

        if not role_ids:
            logging.warning("No roles requested")
            return rm_core.Ok("No additional IDM roles required for {}".format(self.component_name))
        else:

            link = self.build_role_request_url(role_ids)

            logging.info(
                "\n"
                "Role Requests: {}\n"
                "".format(link)
            )

            return rm_core.Ok(
                "IDM roles requested for {}".format(self.component_name),
                context={
                    "request_link": link,
                },
            )

    def build_role_request_url(self, role_ids):

        base_url = "https://idm.yandex-team.ru/system/arc-vcs/roles"
        sep = "#"

        role_ids_str = "%5C,".join([str(rid) for rid in role_ids])

        self._role_request_url = "{base_url}{sep}f-role-id={role_ids_str}".format(
            base_url=base_url,
            sep=sep,
            role_ids_str=role_ids_str,
        )

        return self._role_request_url

    def __str__(self):
        return "<{} {} {}>".format(self.__class__.__name__, self.component_name, self.arcadia_root)

    def __repr__(self):
        return str(self)
