# -*- coding: utf-8 -*-
import logging
import os

import sandbox.projects.release_machine.components.all as rmc
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.notify_helper as nh
import sandbox.projects.release_machine.input_params2 as rm_params
import sandbox.projects.release_machine.tasks.base_task as rm_bt
import sandbox.sdk2 as sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.release_machine import rm_notify
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.components.configs.all import get_all_branched_names
from sandbox.projects.common.testenv_client import TEClient
from sandbox.sdk2.vcs.svn import Arcadia


@rm_notify.notify2()
class PrepareRmComponentEnvironment(rm_bt.BaseReleaseMachineTask):
    """
        **Release-machine**

        Task to create new trunk database in Testenv with tests and owners.
    """

    class Requirements(task_env.TinyRequirements):
        disk_space = 512  # 512 Mb

    class Parameters(rm_params.BaseReleaseMachineParameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        kill_timeout = 20 * 60  # 20 min
        with sdk2.parameters.String("Component name", default="", required=True) as component_name:
            rm_params.set_choices(component_name, get_all_branched_names())
        prepare_testenv = sdk2.parameters.Bool("Prepare testenv databases", default_value=True)
        new_dir_for_branches = sdk2.parameters.String(
            "Path to be added in arcadia for branches. "
            "For example, branches/maps/geo/, where directory branches/maps "
            "already exists. Robot can't add new dir in branches because of "
            "lack of rights.",
        )

    class Context(rm_bt.BaseReleaseMachineTask.Context):
        type = "PREPARE_RM_COMPONENT_ENVIRONMENT"

    def create_new_branches_dir(self):
        branches_dir = self.Parameters.new_dir_for_branches
        branches_dir = branches_dir.strip('/')
        dirs = branches_dir.split('/')
        if dirs[0] != 'branches':
            eh.check_failed(
                "You've entered incorrect directory for branches {dir}".format(dir=self.Parameters.new_dir_for_branches)
            )
        if len(dirs) < 3:
            eh.check_failed(
                "You've entered wrong path {dir}. It is too short".format(dir=self.Parameters.new_dir_for_branches)
            )
        parent_dir_path = os.path.join('arcadia:/arc', dirs[:2])
        if not Arcadia.check(parent_dir_path):
            eh.check_failed("Path {path} doesn't exist, failed".format(path=parent_dir_path))
        full_dir_path = os.path.join("arcadia:/arc", dirs)
        try:
            Arcadia.mkdir(url=full_dir_path, parents=True)
            self.set_info("Path for branches {path} was successfully created".format(path=full_dir_path))
        except Exception as exc:
            msg = "Can't create path {}".format(full_dir_path)
            eh.log_exception(msg, exc)
            eh.check_failed(msg)

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)
        rmc.check_component_name(self.Parameters.component_name)
        if self.Parameters.prepare_testenv:
            c_info = rmc.COMPONENTS[self.Parameters.component_name]()
            te_helper = TEClient(rm_sec.get_rm_token(self))
            trunk_db_name = c_info.testenv_cfg__trunk_db
            owners = list(set(c_info.testenv_cfg__testenv_db_owners) | {c_info.get_responsible_for_component()})
            if not te_helper.testenv_database_exists(trunk_db_name):

                svn_revision = te_helper.get_last_svn_revisions()[0]['revision']

                te_helper.create_db_and_add_tests(
                    db_name=trunk_db_name,
                    owners=owners,
                    tests=["_LOG_MERGE__{}".format(c_info.name.upper())],
                    frequencies=[],
                    svn_revision=svn_revision,
                    task_owner=c_info.testenv_cfg__trunk_task_owner,
                    release_machine_shard=True,  # create new projects in RM shard (TESTENV-3978)
                    use_api_client=True,  # enable RMDEV-1638 for all
                    login=self.author,
                )

            te_helper.set_db_option(
                database_name=trunk_db_name,
                option_name="notifications_on_task_warnings",
                enable=c_info.testenv_cfg__enable_notifications_on_task_warnings,
            )
            te_helper.set_db_option(
                database_name=trunk_db_name,
                option_name="notification_on_db_events",
                enable=c_info.testenv_cfg__enable_notifications_on_db_events,
            )
            if c_info.testenv_cfg__enable_all_notifications_for_db_owners:
                for owner in owners:
                    te_helper.switch_owner_param_in_db(
                        database_name=trunk_db_name,
                        login=owner,
                        param_name="notify_database_warnings",
                    )
                    te_helper.switch_owner_param_in_db(
                        database_name=trunk_db_name,
                        login=owner,
                        param_name="notify_database_events",
                    )
            message = "Successfully added trunk database {db} with tests for component {component}".format(
                db=trunk_db_name,
                component=c_info.name,
            )
            logging.info(message)
            nh.telegram(self, message, chat_ids=c_info.notify_cfg__tg__chats)
        if self.Parameters.new_dir_for_branches:
            self.create_new_branches_dir()
