# -*- coding: utf-8 -*-

import logging

import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.input_params2 as rm_params
import sandbox.projects.release_machine.rm_notify as rm_notify
import sandbox.projects.release_machine.security as rm_sec
import sandbox.projects.release_machine.tasks.base_task as rm_bt
import sandbox.sdk2 as sdk2
from sandbox.common.rest import Client as SandboxClient
from sandbox.projects.common import binary_task
from sandbox.projects.common.nanny.client import NannyClient
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.components import configs as rm_config
from sandbox.projects.release_machine.core import const as rm_const


@rm_notify.notify2()
class PushServicesConfiguration(rm_bt.BaseReleaseMachineTask):
    """ Deploy resource to list of services """
    class Requirements(task_env.TinyRequirements):
        disk_space = 512  # Mb

    class Parameters(rm_params.ComponentNameResources):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        do_deploy = sdk2.parameters.Bool("Do deploy (otherwise just show diff)", default=False)
        with sdk2.parameters.String("Stage", default=rm_const.ReleaseStatus.stable) as stage:
            rm_params.set_choices(stage, [x for x in dir(rm_const.ReleaseStatus) if not x.startswith("__")])

    class Context(rm_bt.BaseReleaseMachineTask.Context):
        waited_times = 0

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)

        self._sb_client = SandboxClient()
        self._nn_client = NannyClient(api_url=rm_const.Urls.NANNY_BASE_URL, oauth_token=rm_sec.get_rm_token(self))
        self._c_info = rmc.get_component(self.Parameters.component_name)

        for name, resource_id in self.Parameters.component_resources.items():
            self._deploy_one_resource(name, resource_id)

    def _deploy_one_resource(self, name, resource_id):
        sb_resource = self._sb_client.resource[resource_id].read()
        sb_task = self._sb_client.task[str(sb_resource["task"]["id"])].read()
        logging.info("Resource info: resource_type {}, task_id {}, task_type {}".format(
            sb_resource["type"], sb_task["id"], sb_task["type"]
        ))

        service_ids = []
        for res_info in self._filter_resources(self._c_info, {
            "resource_type": sb_resource["type"],
            "name": name,
            "deploy": ""
        }):
            if res_info.attributes and not self._check_atributes(res_info.attributes, sb_resource["attributes"]):
                continue
            for deploy_info in res_info.deploy:
                if not isinstance(deploy_info, rm_config.DeployServicesInfo):
                    self.set_info("Deploy configuration has to be instance of DeployServicesInfo")
                    return
                if deploy_info.level == self.Parameters.stage:
                    service_ids = self._c_info.get_deploy_services(deploy_info, self._nn_client)
        if not service_ids:
            self.set_info("Deploy configuration for {} doesnt exist".format(sb_resource["type"]))
        for service_id in service_ids:
            self._push_service_configuration(service_id, sb_resource, sb_task)

    def _push_service_configuration(self, service_id, sb_resource, sb_task):
        resources = self._nn_client.get_service_resources(service_id)
        sandbox_files = resources["content"]["sandbox_files"]
        modified = False
        for i, sandbox_file in enumerate(sandbox_files):
            if sandbox_file["resource_type"] == sb_resource["type"]:
                if modified:
                    self.set_info("{}: more then one {} resource detected".format(service_id, sb_resource["type"]))
                if sandbox_file["resource_id"] == sb_resource["id"]:
                    self.set_info("{}: {} resource is already deployed".format(service_id, sb_resource["id"]))
                    continue
                old_resource = self._sb_client.resource[sandbox_files[i]["resource_id"]].read()
                self.set_info("{}:\nid: {}\n{}\n>>>\nid: {}\n{}".format(
                    service_id,
                    old_resource["id"], old_resource["description"],
                    sb_resource["id"], sb_resource["description"]
                ))
                if self.Parameters.do_deploy:
                    sandbox_files[i]["resource_id"], sandbox_files[i]["task_id"], sandbox_files[i]["task_type"] = \
                        str(sb_resource["id"]), str(sb_task["id"]), sb_task["type"]
                    self._nn_client.update_service_resources(service_id, {
                        "content": resources["content"],
                        "comment": "Change configuration from RM",
                        "snapshot_id": resources["snapshot_id"]
                    })
                modified = True
        if not modified:
            self.set_info("Service {} doesn't have {} resources".format(service_id, sb_resource["type"]))

    @staticmethod
    def _check_attributes(attributes, sb_attributes):
        for k, v in attributes.items():
            if k not in sb_attributes or sb_attributes[k] != v:
                return False
        return True

    @staticmethod
    def _filter_resources(c_info, filters):
        for res_info in c_info.releases_cfg__resources_info:
            logging.info("filtering: {}".format(res_info))
            if all([v == getattr(res_info, k, None) or v == "" and getattr(res_info, k, None)
                    for k, v in filters.items()]):
                logging.info("take it")
                yield res_info
