#!/usr/bin/env python
# -*- coding: utf-8 -*-

import cgi
import copy
import logging
import json
import os
import random
import re
import string
import time
import urllib2
from datetime import timedelta
from collections import defaultdict
from collections import namedtuple

import jinja2
import sandbox.projects.release_machine.core.task_env as task_env

from sandbox.common import rest
from sandbox.common.types import resource
from sandbox.common.types.client import Tag
from sandbox.projects import GetRandomRequests as grr
from sandbox.projects.release_machine import input_params as rm_params
from sandbox.projects.release_machine.tasks import GetReportSimultaneousResponses as grsr
from sandbox.projects.release_machine.tasks.ScrapeRequests2 import parameters as sr_params
from sandbox.projects.release_machine.tasks.ScrapeRequests2 import static as ss
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects import resource_types
from sandbox.projects.common import apihelpers
from sandbox.projects.common import environments
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import utils
from sandbox.projects.common import ra2
from sandbox.projects.common.search import bugbanner
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.svn import Arcadia

MR_USER = "rearrange"
YT_TABLE_WITH_JSON_NAME = "table_with_json_data"
VISIBLE_QUERY_LENGTH = 100

TIMING_KEY = "execution_timing"
REQUESTS_TIMING = "requests_timing"
CONTEXT_DIFF_TIMING = "two_contexts_features_diff_timing"
SEPARATING_JSON_DUMP_TIMING = "separating_json_dump_timing"
TIMESTAMP_START = "timestamp_start"
TIMESTAMP_FINISH = "timestamp_finish"
TOTAL_TIME = "total_time"

REQUESTS_COUNT_FOR_DOMAIN = "requests_count_for_domain"
TESTS_STATUSES = "tests_statuses"
PREVIOUS_TESTS_STATUSES = "previous_tests_statuses"
TESTS_FINAL_STATUSES = "tests_final_statuses"
PREVIOUS_TESTS_FINAL_STATUSES = "previous_tests_final_statuses"

HTML_REPORT_RESOURCE_PROXY_URL = "html_report_resource_proxy_url"

TestsResult = namedtuple("TestsResult", ("factor_name", "test_type", "value", "threshold", "test_status"))


COLOR_SUCCESS = '#006633'
COLOR_FAILURE = '#CC0000'


class FeaturesConfig(sp.LastReleasedResource):
    name = "features_config"
    description = "File with proto config with required features for two_contexts_features_diff tool"
    resource_type = resource_types.REARRANGE_ACCEPTANCE_CONFIG
    required = False
    default_value = ""


class UseConfigFromArcadia(sp.SandboxBoolParameter):
    name = "use_config_from_arcadia"
    description = "Use config from Arcadia instead of resource"
    required = False
    default_value = False


class ArcadiaConfigPath(sp.SandboxStringParameter):
    name = "arcadia_config_path"
    description = "File path in Arcadia"
    required = False
    default_value = "/sandbox/projects/release_machine/tasks/RearrangeAcceptance2/config/RA_config"


class TimeToKill(sp.SandboxIntegerParameter):
    name = "time_to_kill"
    description = "Time to kill for GetReportSimultaneousResponses task in hours"
    required = True
    default_value = 5


class ProcessedReqsPercentThreshold(sp.SandboxFloatParameter):
    name = "processed_reqs_percent_threshold"
    description = "Required fraction of processed requests (for example, 0.75), FAILURE otherwise"
    required = False
    default_value = 0


class UseLatestParserBinary(sp.SandboxBoolParameter):
    name = "use_latest_parser_binary"
    description = "Use latest json-parser two_contexts_features_diff binary, instead of building it"
    default_value = False


class UseLatestConfigWithBlenderFeatures(sp.SandboxBoolParameter):
    name = "use_latest_config_with_blender_features"
    description = "Use latest config with blender features"
    default_value = False


class IgnorePassed(sp.SandboxBoolParameter):
    name = "ignore_passed"
    description = "Don't run tests that passed in previous run; uses PreviousTaskID value"
    default_value = False


class PreviousTaskID(sp.SandboxIntegerParameter):
    name = "previous_task_id"
    description = "Id of previous task, runned by TestAllTdiAndRearrange; 0 if this is the first run"
    default_value = 0


class SendNotificationToStartrek(sp.SandboxBoolParameter):
    name = "send_notification_to_startrek"
    description = "Whether to send a notification to Startrek or not"
    required = False
    default_value = True


class SendNotificationToStartrekOnTaskFailure(sp.SandboxBoolParameter):
    name = "send_notification_to_startrek_on_task_failure"
    description = "Whether to send a notification  to Startrek (write comment to the appropriate ticket) " \
                  "when task fails. You probably don't want to use it when starting RA2 as a subtask. " \
                  "Should be removed after RMDEV-492"
    required = False
    default_value = False


class RearrangeAcceptance2(bugbanner.BugBannerTask):
    """
    New version of RearrangeAcceptance task
    Get JSON dump for queries simultaneously,
    calculate different statistics for each requested field

    https://wiki.yandex-team.ru/jandekspoisk/kachestvopoiska/rearrange/rearrangeacceptancenew
    """
    type = "REARRANGE_ACCEPTANCE_2"
    input_parameters = (
        (
            FeaturesConfig,
            UseConfigFromArcadia,
            ArcadiaConfigPath,
            TimeToKill,
            ProcessedReqsPercentThreshold,
            sr_params.ParserBinaryParameter,
            UseLatestParserBinary,
            UseLatestConfigWithBlenderFeatures,
            IgnorePassed,
            PreviousTaskID,
            grr.SearchPropFilterParameter,
            grr.UseLatestRandomQueriesBinary,
        ) +
        grsr.FIRST_BETA_PARAMETERS +
        grsr.SECOND_BETA_PARAMETERS +
        grsr.GENERAL_PARAMETERS +
        grsr.QUERIES_INFO_PARAMETERS +
        ra2.PERSONAL_UIDS_PARAMETERS +
        (
            sr_params.YtServerParameter,
        ) +
        grsr.UNANSWERED_PARAMETERS +
        (
            SendNotificationToStartrekOnTaskFailure,
            SendNotificationToStartrek,
        ) +
        (
            sr_params.ScraperOverYtPoolParameter,
        ) +
        grsr.BATCHED_MODE_PARAMETERS +
        ra2.RELEASE_MACHINE_PARAMETERS
    )
    cores = 1
    required_ram = 40 * 1024
    execution_space = 40 * 1024  # 40 Gb
    client_tags = Tag.LINUX_PRECISE & task_env.TaskTags.startrek_client
    environment = (
        task_env.TaskRequirements.startrek_client,
        environments.PipEnvironment('yandex-yt', use_wheel=True),
        environments.PipEnvironment('yandex-yt-yson-bindings-skynet', use_wheel=True),
    )

    @property
    def footer(self):
        if not self.is_completed():
            return
        logging.debug("Setting footer...")
        result = []
        for domain, proxy_url in self.ctx.get(HTML_REPORT_RESOURCE_PROXY_URL, {}).iteritems():
            processed_reqs = self.ctx.get(REQUESTS_COUNT_FOR_DOMAIN, {}).get(domain)
            total_reqs = self.ctx.get(grsr.TOTAL_REQUEST_COUNT, -1) / len(self.ctx.get(HTML_REPORT_RESOURCE_PROXY_URL))
            if processed_reqs is None:
                processed_reqs = self.ctx.get(grsr.REQUEST_COUNT, -1)
                total_reqs = self.ctx.get(grsr.TOTAL_REQUEST_COUNT, -1)
            color = COLOR_SUCCESS if processed_reqs == total_reqs else COLOR_FAILURE
            final_test_status = self.ctx.get(TESTS_FINAL_STATUSES, {}).get(domain, "UNKNOWN")
            result.append({
                "helperName": "",
                "content": (
                    "<h4>Total number of processed requests for {}:".format(domain) +
                    "<span style='color:{color};'> {processed} / {total}</span>"
                    "<br><a href='{url}/index.html' target='_blank'>All tests results for {domain}</a>".format(
                        color=color,
                        processed=processed_reqs,
                        total=total_reqs,
                        url=proxy_url,
                        domain=domain,
                    ) +
                    "<br>Status: <span style='color:{color};'>{status_name}</span></h4>".format(
                        color=COLOR_SUCCESS if final_test_status == "PASSED" else COLOR_FAILURE,
                        status_name=final_test_status,
                    )
                )
            })
        return result

    @staticmethod
    def fill_template(template_file_name, result_file_name, result_dict, **additional_params):
        html_template = fu.read_file(template_file_name)

        jinja_index_template = jinja2.Template(html_template)
        html = jinja_index_template.render(result=result_dict, **additional_params)

        fu.write_file(result_file_name, html.encode("utf-8"))

    def separate_json_dumps_in_one_time(self, dir_names, env):
        json_dir_paths = dict()
        json_dir_name = "json_dumps"
        for domain, dir_name in dir_names.iteritems():
            json_dir_paths[domain] = os.path.join(".", dir_name, json_dir_name)
            if json_dir_name not in os.listdir(os.path.join(".", dir_name)):
                os.mkdir(json_dir_paths[domain])

        from yt.wrapper import YtClient
        from yt.wrapper import JsonFormat

        client = YtClient(env["DEF_MR_SERVER"], env["YT_TOKEN"])

        logging.info("Reading YT table with json dumps in one time: %s", self.ctx[YT_TABLE_WITH_JSON_NAME])
        self.ctx[REQUESTS_COUNT_FOR_DOMAIN] = {domain: 0 for domain in dir_names}
        for row in client.read_table(
                self.ctx[YT_TABLE_WITH_JSON_NAME],
                format=JsonFormat(attributes={"encode_utf8": False}),
        ):
            domain = row["domain"]
            if domain in json_dir_paths:
                self.ctx[REQUESTS_COUNT_FOR_DOMAIN][domain] += 1
                file_name = row["key"] + "_" + row["subkey"].split("\t")[-1]
                file_path = os.path.join(json_dir_paths[domain], file_name)
                fu.write_file(file_path, row["value"])
        for domain in self.ctx[REQUESTS_COUNT_FOR_DOMAIN]:
            self.ctx[REQUESTS_COUNT_FOR_DOMAIN][domain] /= 2

    def separate_json_dumps(self, dir_name, env, domain=None):
        json_dir_name = "json_dumps"
        json_dir_path = os.path.join(".", dir_name, json_dir_name)
        if json_dir_name not in os.listdir(os.path.join(".", dir_name)):
            os.mkdir(json_dir_path)

        from yt.wrapper import YtClient
        from yt.wrapper import JsonFormat

        client = YtClient(env["DEF_MR_SERVER"], env["YT_TOKEN"])

        logging.info("Reading YT table with json dumps: %s", self.ctx[YT_TABLE_WITH_JSON_NAME])

        for row in client.read_table(
                self.ctx[YT_TABLE_WITH_JSON_NAME],
                format=JsonFormat(attributes={"encode_utf8": False}),
        ):
            if not domain or "domain" not in row or row["domain"] == domain:
                file_name = row["key"] + "_" + row["subkey"].split("\t")[-1]
                file_path = os.path.join(json_dir_path, file_name)
                fu.write_file(file_path, row["value"])

    def _build_request_url(self, template_key, query, region, uid, domain):
        result = self.ctx[template_key].get(
            domain.lower(),
            self.ctx[template_key].get('*', '')
        ).format(text=query, lr=region)
        if uid:
            param = 'yandexuid={}'.format(uid)
            result += ('&' + param) if not result.endswith(('?', '&')) else param
        return result

    def create_test_response(self, test_name, dir_name, env, domain=None):
        test_file_name = test_name.replace("/", "_").replace("\\", "_")
        template_file_name = os.path.join(os.path.dirname(__file__), "templates", "test.html.jinja")
        tmp_file_name = os.path.join("tmp", test_file_name)
        file_name = os.path.join(dir_name, test_file_name + ".html")

        command = (
            "./two_contexts_features_diff sort_diff "
            "-i {input} -s {server} -f '{feature}' --domain {domain} > '{output}'"
        ).format(
            input=self.ctx[YT_TABLE_WITH_JSON_NAME] + "_features",
            output=tmp_file_name,
            server=utils.get_or_default(self.ctx, sr_params.YtServerParameter),
            feature=test_name,
            domain=domain,
        )

        logging.info("Create html report for test %s", test_name)
        process.run_process(
            [command], work_dir="./", timeout=7200, shell=True, check=True,
            log_prefix="sort_diff for {}".format(test_file_name),
            outputs_to_one_file=True,
            environment=env,
        )
        result = {
            "test": test_name.decode("utf-8"),
            "data": []
        }
        for line in open(tmp_file_name, "r"):
            line = line.decode("utf-8").strip("\n").rsplit("\t", 2)
            parts = line[0].split("\t")
            parts += [None] * (4 - len(parts))
            query_id, query, region, rest = parts

            uid = None
            if rest:
                uid = rest.split("\t")[0]

            visible_query = cgi.escape(query)
            if len(query) > VISIBLE_QUERY_LENGTH:
                visible_query = visible_query[:VISIBLE_QUERY_LENGTH] + "..."
            visible_query += "\t" + region + (("\t" + uid) if uid else "")

            quoted_query = urllib2.quote(query.encode("utf-8"))
            current = {
                "query": visible_query,
                "first": line[1],
                "second": line[2],
                "first_beta": self._build_request_url(
                    grsr.FIRST_BETA_TEMPLATE_DICT, quoted_query, region, uid, domain
                ),
                "second_beta": self._build_request_url(
                    grsr.SECOND_BETA_TEMPLATE_DICT, quoted_query, region, uid, domain
                ),
                "first_json": "json_dumps/" + query_id + "_first",
                "second_json": "json_dumps/" + query_id + "_second",
            }

            result["data"].append(current)

        self.fill_template(template_file_name, file_name, result)

    def create_response(self, input_test_result, env, enough_requests, dir_name="html_report", domain=None):
        test_result = []
        tests = set()
        self.ctx[TESTS_STATUSES][domain] = copy.deepcopy(self.ctx[PREVIOUS_TESTS_STATUSES].get(domain, dict()))
        total_tests_count = 0
        prev_passed_tests_count = 0
        for feature_name, feature_name_dict in self.ctx[TESTS_STATUSES][domain].iteritems():
            for field_name, status in feature_name_dict.iteritems():
                total_tests_count += 1
                if status == "PASSED":
                    self.ctx[TESTS_STATUSES][domain][feature_name][field_name] = "PREV_PASSED"
                if self.ctx[TESTS_STATUSES][domain][feature_name][field_name] == "PREV_PASSED":
                    prev_passed_tests_count += 1

        for test in input_test_result:
            current = {}

            overall_test_status = self.ctx[TESTS_STATUSES][domain].get(test.factor_name, dict()).get(test.test_type)
            current_test_status = test.test_status

            if overall_test_status == "PREV_PASSED":
                if current_test_status == "FAILED":
                    result_test_status = "IGNORED_FAILURE"
                else:
                    continue
            else:
                result_test_status = current_test_status

            self.ctx[TESTS_STATUSES][domain].setdefault(test.factor_name, {})[test.test_type] = result_test_status

            current["status"] = self.ctx[TESTS_STATUSES][domain][test.factor_name][test.test_type]

            if (
                not enough_requests and
                self.ctx[TESTS_STATUSES][domain][test.factor_name][test.test_type] != "PREV_PASSED"
            ):
                self.ctx[TESTS_STATUSES][domain][test.factor_name][test.test_type] = "NOT_ENOUGH_REQUESTS"

            current["name"] = "".join((test.factor_name, ": ", test.test_type))
            current["value"] = test.value
            current["threshold"] = test.threshold
            current["link"] = test.factor_name.replace("/", "_").replace("\\", "_") + ".html"
            test_result.append(current)
            tests.add(test.factor_name)

        template_index_file_name = os.path.join(os.path.dirname(__file__), "templates", "index.html.jinja")
        index_file_name = os.path.join(dir_name, "index.html")
        self.fill_template(
            template_index_file_name,
            index_file_name,
            test_result,
            enough_requests=enough_requests,
            total_tests_count=total_tests_count,
            prev_passed_tests_count=prev_passed_tests_count,
        )

        # FRESHNESS-2014
        json_file_name = os.path.join(dir_name, "index.json")
        fu.json_dump(json_file_name, {"tests": test_result}, indent=4)

        for test in tests:
            self.create_test_response(test, dir_name, env, domain)

        details_resource = self.create_resource(
            description="Results of {} test #{}".format(self.type, self.id),
            resource_path=dir_name,
            resource_type=resource_types.REARRANGE_FACTOR_DETAILS,
        )

        self.mark_resource_ready(details_resource.id)
        logging.debug("Setting html_report_resource_proxy_url to ctx...")
        self.ctx.setdefault(HTML_REPORT_RESOURCE_PROXY_URL, dict())[domain] = details_resource.proxy_url
        self.ctx.setdefault("html_report_resource_id", dict())[domain] = details_resource.id

    def on_execute(self):
        self.add_bugbanner(bugbanner.Banners.ReleaseMachine, add_responsibles=["nkmakarov"])

        if PREVIOUS_TESTS_STATUSES not in self.ctx or PREVIOUS_TESTS_FINAL_STATUSES not in self.ctx:
            self.ctx[PREVIOUS_TESTS_STATUSES] = {}
            self.ctx[PREVIOUS_TESTS_FINAL_STATUSES] = {}
            if utils.get_or_default(self.ctx, IgnorePassed):
                try:
                    prev_task = channel.sandbox.get_task(self.ctx[PreviousTaskID.name])
                    self.ctx[PREVIOUS_TESTS_STATUSES] = prev_task.ctx[TESTS_STATUSES]
                    self.ctx[PREVIOUS_TESTS_FINAL_STATUSES] = prev_task.ctx[TESTS_FINAL_STATUSES]
                    # config_path = self._patch_config(config_path, self.ctx[PREVIOUS_TESTS_STATUSES])
                    # Temporary removing config patching
                except Exception as e:
                    eh.log_exception("Failed to ignore passed tests. Will run all tests.", e)
                # Don't run tests which were passed in previous task
                domains = self.ctx[grsr.FirstBetasDict.name].keys()
                for domain in domains:
                    if self.ctx[PREVIOUS_TESTS_FINAL_STATUSES].get(domain) == "PASSED":
                        logging.info("Tests for domain %s will be ignored", domain)
                        self.ctx[grsr.FirstBetasDict.name].pop(domain, None)
                        self.ctx[grsr.SecondBetasDict.name].pop(domain, None)
                        self.ctx[grsr.QueriesDict.name].pop(domain, None)

        bundle_path = self._prepare_bundle()

        if utils.get_or_default(self.ctx, UseConfigFromArcadia):
            arcadia_path = utils.get_or_default(self.ctx, ArcadiaConfigPath)
            config_name = "RA_config"
            config_path = Arcadia.export(
                Arcadia.trunk_url(arcadia_path),
                config_name,
            )
            logging.info("Exported config from Arcadia with path %s:\n%s", arcadia_path, fu.read_file(config_path))
        else:
            if not utils.get_or_default(self.ctx, FeaturesConfig):
                if utils.get_or_default(self.ctx, UseLatestConfigWithBlenderFeatures):
                    self.ctx[FeaturesConfig.name] = apihelpers.get_last_resource_with_attribute(
                        resource_types.REARRANGE_ACCEPTANCE_CONFIG,
                        "blender_priemka-stream-config-release", '1'
                    ).id
                else:
                    self.ctx[FeaturesConfig.name] = apihelpers.get_last_released_resource(
                        resource_types.REARRANGE_ACCEPTANCE_CONFIG
                    ).id
            config_path = self.sync_resource(self.ctx[FeaturesConfig.name])
            logging.info("Exported proto config resource #%s", self.ctx[FeaturesConfig.name])

        mr_yt_binary_path = self.sync_resource(apihelpers.get_last_released_resource("MAPREDUCE_YT_EXECUTABLE").id)
        env = os.environ.copy()
        yt_server = utils.get_or_default(self.ctx, sr_params.YtServerParameter)
        env["MR_USER"] = MR_USER
        env["DEF_MR_SERVER"] = yt_server
        env["MR_RUNTIME"] = "YT"
        env["YT_TOKEN"] = self.get_vault_data("SEARCH-RELEASERS", "yt_token")
        env["YT_USE_YAMR_DEFAULTS"] = "1"
        env["YT_LOG_LEVEL"] = "INFO"
        yt_spec = {
            "mapper": {
                "memory_limit": 8 * 1024 * 1024 * 1024,  # 8G
            },
            "data_size_per_job": 16 * 1024 * 1024 * 1024,  # 16G
        }
        env["YT_SPEC"] = json.dumps(yt_spec)

        commands = [
            ("tar -zxf {}".format(bundle_path), "unpack_parser_binary"),
            (
                "./two_contexts_features_diff from_json "
                " -i {input} -o {output} -c {config} -s {server} --domain {domain} "
                " --first_beta_platform {first_beta_platform} --second_beta_platform {second_beta_platform}".format(
                    input=self.ctx[YT_TABLE_WITH_JSON_NAME],
                    output=self.ctx[YT_TABLE_WITH_JSON_NAME] + "_features",
                    config=config_path,
                    server=yt_server,
                    domain=self.ctx[sr_params.FirstBetaHostParameter.name].rsplit(".", 1)[1],
                    first_beta_platform=ss.COLLECTIONS[
                        self.ctx[sr_params.FirstBetaCollectionParameter.name]
                    ].get("platform", ""),
                    second_beta_platform=ss.COLLECTIONS[
                        self.ctx[sr_params.SecondBetaCollectionParameter.name]
                    ].get("platform", ""),
                ), "from_json"),
            ("./two_contexts_features_diff just_calc -i {input} -o {output} -c {config} -s {server}".format(
                input=self.ctx[YT_TABLE_WITH_JSON_NAME] + "_features",
                output=self.ctx[YT_TABLE_WITH_JSON_NAME] + "_statistics",
                config=config_path,
                server=yt_server,
            ), "just_calc"),
            ("./two_contexts_features_diff calc_metrics -i {input} -o {output} -c {config} -s {server}".format(
                input=self.ctx[YT_TABLE_WITH_JSON_NAME] + "_statistics",
                output=self.ctx[YT_TABLE_WITH_JSON_NAME] + "_test_result",
                config=config_path,
                server=yt_server,
            ), "calc_metrics"),
            ("cp {} .".format(mr_yt_binary_path), "copy_mr_yt_binary"),
        ]

        self.ctx[TIMING_KEY][CONTEXT_DIFF_TIMING][TIMESTAMP_START] = time.time()
        for cmd, log_prefix in commands:
            logging.info("Start %s part of task with command %s", log_prefix, cmd)
            process.run_process(
                [cmd], work_dir="./", timeout=7200, shell=True, check=True,
                log_prefix=log_prefix, outputs_to_one_file=True, environment=env
            )
        self.ctx[TIMING_KEY][CONTEXT_DIFF_TIMING][TIMESTAMP_FINISH] = time.time()

        self.ctx[TIMING_KEY][SEPARATING_JSON_DUMP_TIMING][TIMESTAMP_START] = time.time()
        from yt.wrapper import YtClient
        from yt.wrapper import JsonFormat

        client = YtClient(env["DEF_MR_SERVER"], env["YT_TOKEN"], config={
            'pickling': {
                'python_binary': '/skynet/python/bin/python'
            }
        })
        tests_result = defaultdict(list)
        for row in client.read_table(
            self.ctx[YT_TABLE_WITH_JSON_NAME] + "_test_result",
            format=JsonFormat(attributes={"encode_utf8": False}),
        ):
            factor_name, test_type = row["key"].split("\t")
            value, threshold = row["subkey"].split("\t")
            test_status = row["value"]
            tests_result[row["domain"]].append(TestsResult(
                factor_name=factor_name,
                test_type=test_type,
                value=value,
                threshold=threshold,
                test_status=test_status,
            ))

        logging.info("Tests result dict created")
        errors = defaultdict(int)
        total = defaultdict(int)
        for domain, tests in tests_result.iteritems():
            for test_res in tests:
                total[domain] += 1
                prev_test_status = self.ctx[PREVIOUS_TESTS_STATUSES].get(domain, dict()).get(
                    test_res.factor_name,
                    dict(),
                ).get(test_res.test_type)
                if test_res.test_status == "FAILED" and prev_test_status not in ("PASSED", "PREV_PASSED"):
                    errors[domain] += 1

        reqs_count_threshold = (
            ProcessedReqsPercentThreshold.cast(self.ctx[ProcessedReqsPercentThreshold.name]) *
            self.ctx.get(grsr.TOTAL_REQUEST_COUNT, 0)
        )
        reqs_count = self.ctx.get(grsr.REQUEST_COUNT, 0)
        self.ctx[TESTS_STATUSES] = dict()
        response_directories = {domain: "html_report_" + domain for domain in tests_result}

        for domain, dir_name in response_directories.iteritems():
            paths.make_folder(dir_name)
        self.separate_json_dumps_in_one_time(response_directories, env)

        for domain, tests in tests_result.iteritems():
            self.create_response(tests, env, reqs_count > reqs_count_threshold, response_directories[domain], domain)

        self.ctx[TIMING_KEY][SEPARATING_JSON_DUMP_TIMING][TIMESTAMP_FINISH] = time.time()
        self.ctx[TESTS_FINAL_STATUSES] = dict()

        logging.info("Saving final statuses of tests to context")

        for domain in tests_result:
            if errors[domain]:
                self.ctx[TESTS_FINAL_STATUSES][domain] = "FAILED"
            else:
                self.ctx[TESTS_FINAL_STATUSES][domain] = "PASSED"

        for domain, status in self.ctx[PREVIOUS_TESTS_FINAL_STATUSES].iteritems():
            if status == "PASSED":
                self.ctx[TESTS_FINAL_STATUSES][domain] = status

        tables_suffixes = ["", "_features", "_statistics", "_test_result"]
        commands = [
            "./mapreduce-yt -server {server} -drop {table_prefix}{table_suffix}".format(
                server=yt_server,
                table_prefix=self.ctx[YT_TABLE_WITH_JSON_NAME],
                table_suffix=table_suffix) for table_suffix in tables_suffixes
        ]
        logging.info("Remove all tables, created along the test")
        for cmd in commands:
            process.run_process(
                [cmd], work_dir="./", timeout=7200, shell=True, check=True,
                log_prefix="remove_tables", outputs_to_one_file=True, environment=env
            )

        self.set_timing_info()

        eh.ensure(
            reqs_count > reqs_count_threshold,
            "Not enough requests processed: {reqs} of {total_reqs}".format(
                reqs=reqs_count, total_reqs=self.ctx.get(grsr.TOTAL_REQUEST_COUNT, 0)
            )
        )
        eh.ensure(not any(errors.itervalues()), "Failed some tests")

    def set_timing_info(self):
        for key, timing in self.ctx[TIMING_KEY].items():
            try:
                timing[TOTAL_TIME] = timing[TIMESTAMP_FINISH] - timing[TIMESTAMP_START]  # In seconds
                self.set_info("{}: {}".format(
                    timing.get("description", "Time spent for {}".format(key)),
                    timedelta(seconds=timing[TOTAL_TIME]),
                ))
            except KeyError:
                self.set_info("Error occured while getting time info for key {}".format(key))
                logging.exception("Can't get total time for key %s", key)

    def on_failure(self):
        if not utils.get_or_default(self.ctx, SendNotificationToStartrekOnTaskFailure):
            return
        component_name = self.ctx.get(rm_params.ComponentName.name)
        if not utils.get_or_default(self.ctx, SendNotificationToStartrek):
            logging.info("Notification to startrek is disabled")
            return
        if not component_name:
            self.set_info("Component name not found so no Startrek notification created")
            return
        st_helper = STHelper(self.get_vault_data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME))
        st_helper.comment(
            self.ctx.get(rm_params.ReleaseNum.name),
            "{task_type} {link} FAILED (some of its subtasks might have failed)".format(
                task_type=self.type,
                link=lb.task_wiki_link(self.id, "task"),
            ),
            rmc.COMPONENTS[component_name]()
        )

    def on_finish(self):
        bugbanner.BugBannerTask.on_finish(self)

    def on_break(self):
        bugbanner.BugBannerTask.on_break(self)

    @staticmethod
    def _patch_config(f_name, prev_results):
        logging.debug("Patching config")
        f_patched_name = 'patched_config'
        lines_gen = fu.read_line_by_line(f_name)
        for line in lines_gen:
            if 'ResultConfig' not in line:
                fu.append_lines(f_patched_name, [line])
            else:
                buf = []
                while '}' not in line:
                    buf.append(line)
                    line = lines_gen.next()
                buf.append(line)
                feature_name, field_name = '', ''
                for buf_line in buf:
                    if 'FeatureName:' in buf_line:
                        feature_name = buf_line.split('"')[1]
                    elif 'FieldName:' in buf_line:
                        field_name = buf_line.split('"')[1]
                if 'PASSED' not in prev_results.get(feature_name, {}).get(field_name, ''):
                    fu.append_lines(f_patched_name, buf)
                else:
                    logging.debug("Removed %s: %s", feature_name, field_name)
        return f_patched_name

    def _prepare_bundle(self):
        utils.check_subtasks_fails()
        build_task_name = "get_binary_task"
        parser_resource_id = "two_contexts_features_diff_binary"
        responses_task_name = "get_responses_dependent_task"

        # Add timing context
        if TIMING_KEY not in self.ctx:
            self.ctx[TIMING_KEY] = {
                REQUESTS_TIMING: {
                    "description": "Time spent for getting requests",
                },
                CONTEXT_DIFF_TIMING: {
                    "description": "Time spent for running two_contexts_features_diff",
                },
                SEPARATING_JSON_DUMP_TIMING: {
                    "description": "Time spent for separating json jumps and rendering all test results",
                }
            }

        # Get two_contexts_features_diff binary
        if not (build_task_name in self.ctx or self.ctx[sr_params.ParserBinaryParameter.name]):
            if self.ctx.get(UseLatestParserBinary.name):
                found_tasks = channel.sandbox.list_tasks(
                    task_type="BUILD_ARCADIA_BINARY", status="FINISHED",
                    completed_only=True,
                    descr_mask="Periodically build search/ra2/two_contexts_features_diff"
                )
                for t in sorted(found_tasks, key=lambda x: int(x.id), reverse=True):
                    if t.ctx.get('out_resource_id'):
                        res = channel.sandbox.get_resource(t.ctx['out_resource_id'])
                        if res and res.is_ready():
                            logging.info("Found " + str(t.id) +
                                         " task which built two_contexts_features_diff. Reusing its result.")
                            self.ctx[sr_params.ParserBinaryParameter.name] = res.id
                            break
            if not self.ctx[sr_params.ParserBinaryParameter.name]:
                logging.info("Start execution of BuildArcadiaBinary task")
                subtask_for_build = self.create_subtask(
                    task_type="BUILD_ARCADIA_BINARY",
                    arch="linux",
                    description="Build two_contexts_features_diff for {} task #{}".format(self.type, self.id),
                    input_parameters={
                        "notify_if_finished": "",
                        "build_path": "search/ra2/two_contexts_features_diff",
                        "binaries": "search/ra2/two_contexts_features_diff/two_contexts_features_diff",
                    },
                )
                self.ctx[build_task_name] = subtask_for_build.id
                self.wait_tasks(
                    [subtask_for_build],
                    list(self.Status.Group.FINISH + self.Status.Group.BREAK),
                    wait_all=True
                )

        # Unpack two_contexts_features_diff binary
        if parser_resource_id not in self.ctx:
            if self.ctx[sr_params.ParserBinaryParameter.name]:
                parser_binary_id = self.ctx.get(sr_params.ParserBinaryParameter.name)
                self.ctx[parser_resource_id] = parser_binary_id
            else:
                bundle_item = rest.Client().resource.read(
                    task_id=self.ctx[build_task_name],
                    type=str(resource_types.ARCADIA_BINARY_ARCHIVE),
                    state=resource.State.READY,
                    limit=1,
                )["items"]
                eh.ensure(bundle_item, "No bundle in task #{}".format(self.ctx[build_task_name]))
                self.ctx[parser_resource_id] = bundle_item[0]["id"]

        # Get betas responses
        if responses_task_name not in self.ctx:
            logging.info("Start execution of GetReportSimultaneousResponses task")
            subtask_for_responses = self.create_subtask(
                task_type="GET_REPORT_SIMULTANEOUS_RESPONSES",
                description=(
                    "GetReportSimultaneousResponses for {} task #{}".format(self.type, self.id)
                ),
                execution_space=5000,  # 5G
                input_parameters={
                    sr_params.FirstBetaHostParameter.name: self.ctx[sr_params.FirstBetaHostParameter.name],
                    sr_params.SecondBetaHostParameter.name: self.ctx[sr_params.SecondBetaHostParameter.name],
                    sr_params.FirstBetaCollectionParameter.name: (
                        self.ctx[sr_params.FirstBetaCollectionParameter.name]
                    ),
                    sr_params.SecondBetaCollectionParameter.name: (
                        self.ctx[sr_params.SecondBetaCollectionParameter.name]
                    ),
                    sr_params.FirstBetaUserAgentParameter.name:
                        self.ctx.get(sr_params.FirstBetaUserAgentParameter.name, ""),
                    sr_params.SecondBetaUserAgentParameter.name:
                        self.ctx.get(sr_params.SecondBetaUserAgentParameter.name, ""),
                    sr_params.FirstBetaCgiParameter.name: self.ctx[sr_params.FirstBetaCgiParameter.name],
                    sr_params.SecondBetaCgiParameter.name: self.ctx[sr_params.SecondBetaCgiParameter.name],
                    sr_params.FirstBetaDumpAllJson.name: self.ctx[sr_params.FirstBetaDumpAllJson.name],
                    sr_params.SecondBetaDumpAllJson.name: self.ctx[sr_params.SecondBetaDumpAllJson.name],
                    sr_params.FirstBetaJsonToStandardFormat.name: True,
                    sr_params.SecondBetaJsonToStandardFormat.name: True,

                    grsr.ScraperProfileName.name: self.ctx[grsr.ScraperProfileName.name],

                    grsr.RequestsParameter.name: self.ctx[grsr.RequestsParameter.name],
                    grsr.NumberOfRandomRequestsParameter.name: self.ctx[grsr.NumberOfRandomRequestsParameter.name],
                    grsr.UseLatestQueries.name: self.ctx[grsr.UseLatestQueries.name],
                    grsr.TriesCountParameter.name: self.ctx[grsr.TriesCountParameter.name],

                    ra2.UsePersonalUidsParameter.name: self.ctx[ra2.UsePersonalUidsParameter.name],
                    ra2.PersonalUidsResourceParameter.name: self.ctx.get(ra2.PersonalUidsResourceParameter.name),

                    grsr.YtUsingParameter.name: True,
                    sr_params.YtServerParameter.name: self.ctx[sr_params.YtServerParameter.name],
                    sr_params.OutputTableParameter.name:
                        "//tmp/GetReportSimultaneousResponses/" +
                        "".join(random.SystemRandom(time.time()).choice(
                            string.ascii_lowercase + string.ascii_uppercase + string.digits
                        ) for _ in range(20)),

                    grsr.UseUnansweredCheckParameter.name: self.ctx[grsr.UseUnansweredCheckParameter.name],
                    grsr.UnansweredSourcesParameter.name: self.ctx[grsr.UnansweredSourcesParameter.name],
                    grsr.FullyAnsweredSourcesParameter.name: self.ctx[grsr.FullyAnsweredSourcesParameter.name],
                    grsr.UnansweredTriesNumberParameter.name: self.ctx[grsr.UnansweredTriesNumberParameter.name],
                    grsr.UnansweredDiffThresholdParameter.name: self.ctx[grsr.UnansweredDiffThresholdParameter.name],
                    grsr.TryTillFullSuccessParameter.name: self.ctx[grsr.TryTillFullSuccessParameter.name],

                    grsr.FirstBetasDict.name: self.ctx.get(grsr.FirstBetasDict.name),
                    grsr.SecondBetasDict.name: self.ctx.get(grsr.SecondBetasDict.name),
                    grsr.QueriesDict.name: self.ctx.get(grsr.QueriesDict.name),

                    sr_params.ParserBinaryParameter.name: self.ctx[parser_resource_id],
                    grr.SearchPropFilterParameter.name: self.ctx[grr.SearchPropFilterParameter.name],
                    grr.UseLatestRandomQueriesBinary.name: self.ctx[grr.UseLatestRandomQueriesBinary.name],
                    sr_params.ScraperOverYtPoolParameter.name: (
                        utils.get_or_default(self.ctx, sr_params.ScraperOverYtPoolParameter)
                    ),
                    "kill_timeout": TimeToKill.cast(self.ctx[TimeToKill.name]) * 7200,
                    "notify_if_finished": "",
                },
            )
            self.ctx[responses_task_name] = subtask_for_responses.id
            self.ctx[YT_TABLE_WITH_JSON_NAME] = subtask_for_responses.ctx[sr_params.OutputTableParameter.name]
            logging.info(
                "Save JSON responses into table %s on server %s",
                self.ctx[YT_TABLE_WITH_JSON_NAME],
                utils.get_or_default(self.ctx, sr_params.YtServerParameter),
            )
            self.ctx[TIMING_KEY][REQUESTS_TIMING][TIMESTAMP_START] = time.time()
            self.wait_tasks(
                [subtask_for_responses],
                list(self.Status.Group.FINISH + self.Status.Group.BREAK),
                wait_all=True
            )

        # Count beta responses
        responses_task = channel.sandbox.get_task(self.ctx[responses_task_name])
        self.ctx[TIMING_KEY][REQUESTS_TIMING][TIMESTAMP_FINISH] = time.time()
        self.ctx[grsr.REQUEST_COUNT] = responses_task.ctx.get(grsr.REQUEST_COUNT, 0)
        self.ctx[grsr.TOTAL_REQUEST_COUNT] = responses_task.ctx.get(grsr.TOTAL_REQUEST_COUNT, 0)

        def _fix(r):
            return re.sub("json_dump=.*?&|json_dump=.*?$", "&", r)

        self.ctx[grsr.FIRST_BETA_TEMPLATE] = _fix(responses_task.ctx[grsr.FIRST_BETA_TEMPLATE])
        self.ctx[grsr.SECOND_BETA_TEMPLATE] = _fix(responses_task.ctx[grsr.SECOND_BETA_TEMPLATE])

        first_beta_templates = responses_task.ctx[grsr.FIRST_BETA_TEMPLATE_DICT]
        second_beta_templates = responses_task.ctx[grsr.SECOND_BETA_TEMPLATE_DICT]

        self.ctx[grsr.FIRST_BETA_TEMPLATE_DICT] = {
            '*': self.ctx[grsr.FIRST_BETA_TEMPLATE],
        }
        self.ctx[grsr.SECOND_BETA_TEMPLATE_DICT] = {
            '*': self.ctx[grsr.SECOND_BETA_TEMPLATE],
        }

        if first_beta_templates:
            self.ctx[grsr.FIRST_BETA_TEMPLATE_DICT] = {
                region: _fix(first_beta_templates[region]) for region in first_beta_templates
            }
        if second_beta_templates:
            self.ctx[grsr.SECOND_BETA_TEMPLATE_DICT] = {
                region: _fix(second_beta_templates[region]) for region in second_beta_templates
            }

        return self.sync_resource(self.ctx[parser_resource_id])


__Task__ = RearrangeAcceptance2
