import logging

from sandbox import sdk2
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine import core as rm_core
from sandbox.projects.release_machine import client
from sandbox.projects.release_machine.components import all as all_rm_components
from sandbox.projects.release_machine.core import task_env

logger = logging.getLogger(__name__)


class ReleaseMachineAutobranchCreator(sdk2.Task):
    """Create new release for 'autobranched' Release machine components."""

    class Requirements(task_env.StartrekRequirements):
        pass

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group("Developer options") as developer_options:
            debug_mode = sdk2.parameters.Bool(
                "Debug mode",
                description="Do not perform any modifying actions",
                default=True,  # TODO: Change to False?
            )
        with sdk2.parameters.Output:
            release_was_triggered = sdk2.parameters.Bool("Release was triggered")

    def process_component(self, c_info, rm_client, nanny_token=None):
        """
        Try to press release.

        Check all conditions for new release.
        Try to determine revision for new release
        Press release.
        :rtype: sandbox.projects.release_machine.core.Result instance
        """
        if not c_info.new_release_trigger_activated(nanny_token=nanny_token):
            return rm_core.Error("New release was not triggered yet")

        if not c_info.new_release_requirements_satisfied():
            return rm_core.Error(
                "New release is currently not allowed by {} component's requirements".format(c_info.name)
            )

        revision = c_info.get_new_release_revision()
        if not revision:
            return rm_core.Error("No revision found for new release")

        if not self.Parameters.release_was_triggered:
            self.Parameters.release_was_triggered = True

        return self.create_pre_release(c_info, revision, rm_client)

    def create_pre_release(self, c_info, revision, rm_client):
        """Create new manual run of pre-release job for component at given revision."""
        if self.Parameters.debug_mode:
            return rm_core.Error("Debug mode enabled, no actions were performed")

        response = rm_client.post_pre_release(
            revision, c_info.testenv_cfg__trunk_db, c_info.name, int(c_info.last_scope_num)
        )
        if response is None:
            return rm_core.Error("post_pre_release failed, see logs for more details")
        return rm_core.Ok("Success")

    def on_execute(self):
        nanny_token = rm_sec.get_rm_token(self)
        rm_client = client.RMClient()

        started_releases = []

        for component_name in all_rm_components.get_component_names():
            try:
                c_info = all_rm_components.get_component(component_name)
            except:
                logger.exception("%s's config is broken!", component_name)
                self.set_info("{}'s config is broken! Skipping it".format(component_name))
                continue

            logger.debug("Inspecting component %s", component_name)

            try:
                if c_info.is_branched and c_info.svn_cfg__allow_autobranches:
                    result = self.process_component(c_info, rm_client, nanny_token=nanny_token)
                    self.set_info(
                        "New release for {} was{} started: {}".format(
                            c_info.name, "" if result.ok else " not", result.result
                        )
                    )
                    if result.ok:
                        started_releases.append(component_name)
            except Exception as exc:
                logger.error("Failed to create new release for %s", c_info.name, exc_info=True)
                self.set_info("New release for {} was not started due to {}".format(c_info.name, exc))

        self.Parameters.tags = started_releases
