# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import json
import logging
import os

import sandbox.projects.common.file_utils as fu
import sandbox.projects.common.time_utils as tu
import sandbox.projects.common.error_handlers as eh
import sandbox.projects.release_machine.components.all as rmc
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.helpers.changelog_formatter as cf
import sandbox.projects.release_machine.changelogs as ch
import sandbox.projects.release_machine.helpers.svn_helper as rm_svn
import sandbox.projects.release_machine.input_params2 as rm_params
import sandbox.projects.release_machine.resources as rm_res
import sandbox.projects.release_machine.tasks.base_task as rm_bt
import sandbox.sdk2 as sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common import decorators
from sandbox.projects.common.arcadia import sdk as asdk


class ReleaseMachineChangelog(rm_bt.BaseReleaseMachineTask):

    class Requirements(task_env.TinyRequirements):
        disk_space = 5 * 1024  # 5 Gb

    class Parameters(rm_params.DefaultReleaseMachineParameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        kill_timeout = 1.5 * 60 * 60  # 1 h 30 min
        major_release_num = sdk2.parameters.Integer("Major release number", default=0)
        minor_release_num = sdk2.parameters.Integer("Minor release number", default=0)

        candidate_path = sdk2.parameters.ArcadiaUrl("Path of release candidate item", required=True)
        candidate_revision = sdk2.parameters.Integer("Revision of release candidate item")
        use_previous_branch_as_baseline = sdk2.parameters.Bool("Use previous branch as baseline", default_value=False)
        with use_previous_branch_as_baseline.value[False]:
            first_revision = sdk2.parameters.Integer("First revision of changelog")
            baseline_path = sdk2.parameters.ArcadiaUrl("Path of baseline item", required=False, default_value="")
            baseline_revision = sdk2.parameters.Integer("Revision of baseline item")

        with sdk2.parameters.Output():
            changelog = sdk2.parameters.Resource("Changelog", resource_type=rm_res.RELEASE_MACHINE_CHANGELOG)
            changelog_html = sdk2.parameters.Resource("Changelog", resource_type=rm_res.ChangelogHtml)

    def on_enqueue(self):
        super(ReleaseMachineChangelog, self).on_enqueue()
        with self.memoize_stage.create_changelog_on_enqueue:
            self.Parameters.changelog = rm_res.RELEASE_MACHINE_CHANGELOG(
                task=self,
                description="Changelog json for {}".format(self.Parameters.component_name),
                path="{}_changelog.json".format(self.Parameters.component_name),
            )
            self.Parameters.changelog_html = rm_res.ChangelogHtml(
                task=self,
                description="Changelog html for {}".format(self.Parameters.component_name),
                path="{}_changelog_html".format(self.Parameters.component_name),
            )

    @decorators.memoized_property
    def candidate_revision(self):
        if self.Parameters.candidate_revision:
            return self.Parameters.candidate_revision
        return int(sdk2.svn.Arcadia.info(self.Parameters.candidate_path)["commit_revision"])

    @decorators.memoized_property
    def c_info(self):
        return rmc.get_component(self.Parameters.component_name)

    @decorators.memoized_property
    def tag_number(self):
        if (
            self.c_info.release_cycle_type == rm_const.ReleaseCycleType.BRANCH or
            self.c_info.release_cycle_type == rm_const.ReleaseCycleType.CI
        ):
            return self.Parameters.minor_release_num
        else:
            return self.Parameters.major_release_num

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)
        self.set_changelog_attributes()
        full_changelog = {
            "component_name": self.Parameters.component_name,
            "all_changes": [],
            "creation_time": tu.date_ymdhm(),
            "major_release_num": self.Parameters.changelog.major_release_num,
            "minor_release_num": self.Parameters.changelog.minor_release_num,
            "candidate_path": self.Parameters.candidate_path,
            "candidate_rev_trunk": rm_svn.SvnHelper.get_last_trunk_revision_before_copy(self.Parameters.candidate_path),
            "candidate_revision": self.candidate_revision,
        }
        self.Parameters.changelog_html.path.mkdir()
        for released_resource in self.c_info.get_active_releases() or [None]:
            release_item_changelog = self.create_changelog_for_one_release_item(released_resource)
            full_changelog["all_changes"].append(release_item_changelog)
        fu.write_file(self.Parameters.changelog.path, json.dumps(full_changelog, indent=2, separators=(',', ': ')))
        if not full_changelog["all_changes"]:  # create dummy file to satisfy sandbox
            html_path = self.Parameters.changelog_html.path / "no_changes.html"
            fu.write_file(html_path, "No changes found")

    def create_changelog_for_one_release_item(self, released_resource):
        baseline_rev_trunk, baseline_path, baseline_rev = self.detect_baseline_info(released_resource)
        first_rev = self.Parameters.first_revision or baseline_rev_trunk
        self.set_info("First revision detected: {}".format(first_rev))
        if int(first_rev) > int(baseline_rev):
            eh.check_failed("[{}] First revision is greater than baseline revision: {} > {}".format(
                released_resource, first_rev, baseline_rev
            ))
        if int(first_rev) > int(self.candidate_revision):
            eh.check_failed("[{}] First revision is greater than candidate revision: {} > {}".format(
                released_resource, first_rev, self.candidate_revision
            ))
        changelog_master = ch.ChangeLogMaster(
            first_rev=first_rev,
            prod_released_path=baseline_path,
            prod_released_revision=baseline_rev,
            candidate_path=self.Parameters.candidate_path,
            candidate_revision=self.candidate_revision,
            filters=self.get_filters(first_rev),
        )
        changelog = [i.to_dict() for i in sorted(changelog_master.get_changelog(), reverse=True)]
        self.set_info("Got {} changelog items for {}".format(len(changelog), released_resource))
        logging.debug("Got changelog:\n%s", json.dumps(changelog, indent=2))
        html_path = self.Parameters.changelog_html.path / "{}.html".format(released_resource or "changelog")
        changelog_table_data = cf.ChangeLogDataHtml(self.c_info, tuple(ch.ChangeLogEntry.Attrs), changelog)
        fu.write_file(html_path, cf.ChangeLogHtml().format_table(changelog_table_data))
        return {
            "first_revision": first_rev,
            "baseline_path": baseline_path,
            "baseline_revision": baseline_rev,
            "baseline_rev_trunk": baseline_rev_trunk,
            "candidate_path": self.Parameters.candidate_path,
            "candidate_rev_trunk": rm_svn.SvnHelper.get_last_trunk_revision_before_copy(self.Parameters.candidate_path),
            "candidate_revision": self.candidate_revision,
            "changes": changelog,
            "release_item": released_resource.to_json() if released_resource else None,
        }

    def detect_baseline_info(self, released_resource):
        baseline_path = self.Parameters.baseline_path
        if not baseline_path:
            if self.Parameters.use_previous_branch_as_baseline:
                logging.info("Trying to use previous branch as baseline")
                branch_num = self.Parameters.major_release_num - 1
                logging.info("Previous branch num = %s", branch_num)
                while branch_num > 0:
                    branch_path = self.c_info.full_branch_path(branch_num)
                    logging.info("Checking previous branch path = %s", branch_path)
                    if sdk2.svn.Arcadia.check(branch_path):
                        baseline_path = branch_path
                        break
                    branch_num -= 1
            else:
                baseline_path = self.c_info.path_for_released_resource(released_resource)
        self.set_info("Baseline path detected: {}".format(baseline_path))
        if not baseline_path:
            rev = self.candidate_revision
            return rev, self.c_info.svn_cfg__trunk_url, rev
        baseline_rev_trunk = rm_svn.SvnHelper.get_last_trunk_revision_before_copy(baseline_path)
        self.set_info("Baseline trunk revision detected: {}".format(baseline_rev_trunk))
        baseline_rev = self.Parameters.baseline_revision
        if not baseline_rev:
            # take last revision of baseline path
            # possibly, will lead to loosing some of merged, but not released revisions (if baseline is branch)
            baseline_rev = int(sdk2.svn.Arcadia.info(baseline_path)["commit_revision"])
        self.set_info("Baseline revision detected: {}".format(baseline_rev))
        return baseline_rev_trunk, baseline_path, baseline_rev

    def get_filters(self, first_revision):
        filters = [
            self._path_filter,
            self._marker_filter,
            self._testenv_filter,
            self._review_filter(first_revision),
            self._banned_authors_filter,
        ]
        filters = [i for i in filters if i]
        self.set_info("Got {} filters: {}".format(len(filters), filters))
        return filters

    @decorators.memoized_property
    def _banned_authors_filter(self):
        return ch.BannedAuthorsFilter(self.c_info.changelog_cfg__banned_authors)

    @decorators.memoize
    def _review_filter(self, first_revision):
        if self.c_info.changelog_cfg__review_groups:
            return ch.ReviewFilter(self.c_info.changelog_cfg__review_groups, first_revision, self.candidate_revision)

    @decorators.memoized_property
    def _testenv_filter(self):
        dbs = self.c_info.changelog_cfg__testenv_dbs
        if dbs:
            if dbs is True:
                dbs = [self.c_info.testenv_cfg__trunk_db]
            return ch.TestenvFilter(dbs)

    @decorators.memoized_property
    def _marker_filter(self):
        if self.c_info.changelog_cfg__markers:
            return ch.MarkerFilter(self.c_info.changelog_cfg__markers)

    @decorators.memoized_property
    def _path_filter(self):
        paths = self.c_info.changelog_cfg__observed_paths

        if self.c_info.changelog_cfg__ya_make_targets:
            build_paths = self._get_build_paths(self.c_info.changelog_cfg__ya_make_targets)

            if build_paths is None:
                raise Exception("Unable to get build paths for ya make targets {}".format(
                    self.c_info.changelog_cfg__ya_make_targets,
                ))

            paths += build_paths

        if paths:
            paths = [i if i.startswith("arcadia") else os.path.join("arcadia", i) for i in paths]
            if (
                self.c_info.changelog_cfg__svn_paths_filter and
                self.c_info.changelog_cfg__svn_paths_filter.permission_type == rm_const.PermissionType.BANNED
            ):
                for i in self.c_info.changelog_cfg__svn_paths_filter.paths:
                    if i not in paths:
                        paths.append(i)
            return ch.PathsFilter([
                self.c_info.changelog_cfg__set_paths_importance(path) for path in paths
            ])

    def _get_build_paths(self, svn_build_paths):
        logging.info("Use dependency dump to get build paths")
        with self._get_arc_mount_point() as arc_url:
            try:
                return self._dump_build_paths(arc_url, svn_build_paths)
            except Exception as e:
                eh.log_exception("Unable to dump build paths, stay with initial paths as fallback", e, task=self)

    @decorators.retries(3, delay=5)
    def _get_arc_mount_point(self):
        return asdk.mount_arc_path(
            "{}/arcadia@{}".format(self.Parameters.candidate_path, self.candidate_revision)
        )

    @decorators.retries(2, delay=2)
    def _dump_build_paths(self, arc_url, svn_build_paths):
        arc_paths = [os.path.join(arc_url, p.replace("arcadia/", "", 1)) for p in svn_build_paths]
        modules_dump = asdk.dump_modules(arc_url, arc_paths)
        filtered_build_paths = list(self._filter_deps(modules_dump))
        logging.info("Filtered build paths: %d", len(filtered_build_paths))
        logging.debug("%s", "\n".join(filtered_build_paths))
        return filtered_build_paths

    @staticmethod
    def _filter_deps(dump):
        """
            No need to return path, which has prefix equal to another path.
            Arcadia.log will show all logs by shortest path.
        """
        sorted_deps = sorted([i.split()[2] for i in dump.split("\n") if i.startswith("module")])
        last_prefix = "#"  # any string, not matched with first element
        for i in sorted_deps:
            if not i.startswith(last_prefix):
                last_prefix = i
                yield i

    def set_changelog_attributes(self):
        self.Parameters.changelog.component = self.Parameters.component_name
        if self.Parameters.major_release_num:
            self.Parameters.changelog.major_release_num = self.Parameters.major_release_num
            self.Parameters.changelog.minor_release_num = self.Parameters.minor_release_num
        else:
            release_numbers = self.c_info.svn_cfg__get_release_numbers(self.Parameters.candidate_path)
            if release_numbers is not None:
                major, minor = release_numbers
            else:
                major, minor = self.c_info.svn_cfg__get_major_release_num(self.Parameters.candidate_path), 0
            self.Parameters.changelog.major_release_num = major
            self.Parameters.changelog.minor_release_num = minor

    def _get_rm_proto_event_hash_items(self, event_time_utc_iso, status=None):
        return (
            self.Parameters.component_name,
            'ChangelogCreated',
            self.Parameters.major_release_num,
            self.tag_number,
            self.Parameters.changelog.id,
            "TASK_{}".format(status or self.status),
        )

    def _get_rm_proto_event_specific_data(self, rm_proto_events, event_time_utc_iso, status=None):

        specific_data = rm_proto_events.ChangelogCreatedData(
            scope_number=str(self.Parameters.major_release_num),
            tag_number=str(self.tag_number),
            job_name=self.get_job_name_from_gsid(),
            changelog_resource_id=str(self.Parameters.changelog.id),
        )

        result = {
            str('changelog_created_data'): specific_data,
        }

        logging.debug("The following specific data created: %s", result)

        return result
