import logging
import sys
import jinja2
import os
import re

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common import decorators
from sandbox.projects.common import time_utils as tu
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import requests_wrapper
from sandbox.projects.common.search import bugbanner2 as bb2
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import task_env
from sandbox.projects.release_machine.helpers import staff_helper
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.components.configs import all as all_cfg
from sandbox.projects.release_machine.tasks.ReleaseMachineChatCrawler import chat_check_result


if sys.version_info[0:2] >= (3, 8):  # python 3.8 or higher
    from functools import cached_property
else:
    from sandbox.projects.common.decorators import memoized_property as cached_property


class ReleaseMachineChatCrawler(bb2.BugBannerTask, binary_task.LastBinaryTaskRelease):

    CORPORATE_EMAIL_HOST = "yandex-team.ru"
    STAFF_NOT_FOUND_MESSAGE = "not found"
    TG_BOT_TEST_MESSAGE = (
        "I'm just testing something, don't mind me. "
        "If I bother you please contact RM support https://t.me/joinchat/AAAAAEN49wUGP2uImQ6Dfw"
    )

    class Requirements(task_env.TinyRequirements):
        pass

    class Context(sdk2.Task.Context):
        results = {}
        report_html = ""

    class Parameters(binary_task.LastBinaryReleaseParameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        rm_host = sdk2.parameters.String(
            "Release Machine service address",
            default_value=rm_const.Urls.RM_HOST,
        )
        resolve_possible_telegram_chat_problems = sdk2.parameters.Bool(
            "Resolve possible Telegram chat problems (such as group -> supergroup transition)",
            default_value=True,
        )

    @cached_property
    def component_names_to_process(self):
        return sorted(set(all_cfg.get_all_names()) - self.Context.results.keys())

    @cached_property
    def notification_client(self):
        from release_machine.release_machine.services.release_engine.services.Notification import NotificationClient
        return NotificationClient.from_address(self.Parameters.rm_host)

    @cached_property
    def telegram_bot(self):
        import telegram
        return telegram.Bot(sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.TELEGRAM_TOKEN_NAME))

    @cached_property
    def staff_client(self):
        return staff_helper.StaffApi(rm_sec.get_rm_token(self))

    @sdk2.report(title="Crawler Report", label="report")
    def get_report_html(self):
        return self.Context.report_html or "<h3>Report is not ready</h3>"

    @cached_property
    def rm_event_client(self):
        from release_machine.release_machine.services.release_engine.services.Event import EventClient
        return EventClient.from_address(self.Parameters.rm_host)

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)
        bb2.BugBannerTask.on_save(self)

    def on_execute(self):

        binary_task.LastBinaryTaskRelease.on_execute(self)
        bb2.BugBannerTask.on_execute(self)

        results_bool = []

        for component_name in self.component_names_to_process:

            component_chats = self._get_component_chats(component_name)

            if component_chats is None:
                result = chat_check_result.ChatCheckResult(component_name, success=False, message="Unable to get chats")
            else:
                result = chat_check_result.ChatCheckResult(component_name)

                for chat in component_chats.chats:
                    result.add_result(self._check_chat(chat))

            results_bool.append(result.is_ok)
            self.Context.results[component_name] = result.to_dict()
            self.Context.save()

            self._report_component_state(
                component_name=component_name,
                messages=result.list_messages(failed_only=(not result.is_ok)),
                ok=result.is_ok,
            )

        eh.ensure(all(results_bool), "Some chats are invalid. Please see Crawler Report tab")

    def on_finish(self, prev_status, status):
        self._prepare_report()

    def on_break(self, prev_status, status):
        self._prepare_report()

    def on_failure(self, prev_status):
        from release_machine.common_proto import events_pb2
        from release_machine.public import events as events_public
        from release_machine.release_machine.proto.structures import message_pb2

        self.rm_event_client.post_proto_events(message_pb2.PostProtoEventsRequest(
            events=[
                events_pb2.EventData(
                    general_data=events_pb2.EventGeneralData(
                        hash=events_public.get_event_hash(
                            self.id,
                            tu.datetime_utc_iso(),
                            rm_const.RELEASE_MACHINE_ERROR_REPORT,
                        ),
                        component_name="release_machine",
                        referrer="sandbox_task:{}".format(self.id),
                    ),
                    custom_message_data=events_pb2.CustomMessageData(
                        message="{task_link} FAILED. See task's Crawler Report tab".format(
                            task_link=lb.task_link(self.id, self.type),
                        ),
                        condition_tag=rm_const.RELEASE_MACHINE_ERROR_REPORT,
                    ),
                ),
            ],
        ))

    def _get_component_chats(self, component_name):
        import grpc
        from release_machine.release_machine.proto.structures import message_pb2

        try:
            return self.notification_client.get_component_chats(
                message_pb2.ComponentRequest(component_name=component_name)
            )
        except grpc.RpcError:
            logging.exception("Unable to get component chats for component %s", component_name)
            self.set_info("Failed to load chats for {}".format(component_name))

    def _check_chat(self, chat_obj):
        from release_machine.release_machine.proto.structures import table_pb2

        if chat_obj.transport_type == table_pb2.TransportType.telegram:
            return self._check_telegram_chat(chat_obj)
        elif chat_obj.transport_type == table_pb2.TransportType.email:
            return self._check_email(chat_obj)

        return chat_check_result.ChatCheckResult(
            chat_obj.name,
            success=True,
            message="{} chats are considered OK by default".format(
                table_pb2.TransportType.DESCRIPTOR.values_by_number[chat_obj.transport_type].name,
            ),
        )

    def _check_email(self, chat_obj):
        logging.info("Processing email %s", chat_obj)

        email = chat_obj.chat_id
        staff_login, _, host = email.partition("@")

        if host != self.CORPORATE_EMAIL_HOST:
            return chat_check_result.ChatCheckResult(
                email,
                success=False,
                message="Alert! Not a corporate email: {}".format(email),
            )

        ok, msg = self._is_staff_login_still_employed(staff_login)

        return chat_check_result.ChatCheckResult(email, success=ok, message=msg)

    @decorators.retries(3)
    def _check_telegram_chat(self, chat_obj):

        from telegram.error import BadRequest, Unauthorized

        logging.info("Processing Telegram chat %s", chat_obj)

        try:

            tg_chat = self.telegram_bot.get_chat(chat_obj.chat_id)
            logging.debug("Got response from Telegram API: %s", tg_chat)

        except BadRequest as br:
            return chat_check_result.ChatCheckResult(
                chat_obj.name,
                success=False,
                message="Cannot get response from Telegram API for chat {}: {}".format(
                    chat_obj.chat_id,
                    br.message,
                ),
            )

        except Unauthorized as unauthorized_err:
            return chat_check_result.ChatCheckResult(
                chat_obj.name,
                success=False,
                message="Got Unauthorized error from Telegram API for chat {}: {}".format(
                    chat_obj.chat_id,
                    unauthorized_err.message,
                ),
            )

        logging.info("Chat recognized as %s", tg_chat.type)

        if tg_chat.type is None:
            return chat_check_result.ChatCheckResult(
                chat_obj.name,
                success=False,
                message="Unrecognizable chat {chat_id}. Telegram returned the following: {tg_response}".format(
                    chat_id=chat_obj.chat_id,
                    tg_response=tg_chat,
                ),
            )

        if tg_chat.type == tg_chat.PRIVATE:
            ok, msg = self._is_telegram_user_still_employed(tg_chat.username)
            return chat_check_result.ChatCheckResult(
                chat_obj.name,
                success=ok,
                message=(
                    "Chat with {username} (chat id  {chat_id}) is forbidden: {msg}".format(
                        username=tg_chat.username,
                        chat_id=chat_obj.chat_id,
                        msg=msg,
                    )
                    if not ok else msg
                ),
            )

        if tg_chat.permissions and not tg_chat.permissions.can_send_messages:
            logging.error("Bot cannot send messages to this chat")
            logging.info("Trying to fix it")
            return self._try_to_fix_telegram_write_permission_problem(chat_obj)

        message = "Telegram {}".format(tg_chat.type)

        return chat_check_result.ChatCheckResult(chat_obj.name, message=message)

    def _is_telegram_user_still_employed(self, telegram_username):
        """
        Check is there is an employee with the given telegram username and that he/she is still employed by Yandex

        :param telegram_username: Telegram username
        :return: a tuple (ok, message); ok is True if staff user exists and is employed, False otherwise; message
        contains some explanatory info
        """
        staff_login = self.staff_client.get_person_by_telegram_login(telegram_username)

        if not staff_login:
            return False, "cannot find staff login for telegram username '{}'".format(telegram_username)

        ok, msg = self._is_staff_login_still_employed(staff_login)

        if ok:
            msg = "private Telegram chat with {}@ ({}) {}".format(staff_login, telegram_username, msg)

        return ok, msg

    def _is_staff_login_still_employed(self, staff_login):
        try:

            staff_info = self.staff_client.get_person_profile_property(staff_login, "official.is_dismissed")

            if staff_info.get("error_message") == self.STAFF_NOT_FOUND_MESSAGE:
                return True, "{} - user not found. Is this a mailing list?".format(staff_login)

            logging.debug("Got staff employment info for %s: %s", staff_login, staff_info)

            is_dismissed = staff_info['official']['is_dismissed']

            if is_dismissed:
                return False, "user {}@ is no longer employed by Yandex".format(staff_login)

        except (requests_wrapper.EmptyResponseError, IndexError, KeyError, AttributeError):
            return False, "unable to check staff login {}@".format(staff_login)

        return True, ""

    def _try_to_fix_telegram_write_permission_problem(self, chat_obj):
        """RMDEV-2204"""

        import grpc
        import telegram.error

        if not self.Parameters.resolve_possible_telegram_chat_problems:
            return chat_check_result.ChatCheckResult(
                chat_obj.name,
                success=False,
                message="Bot is not allowed to send messages to this chat",
            )

        logging.info("Attempting to send a message to %s", chat_obj)

        try:
            response = self.telegram_bot.send_message(
                chat_id=chat_obj.chat_id,
                text=self.TG_BOT_TEST_MESSAGE,
            )

            logging.debug("Telegram response: %s", response)

        except telegram.error.ChatMigrated as cm:

            new_chat_id = str(cm.new_chat_id)

            logging.info(
                "Going to update component chat %s: chat_id %s -> %s",
                chat_obj,
                chat_obj.chat_id,
                new_chat_id,
            )

            chat_obj.chat_id = new_chat_id

            try:
                response = self.notification_client.update_component_chat(request=chat_obj)
                logging.debug(response)
            except grpc.RpcError:
                logging.exception("Unable to update component chat")
                return chat_check_result.ChatCheckResult(
                    chat_obj.name,
                    success=False,
                    message="Bot cannot text to this chat. Tried to update chat ID - unsuccessfully",
                )

            return chat_check_result.ChatCheckResult(
                chat_obj.name,
                message="Updated component chat successfully",
            )

        except telegram.error.TelegramError as tg_err:

            return chat_check_result.ChatCheckResult(
                chat_obj.name,
                success=False,
                message="Bot cannot text to this chat: {}".format(tg_err.message)
            )

        return chat_check_result.ChatCheckResult(
            chat_obj.name,
            success=False,
            message="Bot cannot text to this chat and we don't known how to fix it..."
        )

    def _report_component_state(self, component_name, messages, ok=False):

        from release_machine.common_proto import events_pb2, component_state_pb2
        from release_machine.public import events as events_public
        from release_machine.release_machine.proto.structures import message_pb2

        logging.info("Going to post component state (notification chats) for %s", component_name)

        now = tu.datetime_utc()
        now_ts = tu.datetime_to_timestamp(now)

        logging.debug("Now: %s", now)

        event = events_pb2.EventData(
            general_data=events_pb2.EventGeneralData(
                hash=events_public.get_event_hash(
                    now.isoformat(),
                    component_name,
                    "UpdateComponentState",
                    "notification_chats",
                ),
                component_name=component_name,
                referrer="sandbox_task:{}".format(self.id),
            ),
            task_data=events_pb2.EventSandboxTaskData(
                task_id=self.id,
                status=self.status,
                created_at=self.created.isoformat(),
                updated_at=self.updated.isoformat(),
            ),
            update_component_state_data=events_pb2.UpdateComponentStateData(
                component_state=component_state_pb2.ComponentState(
                    timestamp=int(now_ts),
                    referrer="{}:{}".format(self.type, self.id),
                    notification_chats=component_state_pb2.ComponentSectionState(
                        status=(
                            component_state_pb2.ComponentSectionState.Status.OK if ok
                            else component_state_pb2.ComponentSectionState.Status.CRIT
                        ),
                        info=component_state_pb2.ComponentSectionInfo(
                            title=("Notification chats errors" if not ok else "All OK"),
                            description="\n".join(messages),
                        ),
                    ),
                ),
            ),
        )

        logging.info("Constructed event: %s", event)

        self.rm_event_client.post_proto_events(
            request=message_pb2.PostProtoEventsRequest(
                events=[event],
            ),
        )

    def _prepare_report(self):

        from library.python import resource as py_resource

        logging.info("Preparing report")

        report_template_path = self.get_resource_path("report.jinja2")

        template = py_resource.find(report_template_path)

        if not template:
            self.set_info("Unable to render report: no such file {}".format(report_template_path))
            return

        jinja_index_template = jinja2.Template(template.decode('utf-8'))
        report = jinja_index_template.render(results=self.Context.results)
        report = re.sub(r"\n[\n ]+", "\n", report)

        self.Context.report_html = report
        self.Context.save()

    @staticmethod
    def get_resource_path(resource_file_name):
        return os.path.join(os.path.dirname(__file__), "templates", resource_file_name)
