import inspect
import jinja2
import logging
import os
import re
import sys

from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.projects.common import binary_task
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import time_utils as tu
from sandbox.projects.common import requests_wrapper
from sandbox.projects.common.search import bugbanner2 as bb2
from sandbox.projects.release_machine import resources as rm_res
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.components.configs import all as all_cfg
from sandbox.projects.release_machine.components.config_core import config_serializer
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import task_env
from sandbox.projects.release_machine.helpers import responsibility_helper
from sandbox.projects.release_machine.helpers import staff_helper


class ResultKeys:
    STATISTICS_CONFIG = "statistics_config"
    NOTIFICATION_CONFIG = "notification_config"
    COMPONENT_UPDATES = "component_updates"
    CONFIG_SERIALIZED = "config_serialized"


REPORT_FILENAME = "report.html"
NOTIFICATION_COUNT_MAX = 5


if sys.version_info[0:2] >= (3, 8):  # python 3.8 or higher
    from functools import cached_property
else:
    from sandbox.projects.common.decorators import memoized_property as cached_property


class ReleaseMachineConfigCrawler(bb2.BugBannerTask, binary_task.LastBinaryTaskKosherReleaseMixin):
    """Submit changes in RM components' config to RM backend."""

    _component_names_to_process = None
    _rm_model_client = None
    _rm_notification_client = None
    _rm_event_client = None

    class Requirements(task_env.TinyRequirements):
        disk_space = 1024  # 1Gb

    class Context(sdk2.Task.Context):
        results = {}
        config_paths = {}
        report_resource_link = None
        report_html = ""
        notification_count = 0

    class Parameters(binary_task.LastBinaryReleaseParameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        kill_timeout = 15 * 60  # 15 min
        component_name_filter = sdk2.parameters.String("Component name filter")
        rm_host = sdk2.parameters.String(
            "Release Machine service address",
            default_value=rm_const.Urls.RM_HOST,
        )
        light_mode = sdk2.parameters.Bool("When True runs in simplified mode (no subtasks)")
        update_declarative_notifications = sdk2.parameters.Bool(
            "When True updates declarative notifications on RM service based on notification config",
            default_value=True,
        )

    @cached_property
    def component_names_to_process(self):
        name_filter_re = re.compile(self.Parameters.component_name_filter or ".*", re.IGNORECASE)
        return list(sorted(filter(name_filter_re.match, all_cfg.get_all_names())))

    @property
    def total_of_components_to_process(self):
        return len(self._component_names_to_process)

    @property
    def number_of_components_processed(self):
        return len(self.Context.results)

    @property
    def percent_of_components_processed(self):
        return 100. * self.number_of_components_processed / self.total_of_components_to_process

    @cached_property
    def rm_model_client(self):
        from release_machine.release_machine.services.release_engine.services.Model import ModelClient
        return ModelClient.from_address(self.Parameters.rm_host)

    @cached_property
    def rm_notification_client(self):
        from release_machine.release_machine.services.release_engine.services.Notification import NotificationClient
        return NotificationClient.from_address(self.Parameters.rm_host)

    @cached_property
    def rm_event_client(self):
        from release_machine.release_machine.services.release_engine.services.Event import EventClient
        return EventClient.from_address(self.Parameters.rm_host)

    @cached_property
    def staff_client(self):
        return staff_helper.StaffApi(rm_sec.get_rm_token(self))

    @cached_property
    def event_referrer(self):
        return "sandbox_task:{}".format(self.id)

    @sdk2.header()
    def header(self):
        if not self.Context.report_resource_link:
            return self._render_progress()
        return lb.format_link(lb.LinkType.href, self.Context.report_resource_link, REPORT_FILENAME)

    @sdk2.report(title="Crawler Report")
    def get_report_html(self):
        return self.Context.report_html or "<h3>Report is not ready</h3>"

    def on_save(self):
        binary_task.LastBinaryTaskKosherReleaseMixin.on_save(self)
        bb2.BugBannerTask.on_save(self)

    def on_execute(self):
        binary_task.LastBinaryTaskKosherReleaseMixin.on_execute(self)
        bb2.BugBannerTask.on_execute(self)

        for component_name in self.component_names_to_process:
            try:
                self._process_component(component_name)
            except Exception as exc:
                msg = "Unable to process component {}".format(component_name)
                eh.log_exception(msg, exc, task=self)
                self.set_info(msg)

        report_url = self._prepare_report()

        if report_url:
            self.set_info("Task report available: {}".format(report_url))
        else:
            self.set_info("Unable to build report")

        self._process_deleted_components()

        self._on_after_everything_processed()

    def on_failure(self, prev_status):

        self._notify_authorities_on_error("- task FAILED. See {report_link}".format(
            report_link=lb.format_link(
                lb.LinkType.href,
                self.Context.report_resource_link,
                REPORT_FILENAME,
            ),
        ))

    def _process_component(self, component_name):
        c_cfg = all_cfg.get_config(component_name)
        component_results = {
            ResultKeys.COMPONENT_UPDATES: self._process_component_updates(c_cfg),
            ResultKeys.STATISTICS_CONFIG: self._process_statistics_page_config(c_cfg),
            ResultKeys.NOTIFICATION_CONFIG: self._process_notification_config(c_cfg),
            ResultKeys.CONFIG_SERIALIZED: self._process_component_config_serialization(c_cfg),
        }

        self.Context.results[component_name] = component_results
        self.Context.config_paths[component_name] = self._get_component_config_path(c_cfg)
        self.Context.save()

        try:

            self._process_component_processing_results(
                component_name=component_name,
                component_results=component_results,
            )

        except Exception as e:
            eh.log_exception("Unable to perform component_state update for {}".format(component_name), e)

    def _get_component_config_path(self, c_cfg):
        try:
            return inspect.getfile(c_cfg.__class__)
        except Exception:
            logging.exception("Unable to get config path")
            return ""

    def _process_component_processing_results(self, component_name, component_results):

        logging.info("Going to process component processing results")

        from release_machine.common_proto import component_state_pb2
        from release_machine.common_proto import events_pb2
        from release_machine.public import events as events_public
        from release_machine.release_machine.proto.structures import message_pb2

        request = message_pb2.PostProtoEventsRequest(
            events=[
                events_pb2.EventData(
                    general_data=events_pb2.EventGeneralData(
                        hash=events_public.get_event_hash(
                            self.id,
                            tu.datetime_utc_iso(),
                            "UpdateComponentState",
                            "config_static",
                        ),
                        component_name=component_name,
                        referrer=self.event_referrer,
                    ),
                    task_data=self._get_event_sandbox_task_data(),
                    update_component_state_data=events_pb2.UpdateComponentStateData(
                        component_state=component_state_pb2.ComponentState(
                            timestamp=int(tu.datetime_utc().timestamp()),
                            referrer="config_crawler:{}".format(self.event_referrer),
                            config_static=component_state_pb2.ComponentSectionState(
                                status=(
                                    component_state_pb2.ComponentSectionState.Status.OK
                                    if component_results.get(ResultKeys.CONFIG_SERIALIZED, False)
                                    else component_state_pb2.ComponentSectionState.Status.CRIT
                                ),
                            ),
                        ),
                    ),
                ),
            ],
        )

        logging.info("Post component state event request: %s", request)

        response = self._perform_request_no_exception(request, self.rm_event_client.post_proto_events)

        logging.info("Component state update event response: %s", response)

    def _process_component_config_serialization(self, c_cfg):

        from release_machine.common_proto import events_pb2
        from release_machine.public import events as events_public
        from release_machine.release_machine.proto.structures import message_pb2

        try:
            config_message = config_serializer.config2proto(c_cfg)
        except Exception as exc:
            eh.log_exception("Unable to serialize config for {}".format(c_cfg.name), exc)
            return False

        request = message_pb2.PostProtoEventsRequest(
            events=[
                events_pb2.EventData(
                    general_data=events_pb2.EventGeneralData(
                        hash=events_public.get_event_hash(
                            self.id,
                            tu.datetime_utc_iso(),
                            "ComponentConfigUpdate",
                        ),
                        component_name=config_message.name,
                        referrer=self.event_referrer,
                    ),
                    task_data=self._get_event_sandbox_task_data(),
                    component_config_update_data=events_pb2.ComponentConfigUpdateData(
                        config=config_message,
                    ),
                ),
            ],
        )

        response = self._perform_request_no_exception(request, self.rm_event_client.post_proto_events)

        return response is not None

    def _process_notification_config(self, c_cfg):
        from release_machine.release_machine.proto.structures import message_pb2

        if not c_cfg.notify_cfg.notifications:

            request = message_pb2.PostNotificationRequest(
                notifications=[
                    message_pb2.NotificationConfig(component_name=c_cfg.name),
                ],
            )

        else:

            notifications = []

            try:
                for notification in c_cfg.notify_cfg.notifications:
                    notification_proto = notification.to_protobuf()
                    notification_proto.component_name = c_cfg.name
                    notifications.append(notification_proto)
            except Exception:
                logging.exception("Unable to process %s's declarative notifications", c_cfg.name)
                return False

            request = message_pb2.PostNotificationRequest(notifications=notifications)

        if not self.Parameters.update_declarative_notifications:
            logging.info("Skipping notification update")
            return True

        response = self._perform_request_no_exception(
            request,
            self.rm_notification_client.post_notification_declarative_config,
        )

        return response is not None

    def _process_statistics_page_config(self, c_cfg):
        from release_machine.release_machine.proto.structures import message_pb2, table_pb2

        logging.debug("Processing statistics page config for %s", c_cfg.name)

        request = message_pb2.UpdateComponentStatisticsChartsRequest(
            component_name=c_cfg.name,
            statistics_charts_settings=table_pb2.StatisticsChartsSettings(
                charts=c_cfg.release_viewer_cfg.statistics_page_charts,
            ),
        )

        response = self._perform_request_no_exception(
            request,
            self.rm_model_client.update_component_statistics_charts_settings,
        )

        return response is not None

    def _process_component_updates(self, c_cfg):

        import grpc
        from release_machine.release_machine.proto.structures import message_pb2
        from release_machine.common_proto import events_pb2
        from release_machine.public import events as events_public
        from search.martylib.core.exceptions import MaxRetriesReached

        try:

            self.rm_model_client.get_component(
                request=message_pb2.ComponentRequest(component_name=c_cfg.name),
                # hooks={
                #     grpc.RpcError: self._get_new_component_hook(c_cfg),
                # },
                # hooks can be used after HORADRIC-129
            )

        except grpc.RpcError as rpc_error:
            # todo (ilyaturuntaev): refactor after HORADRIC-129
            hook = self._get_new_component_hook(c_cfg)
            if not hook(rpc_error):
                return False

        except MaxRetriesReached:
            logging.exception("Error while requesting component %s", c_cfg.name)
            return False

        try:
            responsible_for_component = responsibility_helper.get_responsible_user_login(c_cfg.responsible)
        except Exception:
            logging.exception("Unable to get %s's responsible", c_cfg.name)
            return False

        try:
            trunk_te_db = getattr(c_cfg.testenv_cfg, "trunk_db", "")
            self.rm_event_client.post_proto_events(message_pb2.PostProtoEventsRequest(
                events=[
                    events_pb2.EventData(
                        general_data=events_pb2.EventGeneralData(
                            hash=events_public.get_event_hash(
                                c_cfg.name,
                                responsible_for_component,
                                trunk_te_db,
                                rm_const.ReleaseCycleType.to_str(c_cfg.release_cycle_type),
                            ),
                            component_name=c_cfg.name,
                            referrer=u"sandbox_task:{}".format(self.id),
                        ),
                        task_data=self._get_event_sandbox_task_data(),
                        component_update_data=events_pb2.ComponentUpdateData(
                            responsible=responsible_for_component,
                            display_name=c_cfg.display_name,
                            release_cycle=rm_const.ReleaseCycleType.to_str(c_cfg.release_cycle_type),
                            te_db=trunk_te_db,
                        ),
                    ),
                ]
            ))
        except grpc.RpcError:
            logging.exception("%s: component update might have failed", c_cfg.name)
            return False

        if not self._is_user_employed(responsible_for_component):
            self._notify_authorities_on_error(
                "{staff_login} is no longer a member of Yandex team "
                "while still being responsible for {component_name}".format(
                    staff_login=lb.staff_link(responsible_for_component, link_type=lb.LinkType.href),
                    component_name=lb.rm_ui_link(c_cfg.name, link_type=lb.LinkType.href),
                )
            )

        return True

    def _get_new_component_hook(self, c_cfg):

        import grpc

        def new_component_hook(exception, **kwargs):

            if exception.code() == grpc.StatusCode.NOT_FOUND:
                logging.info("New component: %s", c_cfg.name)
                return self._process_new_component__prepare_te_db(c_cfg)

            logging.exception("Error while requesting component %s", c_cfg.name)
            return False

        return new_component_hook

    def _process_new_component__prepare_te_db(self, c_cfg):

        if self.Parameters.light_mode:
            self.set_info(
                "The task is running in light mode "
                "so PrepareRmComponentEnvironment for {} is not going to be created".format(
                    c_cfg.name,
                )
            )
            return True

        if c_cfg.release_cycle_type == rm_const.ReleaseCycleType.CI:
            self.set_info("No TestEnv database required for RMCI component %s", c_cfg.name)
            return True

        prepare_component_task = self.server.task(
            children=True,
            type="PREPARE_RM_COMPONENT_ENVIRONMENT",
            description="Create new component {}".format(c_cfg.name),
            owner=self.owner,
            priority=ctt.Priority(ctt.Priority.Class.SERVICE, ctt.Priority.Subclass.HIGH),
            custom_fields=[
                {
                    "name": "component_name",
                    "value": c_cfg.name,
                },
            ],
        )

        logging.info("PrepareRmComponentTask for %s created: %s", c_cfg.name, prepare_component_task["id"])

        self.server.batch.tasks.start.update([prepare_component_task["id"]])

        return True

    def _process_deleted_components(self):

        from release_machine.release_machine.proto.structures import message_pb2
        from release_machine.release_machine.proto.structures import table_pb2

        components_from_service = self._perform_request_no_exception(
            message_pb2.Dummy(),
            self.rm_model_client.get_components,
        )

        if not components_from_service:
            logging.error("Unable retrieve components from %s", self.Parameters.rm_host)
            return False

        service_active_component_names = {
            component.name for component in components_from_service.components
            if component.status == table_pb2.Component.ComponentStatus.ACTIVE
        }

        logging.debug(
            "Active components retrieved from %s: %s",
            self.Parameters.rm_host,
            service_active_component_names,
        )

        config_component_names = all_cfg.get_all_names()

        logging.debug(
            "Components found in Release Machine configs: %s",
            config_component_names,
        )

        components_to_be_deleted = service_active_component_names - config_component_names

        logging.info("Going to DELETE the following components: %s", components_to_be_deleted)

        for component_name in components_to_be_deleted:
            self.rm_model_client.delete_component(
                message_pb2.DeleteComponentRequest(
                    component_name=component_name,
                    irreversible=False,
                ),
            )
            self.set_info("{} component deleted".format(component_name))

        return True

    def _perform_request_no_exception(self, request, client_method):
        """
        Simply calls `client_method` passing `request` to it suppressing any exceptions

        :param request: Request object for `client_method`
        :param client_method: any GRPC client method
        :return: response object or `None` if any error occurred
        """
        try:
            return client_method(request)
        except Exception:
            logging.exception("Request FAILED: %s with %s", client_method, request)
            return

    def _render_progress(self):

        progress_html = ""

        try:

            from library.python import resource as py_resource

            report_template_path = self.get_resource_path("progress.jinja2")
            template = py_resource.find(report_template_path)
            jinja_template = jinja2.Template(template.decode('utf-8'))

            progress_html = jinja_template.render(percent=self.percent_of_components_processed)

        except Exception:
            logging.exception("Progress bar rendering failed")

        return progress_html

    def _prepare_report(self):

        from library.python import resource as py_resource

        all_keys = [getattr(ResultKeys, key) for key in dir(ResultKeys) if not key.startswith('_') and key.isupper()]

        report_template_path = self.get_resource_path("report.jinja2")

        template = py_resource.find(report_template_path)

        if not template:
            self.set_info("Unable to render report: no such file {}".format(report_template_path))
            return

        jinja_index_template = jinja2.Template(template.decode('utf-8'))
        report = jinja_index_template.render(
            keys=all_keys,
            results=self.Context.results,
            config_paths=self.Context.config_paths,
        )

        fu.write_file(
            REPORT_FILENAME,
            report,
        )
        resource = rm_res.RELEASE_MACHINE_CONFIG_CRAWLER_REPORT(self, "Config Crawler Report", REPORT_FILENAME)
        sdk2.ResourceData(resource).ready()

        self.Context.report_resource_link = resource.http_proxy
        self.Context.report_html = report
        self.Context.save()

        return resource.http_proxy

    def _on_after_everything_processed(self):
        overall_result = all(all(component_result.values()) for component_result in self.Context.results.values())

        eh.ensure(
            overall_result,
            "Config processing FAILED for some components. "
            "Visit {} to see a general report. "
            "Check task logs for more detailed information.".format(
                self.Context.report_resource_link,
            )
        )

    def _is_user_employed(self, staff_login):
        """Check is the given user is still employed by Yandex"""
        try:

            return self.staff_client.is_user_employed(staff_login)

        except (requests_wrapper.EmptyResponseError, IndexError, KeyError, AttributeError):
            logging.exception("Unable to check if %s is still employed", staff_login)
            return False

    def _notify_authorities_on_error(self, message):
        from release_machine.common_proto import events_pb2
        from release_machine.public import events as events_public
        from release_machine.release_machine.proto.structures import message_pb2

        self.Context.notification_count += 1
        self.Context.save()

        # todo (ilyaturuntaev): Remove after RMDEV-666
        if self.Context.notification_count > NOTIFICATION_COUNT_MAX:
            logging.error(
                "Prevent sending error notification since max notification count reached (%s). Original message: %s",
                NOTIFICATION_COUNT_MAX,
                message,
            )
            return

        self.rm_event_client.post_proto_events(message_pb2.PostProtoEventsRequest(
            events=[
                events_pb2.EventData(
                    general_data=events_pb2.EventGeneralData(
                        hash=events_public.get_event_hash(
                            self.id,
                            tu.datetime_utc_iso(),
                            rm_const.RELEASE_MACHINE_ERROR_REPORT,
                        ),
                        component_name="release_machine",
                        referrer="sandbox_task:{}".format(self.id),
                    ),
                    custom_message_data=events_pb2.CustomMessageData(
                        message="{task_link} {message}".format(
                            task_link=lb.task_link(self.id, self.type),
                            message=message,
                        ),
                        condition_tag=rm_const.RELEASE_MACHINE_ERROR_REPORT,
                    ),
                ),
            ],
        ))

    def _get_event_sandbox_task_data(self):

        from release_machine.common_proto import events_pb2

        return events_pb2.EventSandboxTaskData(
            task_id=self.id,
            status=self.status,
            created_at=self.created.isoformat(),
            updated_at=self.updated.isoformat(),
        )

    @staticmethod
    def get_resource_path(resource_file_name):
        return os.path.join(os.path.dirname(__file__), "templates", resource_file_name)
