# -*- coding: utf-8 -*-
import collections
import hashlib
import json
import logging
import os
import textwrap
import datetime
import requests
from concurrent.futures import ThreadPoolExecutor

import sandbox.common.rest as sb_rest
import sandbox.projects.common.vcs.arc as arc_client
import sandbox.projects.common.sdk_compat.task_helper as th
import sandbox.projects.common.search.bugbanner2 as bb2
import sandbox.projects.common.testenv_client as teh
import sandbox.projects.common.binary_task as binary_task
import sandbox.projects.common.decorators as decorators
import sandbox.projects.common.error_handlers as eh
import sandbox.projects.common.link_builder as lb
import sandbox.projects.common.time_utils as tu
import sandbox.projects.release_machine.client as rm_client
import sandbox.projects.release_machine.components.all as rmc
import sandbox.projects.release_machine.components.components_info as rm_ci
import sandbox.projects.release_machine.components.configs as configs
import sandbox.projects.release_machine.components.configs.release_machine_test as rm_test
import sandbox.projects.release_machine.core as rm_core
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.helpers.startrek_helper as rm_st
import sandbox.projects.release_machine.helpers.staff_helper as staff_helper
import sandbox.projects.release_machine.helpers.wiki_helper as rm_wiki
import sandbox.projects.release_machine.input_params2 as rm_params
import sandbox.projects.release_machine.tasks.base_task as rm_bt
import sandbox.sdk2 as sdk2
from sandbox.common import itertools as scit
from sandbox.common.types import task as ctt
from sandbox.projects.common.nanny.client import NannyClient
from sandbox.projects.release_machine import rm_notify
from sandbox.projects.release_machine import security as rm_sec
from sandbox.sdk2.vcs import svn

try:
    from release_machine.common_proto import events_pb2, component_state_pb2
    from release_machine.public import events as events_public
except ImportError:
    pass


RESPONSIBLE_FOR_COMPONENT = "responsible for component"
RESPONSIBLE_FOR_RELEASE = "responsible for release"


@rm_notify.notify2()
class ReleaseMachineFunctionalTest(binary_task.LastBinaryTaskRelease, rm_bt.ComponentErrorReporter, bb2.BugBannerTask):

    class Requirements(task_env.StartrekRequirements):
        disk_space = 5 * 1024  # 5 Gb

    class Parameters(rm_params.BaseReleaseMachineParameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        kill_timeout = 30 * 60  # 30 min

    class Context(sdk2.Task.Context):
        failures = []
        components_failures = {}

    def _add_component_failure(self, component_name, error_message):
        self.Context.components_failures.setdefault(component_name, []).append(error_message)

    def _add_common_failure(self, error_message):
        self.Context.failures.append(error_message)

    @decorators.memoized_property
    def staff_client(self):
        return staff_helper.StaffApi(self.rm_token)

    @decorators.memoized_property
    def rm_token(self):
        return rm_sec.get_rm_token(self)

    @decorators.memoized_property
    def rm_api_client(self):
        return rm_client.RMClient()

    @decorators.memoized_property
    def wiki_client(self):
        return rm_wiki.WikiApi(self.rm_token)

    @decorators.memoized_property
    def nanny_client(self):
        return NannyClient(api_url=rm_const.Urls.NANNY_BASE_URL, oauth_token=self.rm_token)

    @sdk2.header()
    def header(self):
        progress = '[IN PROGRESS] ' if self.status == ctt.TaskStatus.EXECUTING else ''
        if not self.Context.failures:
            return '{}<b style="color: #009f00">NO FAILURES</b>'.format(progress)

        failures_str = '<br />\n'.join(self.Context.failures)
        return '{}<b style="color: red">FAILURES:</b><br/>{}'.format(progress, failures_str)

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)
        self.add_bugbanner(bb2.Banners.ReleaseMachine)

        all_component_names = rmc.get_component_names()
        c_infos = [rmc.COMPONENTS[c_name]() for c_name in all_component_names]

        self.check_ex_employees(c_infos)
        self.check_robot_access_to_st_queue(c_infos)
        self.check_no_lint_in_rm_dirs()

        with ThreadPoolExecutor(8) as executor:
            executor.map(self.test_one_component, c_infos)

        self.find_broken_dirs(c_infos)
        self.check_token_access(c_infos)
        self.check_sdk_compat()

        wrong_wiki_page_owners = []
        for c_info in c_infos:
            wrong_wiki_page_owners.extend(self.check_resp_for_component_is_wiki_page_owner(c_info))
        if wrong_wiki_page_owners:
            self.set_info(
                "These components\n{}\nhave two different people as 'wiki page owner' and 'responsible'.\n"
                "This may lead to problems with access to wiki pages!".format([i.name for i in wrong_wiki_page_owners])
            )

        # We send email to responsible for component once a week on Friday in the morning
        should_send_email = datetime.datetime.today().weekday() == 4 and 10 <= datetime.datetime.today().hour <= 12
        for c_name, errors in self.Context.components_failures.items():
            if not errors:
                continue
            errors_string = "\n\n".join(errors)
            c_info = rmc.COMPONENTS[c_name]()
            responsible_for_comp = c_info.get_responsible_for_component(self.rm_token)
            if RESPONSIBLE_FOR_COMPONENT in errors_string:
                self.set_info(
                    "Responsible {responsible} for component {c_name} is no longer a member of Yandex team, "
                    "need to change it".format(
                        responsible=responsible_for_comp, c_name=c_name,
                    )
                )
            else:
                if should_send_email:
                    self.email_responsible_for_component(
                        c_info=c_info, errors_string=errors_string, responsible_for_comp=responsible_for_comp,
                    )

        for component_name in all_component_names:
            errors = self.Context.components_failures.get(component_name, [])

            self._report_component_state(
                component_name=component_name,
                messages=errors,
                ok=(not errors),
            )

        eh.ensure(
            not self.Context.components_failures and not self.Context.failures,
            "Some test cases are broken"
        )

    def check_ex_employees(self, c_infos):
        user_roles = collections.defaultdict(list)
        with ThreadPoolExecutor(16) as executor:
            for user_roles_for_component in executor.map(self._get_user_roles, c_infos):
                for staff_login, roles in user_roles_for_component.items():
                    user_roles[staff_login].extend(roles)
        logging.info("Got %s user roles", len(user_roles))
        logging.debug("User roles:\n%s", json.dumps(user_roles, indent=2))
        employment_statuses = (
            self.staff_client.get_current_employment_status(chunk) for chunk in scit.chunker(user_roles.keys(), 45)
        )  # use chunks because of pagination and query/response size limits
        ex_employees = [
            user
            for employment_status_chunk in employment_statuses
            for user, employed in employment_status_chunk.items() if not employed
        ]
        logging.info("Ex employees: %s", ex_employees)
        if ex_employees:
            for user in ex_employees:
                roles = user_roles[user]
                for component_name, role_name in roles:
                    self._add_component_failure(
                        component_name,
                        "{user} is no longer a member of Yandex team while still having following role: {role}".format(
                            user=user,
                            role=role_name,
                        )
                    )
            self.Context.save()

    def check_no_lint_in_rm_dirs(self):
        self._arc = arc_client.Arc(secret_name=rm_const.COMMON_TOKEN_NAME, secret_owner=rm_const.COMMON_TOKEN_OWNER)
        with self._arc.mount_path(None, None, fetch_all=False, extra_params=["--vfs-version", "2"]) as arcadia_src_dir:
            whole_files = os.walk(os.path.join(arcadia_src_dir, "sandbox/projects/release_machine"))
            ya_make_files = []
            for root, _, files in whole_files:
                if "ya.make" in files:
                    ya_make_files.append(os.path.join(arcadia_src_dir, root, "ya.make"))
            for path in ya_make_files:
                with open(path, "r") as ya_make_file:
                    ya_make = ya_make_file.read()
                    if "NO_LINT" in ya_make:
                        self._add_common_failure(
                            "Got `NO_LINT()` marker in file {path}, it is forbidden".format(
                                path=path.split(arcadia_src_dir)[1],
                            )
                        )

    def email_responsible_for_component(self, c_info, errors_string, responsible_for_comp):
        """
        Send email to responsible for component with errors.
        :param c_info: Component info class instance
        :param errors_string: Errors relevant to this component
        :param responsible_for_comp: Responsible for component
        :return:
        """

        from release_machine.common_proto import events_pb2, notifications_pb2

        email_body = textwrap.dedent(
            """
            Dear {responsible}!

            There are some errors in your component {comp_name} config below, please fix them.
            In case of any problems contact Release Machine team in support chat {chat}.

            Errors:
            {errors_string}

            Sent by SB: {url}
            """
        ).format(
            responsible=responsible_for_comp,
            comp_name=c_info.name,
            errors_string=errors_string,
            chat=lb.WIKILINK_TO_ITEM.format(link="https://nda.ya.ru/3UXG4F", name="chat"),
            url=lb.task_link(task_id=self.id),
        )

        logging.debug(email_body)

        event = events_pb2.EventData(
            general_data=events_pb2.EventGeneralData(
                hash=hashlib.md5(
                    rm_const.EVENT_HASH_SEPARATOR.join(
                        [str(datetime.datetime.today()), c_info.name]
                    ).encode("utf-8")
                ).hexdigest(),
                component_name=c_info.name,
                referrer="sandbox_task:{}".format(self.id),
            ),
            task_data=events_pb2.EventSandboxTaskData(
                task_id=self.id,
                status=self.status,
                created_at=self.created.isoformat(),
                updated_at=self.updated.isoformat(),
            ),
            custom_message_data=events_pb2.CustomMessageData(
                message="{task_link}".format(
                    task_link=lb.task_link(self.id, self.type),
                ),
                condition_tag=rm_const.RELEASE_MACHINE_ERROR_REPORT,
            ),
            custom_notifications=[
                notifications_pb2.NotificationBundle(
                    email_data=notifications_pb2.EmailEventNotificationData(
                        addr=responsible_for_comp + "@yandex-team.ru",
                        subject="Errors list for component config {c_name}".format(c_name=c_info.name),
                        body=email_body.encode("utf-8"),
                    ),
                )
            ]
        )

        self.rm_api_client.post_proto_events([event, ])

    def test_one_component(self, c_info):
        results = []
        results.extend(self.test_last_branch_and_tag(c_info))
        results.extend(self.test_get_followers(c_info))
        results.extend(self.test_deploy_info(c_info))
        results.extend(self.check_robot_access_to_wiki(c_info))
        results.extend(self.test_dirs_is_iterable(c_info))
        results.extend(self.check_task_owner(c_info))
        results.extend(self.check_yappy_cfg(c_info))
        results.extend(self.check_trunk_db(c_info))

        errors = [result.result for result in results if not result.ok]

        for error in errors:
            self._add_component_failure(c_info.name, error)

        if errors:
            self._report_error(component_name=c_info.name, message='Component is broken', exception=errors[0])

        self.Context.save()

    def check_trunk_db(self, c_info):
        results = []
        if c_info.testenv_cfg:
            try:
                response = self.rm_api_client.get_component(c_info.name)
                if response:
                    exists = teh.TEClient.testenv_database_exists(c_info.testenv_cfg__trunk_db)
                    if not exists:
                        results.append(rm_core.Error("No trunk database '{}' for component: '{}'".format(
                            c_info.testenv_cfg__trunk_db, c_info.name
                        )))
            except Exception as exc:
                results.append(rm_core.Error("Got exception in `check_trunk_db` {}".format(exc)))
        return results

    @staticmethod
    def test_dirs_is_iterable(c_info):
        results = []
        if not isinstance(c_info, rm_ci.mixin.Changelogged):
            return results
        if not isinstance(c_info.changelog_cfg__observed_paths, collections.Iterable):
            results.append(rm_core.Error("Changelog observed_paths are not iterable"))
        if not isinstance(c_info.changelog_cfg__ya_make_targets, collections.Iterable):
            results.append(rm_core.Error("Changelog ya_make_targets are not iterable"))
        return results

    def find_broken_dirs(self, c_infos):
        broken_paths = set()
        healthy_paths = set()

        for c_info in c_infos:
            if not isinstance(c_info, rm_ci.mixin.Changelogged):
                continue

            all_paths = c_info.changelog_cfg__observed_paths + c_info.changelog_cfg__ya_make_targets
            # Some components redefine `svn_cfg__main_url` therefore we don't need to add `arcadia` in these paths
            if c_info.svn_cfg__main_url == c_info.svn_cfg__trunk_url:
                all_paths = [i if i.startswith("arcadia") else os.path.join("arcadia", i) for i in all_paths]
            for path in all_paths:

                if path in healthy_paths:
                    continue

                if (
                    path in broken_paths
                    or not svn.Arcadia.check(os.path.join(c_info.svn_cfg__main_url, path))
                ):
                    self._report_error(c_info.name, message="Cannot resolve changelog path {}".format(path))
                    self._add_component_failure(c_info.name, "Bad changelog path {}".format(path))
                    broken_paths.add(path)
                    continue

                healthy_paths.add(path)

        if broken_paths:
            self.Context.save()

    def check_task_owner(self, c_info):
        if c_info.is_branched:
            return []

        owner = c_info.testenv_cfg__trunk_task_owner
        if owner not in [
            'SEARCH-RELEASERS',
            'ARC_AUTOCOMMIT_USERS',
        ]:
            return []

        if owner == 'SEARCH-RELEASERS' and c_info.name in [
            # This list is signed off by mvel@.
            # Please do not add component names here without mvel@ permission,
            # or you'll pay for it with your own hardware quota.
            'base',
            'clickdaemon',
            'cores',
            'findurl',
            'itditp_middle',
            'middle',
            'prs_ops',
            'release_machine',
            'release_machine_test',
            'sawmill',
            'scraper_over_yt',
            'service_controller',
            'setrace',
            'src_setup',
            'yp_cauth_export',
            'yp_dns',
            'yp_export',
            'yp_idm_role_provider',
            'yp_inet_mngr',
            'yp_master',
            'yp_service_discovery',
        ]:
            return []

        self.set_info(
            "Component {name} uses group {group}".format(
                name=c_info.name,
                group=c_info.testenv_cfg__trunk_task_owner,
            )
        )
        return []

    @staticmethod
    def has_push_scheme(c_info):
        """
        Check whether jobs with substring `__PUSH` in component JobGraph.
        :param c_info: Component info class instance
        :return: bool
        """
        logging.debug("Check component {c_name} has push-scheme in JobGraph".format(c_name=c_info.name))
        jobs = c_info.testenv_cfg__job_graph__graph
        for job in jobs:
            if "__PUSH" in job.job_params.get("job_name_parameter", ""):
                logging.debug("Got push-scheme")
                return True
        return False

    def test_deploy_info(self, c_info):
        results = []

        def check_access(
            auth_attrs,
            logins=frozenset([rm_const.ROBOT_RELEASER_USER_NAME, "robot-morty"]),
            groups=frozenset([rm_const.RM_ABC_GROUP])
        ):
            return (
                any([login in logins for login in auth_attrs["logins"]]) or
                any([group in groups for group in auth_attrs["groups"]])
            )
        try:
            if c_info.releases_cfg__deploy_system not in [
                rm_const.DeploySystem.nanny, rm_const.DeploySystem.nanny_push
            ]:
                logging.info("Deploy system: %s. Skip nanny service check", c_info.releases_cfg__deploy_system.name)
                return results
            if not self.has_push_scheme(c_info):
                logging.debug(
                    "There is not push-scheme in {c_name}, skip nanny service check".format(c_name=c_info.name)
                )
                return results
            for res_info in filter(lambda x: x.deploy, c_info.releases_cfg__resources_info):
                for deploy_info in res_info.deploy:

                    if isinstance(deploy_info, tuple):
                        continue

                    logging.info("Check deploy for %s", res_info.resource_name)

                    if not(
                        isinstance(deploy_info, configs.DeployServicesInfo) and
                        all(["error" not in self.nanny_client.get_service(s) for s in deploy_info.services]) and
                        all(["error" not in self.nanny_client.get_dashboard_content(d) for d in deploy_info.dashboards])
                    ):
                        results.append(rm_core.Error("Some services or dashboards don't exist"))

                    errors = []

                    for service in c_info.get_deploy_services(deploy_info, self.nanny_client):
                        auth = self.nanny_client.get_service_auth_attrs(service)["content"]
                        if not (
                            check_access(auth["owners"])

                            # Don't check check_access(auth["ops_managers"] here
                            # because Release Machine does not perform *deploy* process by now,
                            # even when push scheme (SPPROBLEM-30) is enabled.
                            or check_access(auth["conf_managers"])
                        ):
                            errors.append(service)

                    if errors:
                        results.append(rm_core.Error(
                            "Robot {} has no access to services: {}".format(rm_const.ROBOT_RELEASER_USER_NAME, errors)
                        ))

                    resource_type_check_result = self._check_resource_type(res_info.resource_type)

                    if not resource_type_check_result.ok:
                        results.append(resource_type_check_result)

        except Exception as exc:
            results.append(rm_core.Error("Got exception in `test_deploy_info` {}".format(exc)))
        return results

    def _check_resource_type(self, resource_type):
        """Check if the given resource type exists in Sandbox"""

        try:
            resource_meta_data = self.server.resource.meta[resource_type].read()
            logging.debug("Resource %s meta data: %s", resource_type, resource_meta_data)
        except self.server.HTTPError as http_error:
            if http_error.status == requests.status_codes.codes.not_found:
                return rm_core.Error("Resource {} does not exist".format(resource_type))

        return rm_core.Ok()

    def test_get_followers(self, c_info):
        results = []
        try:
            if isinstance(c_info, rm_ci.mixin.Startreked):
                followers = c_info.get_followers(self.rm_token)
                self._check_true(
                    isinstance(followers, list), "get_followers returns List"
                )
                self._check_true(
                    (len(followers) == 0) ==
                    (
                        not hasattr(c_info, 'notify_cfg__st__followers') or
                        c_info.notify_cfg__st__followers == [] or
                        not (
                            isinstance(c_info.notify_cfg__st__followers, list) or
                            c_info.notify_cfg__st__followers.staff_groups or
                            c_info.notify_cfg__st__followers.abc_services or
                            c_info.notify_cfg__st__followers.logins
                        )
                    ),
                    "followers empty/not_empty by reason"
                )
                if isinstance(c_info, rm_test.ReleaseMachineTestCfg):
                    logins = ['abc', 'def', 'ghk']
                    c_info.notify_cfg__st__followers = logins
                    self._check_true(
                        c_info.get_followers(self.rm_token) == logins,
                        "List instead of PeopleGroup"
                    )
                    c_info.notify_cfg__st__followers = configs.PeopleGroups(None, None, None)
                    self._check_true(
                        c_info.get_followers(self.rm_token) == [],
                        "PeopleGroup of Nones"
                    )
                    c_info.notify_cfg__st__followers = configs.PeopleGroups(None, None, logins)
                    self._check_true(
                        c_info.get_followers(self.rm_token) == logins,
                        "PeopleGroup.logins"
                    )
                    c_info.notify_cfg__st__followers = configs.PeopleGroups(
                        ['yandex_search_tech_quality_component_8875'], None, None
                    )
                    self._check_gt(
                        len(c_info.get_followers(self.rm_token)), 0,
                        "PeopleGroup.staff_groups"
                    )
                    c_info.notify_cfg__st__followers = configs.PeopleGroups(
                        None, [configs.Abc(2100, None)], None
                    )
                    self._check_gt(
                        len(c_info.get_followers(self.rm_token)), 0,
                        "PeopleGroup.abc_services"
                    )
        except Exception as exc:
            results.append(rm_core.Error("Got exception in `test_get_followers` {}".format(exc)))
        return results

    def test_last_branch_and_tag(self, c_info):
        results = []
        try:
            if isinstance(c_info, rm_ci.Branched):
                last_branch_num = c_info.last_branch_num
                results.append(self._check_int(last_branch_num, "last_branch_num"))
                last_tag_num = c_info.last_tag_num(last_branch_num)
                results.append(self._check_int(last_tag_num, "last_tag_num"))
            elif isinstance(c_info, rm_ci.Tagged):
                results.append(self._check_int(c_info.last_tag_num(), "last_tag_num"))
        except Exception as exc:
            results.append(rm_core.Error("Got exception in `test_last_branch_and_tag` {}".format(exc)))
        return results

    def ensure_no_fail(self, condition, message):
        if not condition:
            logging.warn(message)
            self.set_info(message)

    def _check_true(self, x, x_name):
        logging.info("Check if %s worked", x_name)
        self.ensure_no_fail(x, "{} is expected to work".format(x_name))

    def _check_lt(self, i, i_bound, i_name):
        logging.info("Check if %s < %s", i_name, i_bound)
        self.ensure_no_fail(i < i_bound, "{i_name} = {i_value} is expected to be < {i_bound}".format(
            i_name=i_name,
            i_value=i,
            i_bound=i_bound,
        ))

    def _check_gt(self, i, i_bound, i_name):
        logging.info("Check if %s > %s", i_name, i_bound)
        self.ensure_no_fail(i > i_bound, "{i_name} = {i_value} is expected to be > {i_bound}".format(
            i_name=i_name,
            i_value=i,
            i_bound=i_bound,
        ))

    @staticmethod
    def _check_int(i, i_name):
        logging.info("%s = %s", i_name, i)
        if not isinstance(i, int):
            return rm_core.Error("{} should be int".format(i_name))
        return rm_core.Ok()

    @decorators.retries(2, delay=1)
    def check_robot_access_to_wiki(self, c_info):
        results = []
        if not isinstance(c_info, rm_ci.mixin.Changelogged):
            # ok: no changelogs - no errors
            return results

        page = None
        try:
            page = c_info.changelog_cfg__wiki_page
            logging.debug('[check_robot_access_to_wiki] Checking page %s', page)
        except NotImplementedError:
            results.append(rm_core.Error(
                'ChangeLog wiki page (`wiki_page` property) was not set, although component is changelogged'
            ))

        if not page:
            # `None` specified as `wiki_page` is a valid value meaning "disabled wiki page generation"
            return results

        try:
            url = c_info.changelog_major_url(1)
            logging.debug("Changelog major url: %s", url)
            self.wiki_client.check_wiki_path_correctness(url)
        except Exception as exc:
            eh.log_exception("Can't get access to wiki", exc)
            results.append(rm_core.Error('Cannot get access to {}'.format(lb.wiki_page_link(page))))
        return results

    def check_robot_access_to_st_queue(self, c_infos):
        queues = collections.defaultdict(list)
        startrek_client = rm_st.STHelper(self.rm_token)
        for c_info in c_infos:
            if not isinstance(c_info, rm_ci.mixin.Startreked) or not c_info.notify_cfg__st__queue:
                logging.debug("Component %s doesn't have startrek queue defined in code", c_info.name)
                continue
            queues[c_info.notify_cfg__st__queue].append(c_info.name)
        logging.info("Got startrek queues (%d): %s", len(queues), queues.keys())
        for queue, c_names in queues.items():
            if not startrek_client.check_robot_access_for_queue(queue):
                for component_name in c_names:
                    self._add_component_failure(
                        component_name,
                        "Robot robot-srch-releaser doesn't have access to startrek queue {}".format(queue)
                    )

    def check_resp_for_component_is_wiki_page_owner(self, c_info):
        try:
            if isinstance(c_info, rm_ci.mixin.Changelogged) and c_info.changelog_cfg__wiki_page:
                wiki_page_owner = self.wiki_client.get_page_owner(c_info.changelog_major_url())
                if wiki_page_owner != c_info.get_responsible_for_release(self.rm_token):
                    return [c_info]
                logging.debug(
                    "Responsible for release and wiki page owner for comp %s is %s",
                    c_info.name,
                    c_info.get_responsible_for_release(self.rm_token),
                )
            else:
                logging.debug("Component %s doesn't have wiki page defined in code", c_info.name)
        except Exception as exc:
            eh.log_exception("Got exception while checking wiki page owner", exc)
            self.set_info("Can't check wiki page owner for component {}".format(c_info.name))
        return []

    def check_yappy_cfg(self, c_info):
        # RMINCIDENTS-305
        # Check that yappy_cfg exists for every component
        results = []
        try:
            c_info.yappy_cfg
        except AttributeError:
            self._add_component_failure(c_info.name, 'Attribute `yappy_cfg` is missing')
            self.Context.save()
            self.set_info("Failed to get yappy_cfg from component {}".format(c_info.name))
            results.append(rm_core.Error('`yappy_cfg` attribute is missing'))
        return results

    def check_sdk_compat(self):
        eh.ensure(
            th.ctx_field(self, 'missing_test_field', 'default-value') == 'default-value',
            'Handling default values in `ctx_field` is broken'
        )
        eh.ensure(
            th.ctx_field(self, 'missing_test_field') is None,
            'Handling default values in `ctx_field` is broken'
        )
        self.Context.new_test_field = 100
        eh.ensure(
            th.ctx_field(self, 'new_test_field') == 100,
            'Handling default values in `ctx_field ` is broken'
        )

    def check_token_access(self, c_infos):
        vault_infos = self._get_vault_infos()
        errors = []
        for c_info in c_infos:
            if not c_info.is_branched:
                continue
            for t_name, vault_info in vault_infos.items():
                if c_info.testenv_cfg__trunk_task_owner not in vault_info["shared"]:
                    errors.append("Token '{}' is not shared for {}.testenv_cfg__trunk_task_owner = {}".format(
                        t_name, c_info.name, c_info.testenv_cfg__trunk_task_owner
                    ))
        if errors:
            self.set_info("\n".join(errors))

    @staticmethod
    def _get_vault_infos():
        rest = sb_rest.Client()
        tokens = [
            (rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME),
            (rm_const.COMMON_TOKEN_OWNER, rm_const.ARC_ACCESS_TOKEN_NAME),
            (rm_const.COMMON_TOKEN_OWNER, rm_const.TELEGRAM_TOKEN_NAME),
        ]
        vault_infos = {}
        for t_owner, t_name in tokens:
            vault_info = rest.vault.read(name=t_name, owner=t_owner, limit=1)["items"][0]
            logging.debug("Vault info for %s:\n%s", t_name, vault_info)
            vault_infos[t_name] = vault_info
        return vault_infos

    def _get_user_roles(self, c_info):
        user_roles = collections.defaultdict(list)
        user_roles[c_info.get_responsible_for_component(self.rm_token)].append((c_info.name, RESPONSIBLE_FOR_COMPONENT))
        user_roles[c_info.get_responsible_for_release(self.rm_token)].append((c_info.name, RESPONSIBLE_FOR_RELEASE))
        if c_info.is_branched or c_info.is_tagged:
            for staff_login in c_info.testenv_cfg__testenv_db_owners:
                user_roles[staff_login].append((c_info.name, "Testenv DB owner"))

        if c_info.notify_cfg__use_startrek:
            user_roles[c_info.st_assignee].append((c_info.name, "Startrek release ticket assignee"))
        return user_roles

    def _report_component_state(self, component_name, messages, ok=False):
        """
        :param component_name: component name
        :param messages: message list
        :param ok: whether the state/result is ok or not
        """

        logging.info("Going to post component state (config functional) for %s", component_name)

        now = tu.datetime_utc()
        now_ts = tu.datetime_to_timestamp(now)

        logging.debug("Now: %s", now)

        event = events_pb2.EventData(
            general_data=events_pb2.EventGeneralData(
                hash=events_public.get_event_hash(
                    now.isoformat(),
                    component_name,
                    "UpdateComponentState",
                    "config_functional",
                ),
                component_name=component_name,
                referrer="sandbox_task:{}".format(self.id),
            ),
            task_data=events_pb2.EventSandboxTaskData(
                task_id=self.id,
                status=self.status,
                created_at=self.created.isoformat(),
                updated_at=self.updated.isoformat(),
            ),
            update_component_state_data=events_pb2.UpdateComponentStateData(
                component_state=component_state_pb2.ComponentState(
                    timestamp=int(now_ts),
                    referrer="{}:{}".format(self.type, self.id),
                    config_functional=component_state_pb2.ComponentSectionState(
                        status=(
                            component_state_pb2.ComponentSectionState.Status.OK if ok
                            else component_state_pb2.ComponentSectionState.Status.CRIT
                        ),
                        info=component_state_pb2.ComponentSectionInfo(
                            title=("Config functional errors" if not ok else "All OK"),
                            description="\n".join(messages),
                        ),
                    ),
                ),
            ),
        )

        logging.info("Constructed event: %s", event)

        self.rm_api_client.post_proto_events([event, ])
