# -*- coding: utf-8 -*-

import time
import datetime
import os

import jinja2
import logging
import re
import sys

from dateutil import parser as dt_parser

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import time_utils
from sandbox.projects.common.search import bugbanner2 as bb2
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import task_env
from sandbox.common import itertools as sb_itertools

if sys.version_info[0:2] >= (3, 8):  # python 3.8 or higher
    from functools import cached_property
else:
    from sandbox.projects.common.decorators import memoized_property as cached_property


RESULT_STATUS_OK = "OK"
RESULT_STATUS_FAIL = "FAIL"

HALF_A_YEAR_WEEKS = 6 * 4


class MethodInsecureUse(Exception):
    pass


class RMGCParamInvalid(Exception):

    def __init__(self, param_name, error):
        super(RMGCParamInvalid, self).__init__("Parameter {} invalid: {}".format(param_name, error))


class ResourceHitman(object):

    DEFAULT_LIMIT = 10000
    DEFAULT_RESOURCE_TYPES = [
        "BEGEMOT_REALTIME_PACKAGE",
        "REARRANGE_DATA_FAST",
        "REARRANGE_DATA_FAST_BUNDLE",
    ]
    DEFAULT_DAYS_AGO_LEFT = 90
    DEFAULT_DAYS_AGO_RIGHT = 60

    TTL_NAME = "ttl"

    SLEEP_S = 0.1
    RESOURCE_SEARCH_STEP = 100
    OWNER = "SEARCH-RELEASERS"
    DESTROY_BATCH_SIZE = 500

    def __init__(self, limit, types, days_ago_left, days_ago_right, sb_client):
        self._limit = limit
        self._types = types
        self._now = time_utils.datetime_utc()
        self._start = self._now - datetime.timedelta(days=int(days_ago_left))
        self._end = self._now - datetime.timedelta(days=int(days_ago_right))
        self._resources_deleted = 0

        self._sb_client = sb_client

    @property
    def resources_deleted(self):
        return self._resources_deleted

    def start_massacre(self):

        for resource_type in self._types:

            logging.info("Considering %s resources", resource_type)

            resource_type = resource_type.strip()

            victims, complicated_victims = self._find_victims(resource_type)

            for resource_id in complicated_victims:
                self._prepare_to_die(resource_id)

            for batch in sb_itertools.chunker(data=victims, size=self.DESTROY_BATCH_SIZE):
                self._die(batch)

    def _find_victims(self, resource_type):
        offset = 0

        result = []
        require_special_attention = []

        while offset < self._limit:

            resources = self._sb_client.resource.read(
                limit=self.RESOURCE_SEARCH_STEP,
                offset=offset,
                **self._get_search_parameters(resource_type)
            )

            offset += self.RESOURCE_SEARCH_STEP

            result += [resource["id"] for resource in resources["items"]]
            require_special_attention += [
                resource["id"] for resource in resources["items"]
                if "ttl" in resource["attributes"] and resource["attributes"]["ttl"] == "inf"
            ]

        logging.debug("Got resource IDs: %s", result)
        logging.info("%d resources of type %s to be destroyed", len(result), resource_type)

        return result, require_special_attention

    def _get_search_parameters(self, resource_type):
        return dict(
            state="READY",
            owner=self.OWNER,
            type=resource_type,
            created="{}..{}".format(self._start.isoformat(), self._end.isoformat()),
            order="-size",

        )

    def _prepare_to_die(self, resource_id):
        """For those who refuse to die we reduce the time to live. And then they can be destroyed just like others"""

        logging.info("Preparing sbr:%s for execution (set ttl=1)", resource_id)

        try:
            self._sb_client.resource[resource_id].attribute[self.TTL_NAME].update(name=self.TTL_NAME, value=1)
        except:
            eh.log_exception("... Unable to prepare resource sbr:%s", resource_id)

        logging.info("... set OK. Going to sleep for %ss now", self.SLEEP_S)

        time.sleep(self.SLEEP_S)

    def _die(self, resource_id_list):

        self._sb_client.batch.resources["delete"].update(
            id=resource_id_list,
            comment="Drop old useless resources for the sake of RMINCIDENTS-577",
        )

        self._resources_deleted += len(resource_id_list)


class ReleaseMachineGarbageCollector(bb2.BugBannerTask, binary_task.LastBinaryTaskRelease):

    class Requirements(task_env.TinyRequirements):
        pass

    class Context(sdk2.Task.Context):
        results = {}
        report_html = ""

    class Parameters(binary_task.LastBinaryReleaseParameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        rm_host = sdk2.parameters.String(
            "Release Machine service address",
            default_value=rm_const.Urls.RM_HOST,
        )
        event_drop_delta_weeks = sdk2.parameters.Integer(
            "Drop all events older than this number of weeks ago",
            default_value=HALF_A_YEAR_WEEKS,
        )
        component_drop_delta_weeks = sdk2.parameters.Integer(
            "Drop all components that have been deleted earlier than this number of weeks ago. "
            "Note: we're talking about irreversible drop here",
            default_value=HALF_A_YEAR_WEEKS,
        )
        component_drop_limit = sdk2.parameters.Integer(
            "Never drop more than this number of components per launch",
            default_value=1,
        )

        with sdk2.parameters.Group("MDS Quota Cleanup Settings"):

            perform_resource_cleanup = sdk2.parameters.Bool(
                "Perform resource cleanup?",
                default_value=False,
            )

            resource_cleanup_limit = sdk2.parameters.Integer(
                "Max number of resources of each type that are going to be deleted",
                default_value=ResourceHitman.DEFAULT_LIMIT,
                required=True,
            )

            resource_cleanup_types = sdk2.parameters.String(
                "Comma-separated resource types. The task will search and destroy resources of these types",
                default_value=",".join(ResourceHitman.DEFAULT_RESOURCE_TYPES),
                required=True,
            )

            resource_cleanup_datetime_left_timedelta_days = sdk2.parameters.Integer(
                "Take resources created since this number of days ago",
                default_value=ResourceHitman.DEFAULT_DAYS_AGO_LEFT,
                required=True,
            )

            resource_cleanup_datetime_right_timedelta_days = sdk2.parameters.Integer(
                "Take resources created before this number of days ago",
                default_value=ResourceHitman.DEFAULT_DAYS_AGO_RIGHT,
                required=True,
            )

    @cached_property
    def rm_event_client(self):
        from release_machine.release_machine.services.release_engine.services.Event import EventClient
        logging.debug("Creating RM Event client for %s", self.Parameters.rm_host)
        return EventClient.from_address(self.Parameters.rm_host)

    @cached_property
    def rm_model_client(self):
        from release_machine.release_machine.services.release_engine.services.Model import ModelClient
        logging.debug("Creating RM Model client for %s", self.Parameters.rm_host)
        return ModelClient.from_address(self.Parameters.rm_host)

    @cached_property
    def events_drop_datetime_threshold_iso(self):
        return (
            datetime.datetime.utcnow() - datetime.timedelta(weeks=int(self.Parameters.event_drop_delta_weeks))
        ).isoformat() + "Z"

    @cached_property
    def component_drop_datetime_threshold(self):
        return time_utils.datetime_utc() - datetime.timedelta(weeks=int(self.Parameters.component_drop_delta_weeks))

    @sdk2.header()
    def header(self):
        return self.Context.report_html

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)
        bb2.BugBannerTask.on_save(self)

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)
        bb2.BugBannerTask.on_execute(self)

        self._drop_old_events()
        self._drop_old_deleted_components()

        self._cleanup_release_machine_mds_quota()

        eh.ensure(self._is_overall_result_ok(), "Some actions failed. See task report")

    def _drop_old_events(self):

        import grpc
        from release_machine.release_machine.proto.structures import message_pb2
        from search.martylib.core.exceptions import MaxRetriesReached

        try:

            res = self.rm_event_client.drop_old_events(
                request=message_pb2.DropOldEventsRequest(older_than_datetime_iso=self.events_drop_datetime_threshold_iso),
            )

            self._fill_result("drop_old_events", RESULT_STATUS_OK, "{} events dropped".format(res.dropped_count))

            return

        except grpc.RpcError as rpc_error:
            message, exc = "Unable to drop old events: server responded with error", rpc_error

        except MaxRetriesReached as mrr:
            message, exc = "Unable to drop old events: max retries reached", mrr

        eh.log_exception(message, exc)

        self._fill_result("drop_old_events", RESULT_STATUS_FAIL, message)

    def _drop_old_deleted_components(self):

        if not self.Parameters.component_drop_delta_weeks:
            logging.info("component_drop_delta_weeks = 0 — skipping component deletion")
            return

        logging.info("Check deleted components")
        logging.info("Component drop threshold is %s", self.component_drop_datetime_threshold.isoformat())

        import grpc
        from release_machine.release_machine.proto.structures import message_pb2
        from search.martylib.core.exceptions import MaxRetriesReached

        deleted_count = 0

        try:

            deleted_components = self._get_deleted_components()
            deleted_components_data = self._filter_components_with_deletion_time_set(deleted_components)

            for component, deleted_since_utc in deleted_components_data:

                if deleted_count > self.Parameters.component_drop_limit:
                    logging.info(
                        "Stopping component deletion since the limit of %s has been reached",
                        self.Parameters.component_drop_limit,
                    )
                    break

                logging.debug("Considering %s", component.name)

                now = time_utils.datetime_utc()

                if deleted_since_utc < self.component_drop_datetime_threshold:

                    logging.debug(
                        "%s has been deleted since %s (it's %s now)",
                        deleted_since_utc.isoformat(),
                        now.isoformat(),
                    )
                    logging.info("Going to delete component %s for good", component.name)

                    self.rm_model_client.delete_component(
                        request=message_pb2.DeleteComponentRequest(
                            component_name=component.name,
                            irreversible=True,
                        )
                    )

                    deleted_count += 0

                else:

                    logging.debug("Not enough time passed since %s has been deleted. Doing nothing", component.name)

        except (grpc.RpcError, MaxRetriesReached) as exc:
            eh.log_exception("Unable to process deleted components", exc)
            self._fill_result("drop_old_components", RESULT_STATUS_FAIL, "See task logs for more details")
            return

        self._fill_result(
            "drop_old_components",
            RESULT_STATUS_OK,
            "{} components erased".format(deleted_count),
        )

    def _cleanup_release_machine_mds_quota(self):

        if not self.Parameters.perform_resource_cleanup:
            logging.info("Not going to perform resource cleanup")
            return

        self._validate_quota_cleanup_parameters()

        rh = ResourceHitman(
            limit=self.Parameters.resource_cleanup_limit,
            types=self.Parameters.resource_cleanup_types.split(","),
            days_ago_left=self.Parameters.resource_cleanup_datetime_left_timedelta_days,
            days_ago_right=self.Parameters.resource_cleanup_datetime_right_timedelta_days,
            sb_client=self.server,
        )

        rh.start_massacre()

        self._fill_result(
            "resource_cleanup",
            RESULT_STATUS_OK,
            "{} resources deleted".format(rh.resources_deleted),
        )

    def _validate_quota_cleanup_parameters(self):

        if (
            self.Parameters.resource_cleanup_datetime_left_timedelta_days <=
            self.Parameters.resource_cleanup_datetime_right_timedelta_days
        ):
            raise RMGCParamInvalid(
                "resource_cleanup_datetime_right_timedelta_days",
                "should be < resource_cleanup_datetime_left_timedelta_days",
            )

        if not self.Parameters.resource_cleanup_types:
            raise RMGCParamInvalid(
                "resource_cleanup_types",
                "cannot be blank",
            )

    def _get_deleted_components(self):
        from release_machine.release_machine.proto.structures import message_pb2, table_pb2

        result = []

        component_list = self.rm_model_client.get_components(request=message_pb2.Dummy())

        for component in component_list.components:

            if component.status is not table_pb2.Component.ComponentStatus.DELETED:
                continue

            result.append(component)

        logging.debug("All components marked deleted: %s", result)

        return result

    def _filter_components_with_deletion_time_set(self, deleted_components):
        """
        Given a list of components marked for deletion returns a list of pairs `(component, deleted_since_utc)`
        where each `component` of `deleted_components` is paired with its deletion time (`deleted_since_utc`).
        For those components that do not declare a deletion time it is set to now and the component itself is
        omitted in the result. The result is sorted in ascending order by the deletion time (i.e. older deleted
        components come first).

        Note that the list :param deleted_components: should only contain components marked for deletion (i.e., with
        status = DELETED).
        """
        from release_machine.release_machine.proto.structures import message_pb2, table_pb2
        from release_machine.release_machine.src import const

        result = []

        for component in deleted_components:

            if component.status is not table_pb2.Component.ComponentStatus.DELETED:
                raise MethodInsecureUse(
                    "Something is not right: `_filter_components_with_deletion_time_set` received a list of components"
                    "in which at least one is not marked as DELETED: {}. The evaluation of this task should not"
                    "continue until the bug is solved since it's dangerous and may lead to a possible data loss".format(
                        component,
                    ),
                )

            state_deleted_at = self.rm_model_client.get_state(
                request=message_pb2.StateRequest(
                    component_name=component.name,
                    key=const.StateKeys.COMPONENT_DELETED_SINCE,
                ),
            )

            if not state_deleted_at.entries:
                logging.info(
                    "Component %s has been deleted but no %s state value found. Going to set it to now",
                    component.name,
                    const.StateKeys.COMPONENT_DELETED_SINCE,
                )

                self.rm_model_client.post_state(request=message_pb2.PostStateRequest(
                    component_name=component.name,
                    key=const.StateKeys.COMPONENT_DELETED_SINCE,
                    value=time_utils.datetime_utc_iso(),
                ))

                continue

            result.append((component, dt_parser.parse(state_deleted_at.entries[0].value)))

        return sorted(result, key=lambda item: item[1])

    def _fill_result(self, action_name, status, message):
        self.Context.results[action_name] = {
            "status": status,
            "message": message,
        }
        self.Context.save()

        self._rerender_report()

    def _rerender_report(self):
        from library.python import resource as py_resource

        logging.info("Preparing report")

        report_template_path = self.get_resource_path("report.jinja2")

        template = py_resource.find(report_template_path)

        if not template:
            self.set_info("Unable to render report: no such file {}".format(report_template_path))
            return

        jinja_index_template = jinja2.Template(template.decode('utf-8'))
        report = jinja_index_template.render(results=self.Context.results)
        report = re.sub(r"\n[\n ]+", "\n", report)

        self.Context.report_html = report

    def _is_overall_result_ok(self):
        results = self.Context.results

        for action_name in results:
            if results[action_name]["status"] is RESULT_STATUS_FAIL:
                return False

        return True

    @staticmethod
    def get_resource_path(resource_file_name):
        return os.path.join(os.path.dirname(__file__), "templates", resource_file_name)
