# -*- coding:utf-8 -*-
import logging
import os
import re
import six

from sandbox.common import rest
from sandbox.projects.common import binary_task
from sandbox.projects.common import time_utils
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.release_machine.client import RMClient
import sandbox.projects.common.link_builder as lb
import sandbox.projects.common.string as string
import sandbox.projects.common.gsid_parser as gsid_parser
import sandbox.projects.common.search.bugbanner2 as bb2
import sandbox.projects.release_machine.events as rm_events
import sandbox.projects.release_machine.helpers.events_helper as events_helper
import sandbox.projects.release_machine.helpers.merge_helper as merge_helper
import sandbox.projects.release_machine.input_params2 as rm_params
import sandbox.projects.release_machine.components.configs.all as rm_conf_all
import sandbox.projects.release_machine.components.all as rm_comp
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.sdk2 as sdk2

from sandbox.projects.release_machine import rm_notify


SB_TASK_PARSER = re.compile(r"Sandbox task:? https://sandbox\.yandex-team\.ru/task/([0-9]{5,})")
LAST_CRAWLED_REVISION = "last_crawled_revision${}"


@rm_notify.notify2(people=["rm_maintainers", ])
class ReleaseMachineMergesCrawler(binary_task.LastBinaryTaskRelease, bb2.BugBannerTask):

    class Requirements(task_env.TinyRequirements):
        pass

    class Parameters(rm_params.BaseReleaseMachineParameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        post_branch_events = sdk2.parameters.Bool("Post NewBranch events?", default=True)
        post_tag_events = sdk2.parameters.Bool("Post NewTag events?", default=True)
        post_merges = sdk2.parameters.Bool("Post RollbackCommit and MergeCommit events?", default=True)

    @property
    def is_binary_run(self):
        return self.Parameters.binary_executor_release_type != 'none'

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)
        self.add_bugbanner(bb2.Banners.ReleaseMachine)

        keys = ["heads", "logs", "tasks"]
        self._cache = {k: {} for k in keys}
        self._cache_hit = {k: 0 for k in keys}

        self.rm_client = RMClient()
        self.rest_client = rest.Client()

        self._process_branches()
        logging.info("Cache hits: %s", self._cache_hit)
        self._process_tags()
        logging.info("Cache hits: %s", self._cache_hit)

    def _logs_for_dir(self, directory, rev_from, rev_to):
        if self._cache["logs"].get(directory) is None:
            self._cache["logs"][directory] = sdk2.svn.Arcadia.log(
                directory,
                revision_from=rev_from, revision_to=rev_to,
                stop_on_copy=True, track_copy=True
            )
        else:
            self._cache_hit["logs"] += 1
        return self._cache["logs"][directory]

    def _head_for_dir(self, directory):
        if self._cache["heads"].get(directory) is None:
            info = sdk2.svn.Arcadia.info(directory)
            self._cache["heads"][directory] = int(info["entry_revision"]) if info else 0
        else:
            self._cache_hit["heads"] += 1
        return self._cache["heads"].get(directory)

    def _process_tags(self):
        logging.info("Process tags")
        not_ok = []
        for component_cfg_cls in rm_conf_all.ALL_CONFIGS.itervalues():
            component_cfg = component_cfg_cls()
            if component_cfg.is_trunk:
                continue
            head_rev = self._head_for_dir(component_cfg.svn_cfg.tag_dir)
            if head_rev:
                try:
                    events = self.get_tag_proto_events_for_one_component(component_cfg, head_rev)
                    self._post_proto_events(component_cfg, events, head_rev, not_ok, "tags")
                except Exception as e:
                    eh.log_exception("Unable to build or send proto events. Activating fallback to old events", e)
            else:
                self.set_info("[{}] No head revision for tags".format(component_cfg.name))

        self._send_not_ok_alerts(not_ok, "tags")

    def _get_event_type(self, event):
        # New-style event specific data translates to event type by cutting off the '_data' suffix
        # and transforming the remaining snake_case string into CamelCase
        return "".join(map(lambda x: x.capitalize(), event.WhichOneof('specific_data').split('_')[:-1]))

    def _filter_events(self, events):
        for event in events:

            event_type = self._get_event_type(event)

            if not self.Parameters.post_branch_events and event_type == rm_events.EventType.NewBranch:
                continue
            if not self.Parameters.post_tag_events and event_type == rm_events.EventType.NewTag:
                continue
            if not self.Parameters.post_merges and (
                event_type == rm_events.EventType.MergeCommit or
                event_type == rm_events.EventType.RollbackCommit
            ):
                continue
            logging.debug("Yielding event of type %s", event_type)
            yield event

    def _post_proto_events(self, component_cfg, events, head_rev, not_ok, folder):

        from release_machine.common_proto import events_pb2

        events = list(events)
        self._log_nonzero(len(events), "Got {} events before filtering".format(len(events)))
        events = list(self._filter_events(events))
        self._log_nonzero(len(events), "Got {} events after filtering".format(len(events)))
        if not events:
            return
        result_ok = events_helper.post_proto_events(events)
        if result_ok:
            events_helper.post_proto_events([
                events_pb2.EventData(
                    general_data=events_pb2.EventGeneralData(
                        hash=six.text_type(hash((
                            component_cfg.name,
                            LAST_CRAWLED_REVISION.format(folder),
                            head_rev,
                            "",
                        ))),
                        component_name=component_cfg.name,
                        referrer="sandbox_task:{}".format(self.id),
                    ),
                    task_data=events_pb2.EventSandboxTaskData(
                        task_id=int(self.id),
                        status=self.status,
                        created_at=self.created.isoformat(),
                        updated_at=time_utils.datetime_utc_iso(),
                    ),
                    update_state_data=events_pb2.UpdateStateData(
                        key=LAST_CRAWLED_REVISION.format(folder),
                        value=str(head_rev),
                    ),
                )
            ])
        else:
            not_ok.append(component_cfg.name)

    def get_tag_proto_events_for_one_component(self, component_cfg, head_rev):

        from release_machine.common_proto import events_pb2

        last_crawled_tags_revision = self._get_last_crawled_rev(component_cfg.name, "tags")
        svn_log = self._logs_for_dir(
            component_cfg.svn_cfg.tag_dir,
            rev_from=last_crawled_tags_revision or head_rev - 50000,
            rev_to=head_rev,
        )
        self._log_nonzero(len(svn_log), "Got {} tag log entries for '{}'".format(len(svn_log), component_cfg.name))
        tag_path_pattern = re.compile(component_cfg.svn_cfg.tag_path_pattern)
        for log_entry in svn_log:

            copies = self.get_copies(log_entry)

            if copies:
                action_task_info = self.get_action_task_info(log_entry)
                found_pattern = tag_path_pattern.search(log_entry["paths"][0][-1])

                if not found_pattern:
                    continue

                found_tag_path = found_pattern.group()
                scope_number = found_pattern.group(1)

                if component_cfg.is_branched:
                    tag_number = found_pattern.group(2)
                    tag_folder_name = component_cfg.svn_cfg.tag_folder_name(scope_number, tag_number)
                elif component_cfg.is_tagged:
                    tag_folder_name = component_cfg.svn_cfg.tag_folder_name(scope_number)
                    tag_number = scope_number
                else:
                    continue

                origin_commit_id = int(copies[0][1])

                if action_task_info:
                    gsid = action_task_info.get("context.__GSID")
                    job_name = gsid_parser.get_job_name_from_gsid(gsid) or "NEW_TAG"
                else:
                    job_name = ""

                arcadia_path = os.path.join(
                    component_cfg.svn_cfg.REPO_NAME,
                    found_tag_path,
                )
                commit_id = int(log_entry["revision"])
                event = events_pb2.EventData(
                    general_data=events_pb2.EventGeneralData(
                        hash=six.text_type(hash((
                            "TASK_SUCCESS",
                            arcadia_path,
                            str(scope_number),
                            origin_commit_id,
                            commit_id,
                        ))),
                        component_name=component_cfg.name,
                        referrer=u"sandbox_task:{}".format(self.id)
                    ),
                    task_data=events_pb2.EventSandboxTaskData(
                        task_id=int(self.id),
                        status="SUCCESS",
                        created_at=self.created.isoformat(),
                        updated_at=time_utils.datetime_utc_iso(),
                    ),
                    new_tag_data=events_pb2.NewTagData(
                        base_commit_id=str(origin_commit_id),
                        first_commit_id=str(log_entry["revision"]),
                        scope_number=scope_number,
                        tag_number=tag_number,
                        tag_name=tag_folder_name,
                        arcadia_path=arcadia_path,
                        tag_created_at_timestamp=int(time_utils.datetime_to_timestamp(log_entry["date"])),
                        job_name=job_name,
                        tag_log=events_pb2.TagLog(
                            datetime_iso=log_entry["date"].isoformat(),
                            commit_message=log_entry["msg"],
                            revision=str(log_entry["revision"]),
                            author=log_entry["author"],
                        ),
                    ),
                )
                if action_task_info:
                    event.task_data.task_id = int(action_task_info["id"])
                    event.task_data.created_at = action_task_info["time.created"]
                    event.task_data.updated_at = action_task_info["time.updated"]

                yield event

    def _process_branches(self):
        logging.info("Process branches")
        not_ok = []
        for component_name in rm_conf_all.get_all_branched_names():
            component_cfg = rm_conf_all.ALL_CONFIGS[component_name]()
            head_rev = self._head_for_dir(component_cfg.svn_cfg.branch_dir)
            if head_rev:
                try:
                    events = self.get_branch_proto_events_for_one_component(component_cfg, head_rev)
                    self._post_proto_events(component_cfg, events, head_rev, not_ok, "branches")
                except Exception as e:
                    eh.log_exception("Unable to create or send proto events. Activating fallback to old events", e)
            else:
                self.set_info("[{}] No head revision for branches".format(component_cfg.name))
        self._send_not_ok_alerts(not_ok, "branches")

    def _send_not_ok_alerts(self, not_ok, folder):
        if not_ok:
            rm_notify.send_tm_message(
                self,
                "WARNING! NOT OK crawling {folder} result for components: [{component}]\nSB: {task_link}".format(
                    folder=folder,
                    component=", ".join(c_name for c_name in not_ok),
                    task_link=lb.task_link(self.id),
                ),
                [rm_const.RM_USERS["rm_maintainers"].telegram],
            )

    def get_branch_proto_events_for_one_component(self, component_cfg, head_rev):
        last_crawled_revision = self._get_last_crawled_rev(component_cfg.name, "branches")
        svn_log = self._logs_for_dir(
            component_cfg.svn_cfg.branch_dir,
            rev_from=last_crawled_revision or head_rev - 50000,
            rev_to=head_rev,
        )
        self._log_nonzero(len(svn_log), "Got {} branch log entries for '{}'".format(len(svn_log), component_cfg.name))
        if svn_log:
            c_info = rm_comp.COMPONENTS[component_cfg.name]()
            for log_entry in svn_log:
                action_task_info = self.get_action_task_info(log_entry)
                for event in self.all_proto_events_for_one_branch_log_entry(c_info, log_entry):
                    if action_task_info:
                        event.task_data.task_id = int(action_task_info["id"])
                        event.task_data.created_at = action_task_info["time.created"]
                        event.task_data.updated_at = action_task_info["time.updated"]
                    yield event

    def get_action_task_info(self, log_entry):
        action_task_id = SB_TASK_PARSER.search(log_entry["msg"])
        if action_task_id:
            task_id = action_task_id.group(1)
            if self._cache["tasks"].get(task_id) is None:
                action_task_info = self.rest_client.task[{
                    "limit": 1,
                    "id": task_id,
                    "fields": ["time.created", "time.updated", "context.__GSID"],
                    "hidden": True,
                }]["items"]
                self._cache["tasks"][task_id] = action_task_info[0] if action_task_info else {}
            else:
                self._cache_hit["tasks"] += 1
            return self._cache["tasks"][task_id]

    def all_proto_events_for_one_branch_log_entry(self, c_info, log_entry):

        from release_machine.common_proto import events_pb2
        from release_machine.public import events as events_public

        branch_num = c_info.svn_cfg__get_major_release_num(log_entry["paths"][0][-1])

        if not branch_num:
            logging.info("Skipping: No branch number found for log entry %s", log_entry["paths"][0][-1])
            return

        commit_id = int(log_entry["revision"])

        task_data = events_pb2.EventSandboxTaskData(
            task_id=self.id,
            status="SUCCESS",
            created_at=self.created.isoformat(),
            updated_at=time_utils.datetime_utc_iso(),
        )

        copies = self.get_copies(log_entry)

        arcadia_path = os.path.join(
            c_info.svn_cfg__REPO_NAME,
            c_info.svn_cfg__branches_folder,
            c_info.svn_cfg__branch_name,
            c_info.svn_cfg__branch_folder_name(branch_num),
        )

        if copies:  # means maybe revision of branching

            origin_commit_id = int(copies[0][1])

            if string.left_strip(c_info.svn_cfg__main_url, c_info.svn_cfg__repo_base_url) == copies[0][0]:
                # RMINCIDENTS-407
                # not all copies are revisions of branching, need additional filter by main url
                yield events_pb2.EventData(
                    general_data=events_pb2.EventGeneralData(
                        hash=six.text_type(events_public.get_event_hash(
                            "TASK_SUCCESS",
                            arcadia_path,
                            str(branch_num),
                            origin_commit_id,
                            commit_id,
                        )),
                        component_name=c_info.name,
                        referrer=u"sandbox_task:{}".format(self.id),
                    ),
                    task_data=task_data,
                    new_branch_data=events_pb2.NewBranchData(
                        base_commit_id=str(origin_commit_id),
                        first_commit_id=str(commit_id),
                        scope_number=str(branch_num),
                        job_name="",
                        arcadia_path=arcadia_path,
                        branch_created_at_timestamp=int(time_utils.datetime_to_timestamp(log_entry["date"])),
                    ),
                )

            yield events_pb2.EventData(
                general_data=events_pb2.EventGeneralData(
                    hash=six.text_type(events_public.get_event_hash(
                        c_info.name,
                        'MergeCommit',
                        0,  # Fixme (ilyaturuntaev) RMDEV-1949
                        commit_id,
                        str(branch_num),
                        "TASK_SUCCESS",
                    )),
                    component_name=c_info.name,
                    referrer=u"sandbox_task:{}".format(self.id),
                ),
                task_data=task_data,
                merge_commit_data=events_pb2.MergeCommitData(
                    origin_commit_id=str(origin_commit_id),
                    origin_commit_id_list=[str(origin_commit_id)],
                    branch_commit_id=str(commit_id),
                    scope_number=str(branch_num),
                    origin_commit_messages=[""],
                ),
            )

        else:

            merges = merge_helper.get_merges_from_commit_message(log_entry["msg"]) or [None]

            yield events_pb2.EventData(
                general_data=events_pb2.EventGeneralData(
                    hash=six.text_type(events_public.get_event_hash(
                        c_info.name,
                        'MergeCommit',
                        0,
                        commit_id,
                        str(branch_num),
                        "TASK_SUCCESS",
                    )),
                    component_name=c_info.name,
                    referrer=u"sandbox_task:{}".format(self.id),
                ),
                task_data=task_data,
                merge_commit_data=events_pb2.MergeCommitData(
                    origin_commit_id=str(merges[0] or 0),  # RMDEV-2372
                    origin_commit_id_list=[str(origin_commit_id or 0) for origin_commit_id in merges],
                    branch_commit_id=str(commit_id),
                    scope_number=str(branch_num),
                    origin_commit_messages=[""],
                ),
            )

            rollbacks = merge_helper.get_rollbacks_from_commit_message(log_entry["msg"])

            for rolled_back_commit_id in rollbacks:

                yield events_pb2.EventData(
                    general_data=events_pb2.EventGeneralData(
                        hash=six.text_type(events_public.get_event_hash(
                            c_info.name,
                            'RollbackCommit',
                            rolled_back_commit_id,
                            commit_id,
                            "TASK_SUCCESS",
                        )),
                        component_name=c_info.name,
                        referrer=u"sandbox_task:{}".format(self.id),
                    ),
                    task_data=task_data,
                    rollback_commit_data=events_pb2.RollbackCommitData(
                        rolled_back_commit_id=str(rolled_back_commit_id),
                        new_commit_id=str(commit_id),
                        scope_number=str(branch_num),
                        is_trunk_rollback=False,
                    ),
                )

    def _get_last_crawled_rev(self, c_name, folder):
        last_crawled_rev = self.rm_client.get_state(c_name, LAST_CRAWLED_REVISION.format(folder))
        last_crawled_rev = int(last_crawled_rev[0]["value"]) if last_crawled_rev else 0
        logging.info("Last crawled revision for %s: %d", c_name, last_crawled_rev)
        return last_crawled_rev

    @staticmethod
    def get_copies(log_entry):
        copies = log_entry.get("copies")
        if copies:
            logging.debug("Got log entry, copied from %s: %s", copies[0][0], copies[0][1])
        return copies

    @staticmethod
    def _log_nonzero(checked_param, message):
        if checked_param:
            logging.info(message)
        else:
            logging.debug(message)
