# -*- coding: utf-8 -*-
import datetime
import dateutil.tz as tz
import json
import logging
import os
import re

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common import time_utils as tu
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import requests_wrapper
from sandbox.projects.release_machine.components.configs import all as all_cfg
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import task_env
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.tasks.ReleaseMachineStatCrawler import ab_exp_compare

import sandbox.projects.common.search.bugbanner2 as bb2


def _metrics_stats_default():
    """
    Default value for metrics_stats_config parameter.
    See the parameter's description for more info.
    """
    return {
        "ab_experiments": [0, 200],
        "ab_exp": [0, 200],
        "ab_flags": [0, 3],
        "ab_flags_testids": [0, 15],
        "app_host": [0, 3],
        "base": [0, 3],
        "begemot": [0, 3],
        "begemot_request_init": [0, 3],
        "middle": [0, 3],
        "report": [0, 3],
        "src_setup": [0, 3],
        "upper": [0, 3],
        "web_graph_mapping": [0, 3],
    }


class ReleaseMachineStatCrawler(bb2.BugBannerTask, binary_task.LastBinaryTaskRelease):
    """
    Used to crawl various types of Release Machine statistics.

    - Release Frequency (https://nda.ya.ru/t/AgW1qxXy3W4RPU)
    - Release Saw (https://nda.ya.ru/t/gGZI9llp3W4RPa)
    - Metrics TimeSpectre (https://nda.ya.ru/t/O0ue-yxW3W4RPk)
    - Release Machine Component Count

    Notes:
        - The metrics_stats_config parameter is primary for the Metrics TimeSpectre graph calculation, meaning that no
        components that are not listed in this config will be subject to metrics data calculation. If
        component_name_filter parameter is also provided then only the components from metrics_stats_config that
        match component_name_filter are considered.
        - Preloading is essential for metrics data calculation. Using preload_metrics_sla_table_days_count
        parameter correctly decreases the overall calculation time dramatically. This option allows to preload
        Metrics SLA tables for the given amount of days and cache it preventing from excess YQL queries during
        calculation (YQL queries are really expensive!). If there's not enough data in the preloaded table then a new
        YQL query is executed for EACH LaunchMetrics task that requires more data
    """
    MONTH_WINDOW = datetime.timedelta(days=30)
    WEEK_WINDOW = datetime.timedelta(days=7)
    DAY_SECONDS = 86400.

    SOLOMON_PROJECT = "release_machine"
    SOLOMON_CLUSTER = "default"
    STATISTICS_SOLOMON_SERVICE = "release_statistics"
    TIMESPECTRE_METRICS_SERVICE = "timespectre_metrics"
    RM_GENERAL_CHARACTERISTICS = "general_characteristics"
    AB_EXP_LAG = "ab_exp_lag"

    RELEASE_FREQUENCY_MONTH_SENSOR_NAME = "release_frequency__month"
    RELEASE_FREQUENCY_WEEK_SENSOR_NAME = "release_frequency__week"
    RELEASE_SAW_SENSOR_NAME = "release_saw__days"
    RELEASE_SAW_CALENDAR_SENSOR_NAME = "release_saw_calendar__days"
    COMPONENT_COUNT_TOTAL_SENSOR = "component_count__total"
    COMPONENT_COUNT_ACTIVE_SENSOR = "component_count__active"
    COMPONENT_COUNT_CI_SENSOR = "component_count__ci"

    NUMBER_OF_PROCESSED_COMPONENTS_TO_PRINT_IN_INFO = 20

    _token = None
    _release_statistics_solomon_client = None
    _timespectre_metrics_solomon_client = None
    _rm_general_characteristics_solomon_client = None
    _ab_exp_lag_solomon_client = None
    _released_scopes_cache = {}
    _component_names_to_process = None
    _metrics_stat_processor = None

    class Requirements(task_env.TinyRequirements):
        disk_space = 5 * 1024  # FIXME: check usage here

    class Parameters(binary_task.LastBinaryReleaseParameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        solomon_push_interval = sdk2.parameters.Integer(
            "Solomon push interval",
            default=3,
            description="Interval (in seconds) at which the data is pushed to Solomon",
        )
        release_frequency_days_number = sdk2.parameters.Integer(
            "Release frequency days count",
            default=2,
            description="The number of days to calculate release frequency statistics for",
        )
        release_saw_graph_days_number = sdk2.parameters.Integer(
            "Release saw graph days count",
            default=2,
            description="The number of days to calculate release saw graph data for",
        )

        component_name_filter = sdk2.parameters.String("Component name filter")

        with sdk2.parameters.Group("TimeSpectre Metrics Parameters") as metrics_stats_parameters:
            metrics_stats_config = sdk2.parameters.Dict(
                "Metrics Stats Config",
                default=_metrics_stats_default(),
                description="A dict <component name> => [<latest scope>, <limit>]. If <latest scope> is 0 then the "
                            "actual latest scope is considered for the component. <limit> denotes the number of scopes "
                            "to be processed.",
            )
            preload_metrics_sla_table_days_count = sdk2.parameters.Integer(
                "Preload Metrics SLA table since this number of days ago till today",
                default=2,
                description="Preload Metrics SLA table since this number of days ago till today. If 0 then nothing is "
                            "preloaded. Note that a correct preload can increase the speed of metrics stats "
                            "calculation dramatically. Overestimation on the other hand decreases this advantage "
                            "and may also lead to some significant problems.",
            )

    @property
    def token(self):
        if not self._token:
            self._token = rm_sec.get_rm_token(self)
        return self._token

    @property
    def yql_token(self):
        return sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, "RELEASE_MACHINE_BOT_YQL_TOKEN")

    @property
    def metrics_stat_processor(self):
        if self._metrics_stat_processor is None:
            from release_machine.time_spectre.metrics import lm_stats
            self._metrics_stat_processor = lm_stats.LMStatProcessor(self.yql_token, sandbox_token=self.token)
        return self._metrics_stat_processor

    @property
    def release_statistics_solomon_client(self):

        from solomon import OAuthProvider
        from sandbox.projects.release_machine.tasks.ReleaseMachineStatCrawler import solomon_client as sc

        if not self._release_statistics_solomon_client:
            self._release_statistics_solomon_client = sc.LoggedThrottledSolomonReporter(
                project=self.SOLOMON_PROJECT,
                cluster=self.SOLOMON_CLUSTER,
                service=self.STATISTICS_SOLOMON_SERVICE,
                url=rm_const.Urls.SOLOMON_URL,
                common_labels={
                    'host': '',
                },
                push_interval=self.Parameters.solomon_push_interval,
                auth_provider=OAuthProvider(self.token),
            )
        return self._release_statistics_solomon_client

    @property
    def timespectre_metrics_solomon_client(self):

        from solomon import OAuthProvider
        from sandbox.projects.release_machine.tasks.ReleaseMachineStatCrawler import solomon_client as sc

        if not self._timespectre_metrics_solomon_client:
            self._timespectre_metrics_solomon_client = sc.LoggedThrottledSolomonReporter(
                project=self.SOLOMON_PROJECT,
                cluster=self.SOLOMON_CLUSTER,
                service=self.TIMESPECTRE_METRICS_SERVICE,
                url=rm_const.Urls.SOLOMON_URL,
                common_labels={
                    'host': '',
                },
                push_interval=self.Parameters.solomon_push_interval,
                auth_provider=OAuthProvider(self.token),
            )

        return self._timespectre_metrics_solomon_client

    @property
    def rm_general_characteristics_solomon_client(self):

        from solomon import OAuthProvider
        from sandbox.projects.release_machine.tasks.ReleaseMachineStatCrawler import solomon_client as sc

        if not self._rm_general_characteristics_solomon_client:
            self._rm_general_characteristics_solomon_client = sc.LoggedThrottledSolomonReporter(
                project=self.SOLOMON_PROJECT,
                cluster=self.SOLOMON_CLUSTER,
                service=self.RM_GENERAL_CHARACTERISTICS,
                url=rm_const.Urls.SOLOMON_URL,
                common_labels={
                    'host': '',
                },
                push_interval=self.Parameters.solomon_push_interval,
                auth_provider=OAuthProvider(self.token),
            )

        return self._rm_general_characteristics_solomon_client

    @property
    def ab_exp_lag_solomon_client(self):

        from solomon import OAuthProvider
        from sandbox.projects.release_machine.tasks.ReleaseMachineStatCrawler import solomon_client as sc

        if not self._ab_exp_lag_solomon_client:
            self._ab_exp_lag_solomon_client = sc.LoggedThrottledSolomonReporter(
                project=self.SOLOMON_PROJECT,
                cluster=self.SOLOMON_CLUSTER,
                service=self.AB_EXP_LAG,
                url=rm_const.Urls.SOLOMON_URL,
                common_labels={
                    'host': '',
                },
                push_interval=self.Parameters.solomon_push_interval,
                auth_provider=OAuthProvider(self.token),
            )

        return self._ab_exp_lag_solomon_client

    @property
    def now(self):
        return datetime.datetime.utcnow().replace(tzinfo=tz.tzutc())

    @property
    def today(self):
        return datetime.datetime(year=self.now.year, month=self.now.month, day=self.now.day)

    @property
    def component_names_to_process(self):
        if not self._component_names_to_process:
            name_filter_re = re.compile(self.Parameters.component_name_filter or ".*", re.IGNORECASE)
            self._component_names_to_process = list(filter(name_filter_re.match, all_cfg.get_all_names()))
        return self._component_names_to_process

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)
        bb2.BugBannerTask.on_save(self)

    def on_execute(self):

        binary_task.LastBinaryTaskRelease.on_execute(self)
        bb2.BugBannerTask.on_execute(self)

        error_count = 0

        if self.Parameters.release_frequency_days_number:
            error_count += self._run_calculations_for_all_required_components(self._calculate_release_frequencies)

        if self.Parameters.release_saw_graph_days_number:
            error_count += self._run_calculations_for_all_required_components(
                self._calculate_release_saw_graph_data,
            )
            error_count += self._run_calculations_for_all_required_components(
                self._calculate_release_saw_calendar_graph_data,
            )

        if self.Parameters.metrics_stats_config:
            if self.Parameters.preload_metrics_sla_table_days_count:
                self.metrics_stat_processor.preload_metrics_sla_event_tables(
                    self.now - datetime.timedelta(days=self.Parameters.preload_metrics_sla_table_days_count),
                    self.now,
                )
            error_count += self._run_calculations_for_all_required_components(self._calculate_metrics_data)

        self.set_info("Processed components: {}".format(self._get_processed_components_suppressed_str_list()))

        error_count += self._calculate_component_count()

        error_count += self._push_ab_exp_compare_sensors()

        if error_count:
            eh.check_failed("{} errors. See fail messages above".format(error_count))

    def _run_calculations_for_all_required_components(self, calculation_function):
        """
        Runs `calculation_function` for all components in `self.component_names_to_process`.

        :param calculation_function: function with one positional argument, component_name

        :return: error count
        """
        error_count = 0

        for component_name in self.component_names_to_process:
            error_count += calculation_function(component_name)

        return error_count

    def _calculate_release_frequencies(self, component_name):
        """
        Calculate and push release frequency for the given component

        :param component_name: str, component name
        :return: 0 if no errors, 1 otherwise
        """
        logging.info("Running Release Frequency calculations for %s", component_name)

        processing_started_at = datetime.datetime.now()

        try:

            released_scopes = self._get_released_scopes(
                component_name,
                self.Parameters.release_frequency_days_number + self.MONTH_WINDOW.days,
            )

            scope_data_start_index = 0

            for day_index in range(self.Parameters.release_frequency_days_number):

                dt = self.today - datetime.timedelta(days=day_index)
                dt_min_month = tu.datetime_to_timestamp(dt - self.MONTH_WINDOW)
                dt_min_week = tu.datetime_to_timestamp(dt - self.WEEK_WINDOW)
                dt_max = tu.datetime_to_timestamp(dt)

                scope_count_month = 0
                scope_count_week = 0

                for scope_data in released_scopes[scope_data_start_index:]:

                    scope_started_at_ts = int(scope_data.started_at_ts)

                    if scope_started_at_ts > dt_max:
                        scope_data_start_index += 1
                        continue
                    if dt_min_week <= scope_started_at_ts:
                        scope_count_week += 1
                    if dt_min_month <= scope_started_at_ts:
                        scope_count_month += 1
                    if scope_started_at_ts < dt_min_month:
                        break

                self.release_statistics_solomon_client.set_value(
                    sensor=self.RELEASE_FREQUENCY_MONTH_SENSOR_NAME,
                    value=scope_count_month,
                    ts_datetime=dt,
                    labels={
                        'component_name': component_name,
                    },
                )
                self.release_statistics_solomon_client.set_value(
                    sensor=self.RELEASE_FREQUENCY_WEEK_SENSOR_NAME,
                    value=scope_count_week,
                    ts_datetime=dt,
                    labels={
                        'component_name': component_name,
                    },
                )

        except Exception as e:
            self.log_exception("Failed to calculate release frequency data for {}".format(component_name), e)
            return 1

        processing_finished_at = datetime.datetime.now()

        logging.debug("{} release frequency processed in {}".format(
            component_name,
            processing_finished_at - processing_started_at,
        ))

        return 0

    def _calculate_release_saw_graph_data(self, component_name):
        return self._do_calculate_release_saw_graph_data(
            component_name,
            sensor=self.RELEASE_SAW_SENSOR_NAME,
            skip_holidays=True,
        )

    def _calculate_release_saw_calendar_graph_data(self, component_name):
        return self._do_calculate_release_saw_graph_data(
            component_name,
            sensor=self.RELEASE_SAW_CALENDAR_SENSOR_NAME,
            skip_holidays=False,
        )

    def _do_calculate_release_saw_graph_data(
        self,
        component_name,
        sensor,
        skip_holidays=True,
    ):
        """
        Calculates data for release_saw graph

        :param component_name: str, component name
        :return: 0 if no errors, 1 otherwise
        """
        logging.info("Running Release Saw (skip holidays = %s) calculations for %s", skip_holidays, component_name)

        processing_started_at = datetime.datetime.now()

        try:

            released_scopes = self._get_released_scopes(
                component_name,
                self.Parameters.release_saw_graph_days_number,
            )

            if not released_scopes:
                logging.warning("%s has no released scopes (response is empty)", component_name)
                return 0

            skip_level = 0

            if skip_holidays:

                holidays = [
                    item['date'] for item in self._get_russian_holidays(
                        datetime.datetime.utcfromtimestamp(released_scopes[-1].started_at_ts).strftime("%Y-%m-%d"),
                        self.today.strftime("%Y-%m-%d"),
                    )
                ]

            else:

                holidays = []

            for scope_index, scope_data in enumerate(released_scopes):

                next_ts = (
                    (
                        released_scopes[scope_index - 1 - skip_level].started_at_ts +
                        released_scopes[scope_index - 1 - skip_level].wait_release_seconds +
                        released_scopes[scope_index - 1 - skip_level].wait_deploy_seconds -
                        3600

                    ) if scope_index > 0 else tu.datetime_to_timestamp(self.now)
                )

                start_dt = datetime.datetime.utcfromtimestamp(scope_data.started_at_ts)
                next_dt = datetime.datetime.utcfromtimestamp(next_ts)

                logging.debug("Processing scope %s at %s", scope_data.scope, start_dt)

                current_ts = scope_data.started_at_ts
                current_dt = datetime.datetime.utcfromtimestamp(current_ts)
                current_value = scope_data.wait_release_seconds + scope_data.wait_deploy_seconds
                current_deploy_ts = (
                    scope_data.started_at_ts + scope_data.wait_release_seconds + scope_data.wait_deploy_seconds
                )

                while current_ts < current_deploy_ts:

                    if current_dt.strftime("%Y-%m-%d") in holidays:
                        current_value -= self.DAY_SECONDS

                    current_ts += self.DAY_SECONDS
                    current_dt = datetime.datetime.utcfromtimestamp(current_ts)

                current_ts = (
                    scope_data.started_at_ts + scope_data.wait_release_seconds + scope_data.wait_deploy_seconds
                )
                current_dt = datetime.datetime.utcfromtimestamp(current_ts)

                if current_ts > next_ts:
                    logging.warning(
                        "- Skipping scope %s: deploy time > next deployed branch deploy time (%s > %s)",
                        scope_data.scope,
                        current_ts,
                        next_ts,
                    )
                    skip_level += 1
                    continue

                skip_level = 0

                while current_ts < next_ts:

                    self.release_statistics_solomon_client.set_value(
                        sensor=sensor,
                        ts_datetime=current_dt,
                        value=self._seconds_to_days(current_value),
                        labels={
                            'component_name': component_name,
                        }
                    )

                    logging.debug(
                        "- %s - %sd (%ss)",
                        str(current_dt),
                        self._seconds_to_days(current_value),
                        current_value,
                    )

                    current_ts += self.DAY_SECONDS
                    current_dt = datetime.datetime.utcfromtimestamp(current_ts)
                    current_dt_str = current_dt.strftime("%Y-%m-%d")

                    if current_dt_str not in holidays:
                        current_value += self.DAY_SECONDS

                self.release_statistics_solomon_client.set_value(
                    sensor=sensor,
                    ts_datetime=next_dt,
                    value=self._seconds_to_days(
                        current_value + next_ts - current_ts
                    ),
                    labels={
                        'component_name': component_name,
                    }
                )

                logging.debug(
                    "- %s - %sd (%ss)",
                    str(next_dt),
                    self._seconds_to_days(current_value + next_ts - current_ts),
                    current_value + next_ts - current_ts,
                )

        except Exception as e:
            self.log_exception("Failed to calculate release saw data for {}".format(component_name), e)
            return 1

        processing_finished_at = datetime.datetime.now()

        logging.debug("{} release saw processed in {}".format(
            component_name,
            processing_finished_at - processing_started_at,
        ))

        return 0

    def _calculate_metrics_data(self, component_name):
        """
        Calculate Metrics statistics data for the given component

        :param component_name: str, component name
        :return: 0 if no errors, 1 otherwise
        """

        if component_name not in self.Parameters.metrics_stats_config:
            logging.info("Skipping metrics statistics for %s", component_name)
            return 0

        try:
            last_scope, limit = json.loads(self.Parameters.metrics_stats_config[component_name])
        except (TypeError, ValueError):
            self.set_info("Wrong metrics stats configuration for {}: {} ({})".format(
                component_name,
                self.Parameters.metrics_stats_config[component_name],
                type(self.Parameters.metrics_stats_config[component_name]),
            ))
            logging.info("Skipping metrics statistics for %s (due to wrong configuration)", component_name)
            return 1

        logging.info("Running Metrics calculations for %s", component_name)

        from release_machine.time_spectre.metrics import component_manager as metrics_component_manager

        scopes = self._get_scopes(
            component_name,
            int(last_scope),
            int(limit),
        )
        scopes = scopes.branch_scopes if scopes.branch_scopes else scopes.tag_scopes

        scope_range = [int(scope.scope_number) for scope in scopes]

        try:
            metrics_component_manager.process_component_scope_range(
                component_name,
                scope_range,
                yql_token=self.yql_token,
                solomon_client=self.timespectre_metrics_solomon_client,
                sandbox_token=self.token,
                lmsp=self.metrics_stat_processor,
            )
        except Exception as e:
            self.log_exception("Failed to calculate metrics data for {}".format(component_name), e)
            return 1

        return 0

    def _calculate_component_count(self):
        from release_machine.release_machine.services.release_engine.services import ModelClient
        from release_machine.release_machine.proto.structures import message_pb2, table_pb2

        rm_model_client = ModelClient.from_address(rm_const.Urls.RM_HOST)

        try:

            response = rm_model_client.get_components(message_pb2.Dummy())

            component_count_total = len(response.components)

            self.rm_general_characteristics_solomon_client.set_value(
                sensor=self.COMPONENT_COUNT_TOTAL_SENSOR,
                ts_datetime=self.now,
                value=component_count_total,

            )

            component_count_active = sum(
                map(
                    lambda x: x.status == table_pb2.Component.ComponentStatus.ACTIVE,
                    response.components,
                )
            )

            self.rm_general_characteristics_solomon_client.set_value(
                sensor=self.COMPONENT_COUNT_ACTIVE_SENSOR,
                ts_datetime=self.now,
                value=component_count_active,
            )

            component_count_ci = sum(
                map(
                    lambda x: (
                        x.release_cycle == rm_const.ReleaseCycleType.to_str(rm_const.ReleaseCycleType.CI)
                        and
                        x.status == table_pb2.Component.ComponentStatus.ACTIVE
                    ),
                    response.components,
                )
            )

            self.rm_general_characteristics_solomon_client.set_value(
                sensor=self.COMPONENT_COUNT_CI_SENSOR,
                ts_datetime=self.now,
                value=component_count_ci,
            )

        except Exception as e:
            self.log_exception("Failed to calculate component count", e)
            return 1

        return 0

    def _push_ab_exp_compare_sensors(self):

        try:

            ab_exp_compare.calculate_and_push_task_lag_data(
                sandbox_client=self.server,
                solomon_client=self.ab_exp_lag_solomon_client,
                task_type="YABS_SERVER_TEST_AB_EXPERIMENT",
                limit=5000,
            )

        except:
            logging.exception("_push_ab_exp_compare_sensors failed")
            return 1

        return 0

    def _get_scopes(self, component_name, last_scope_number=0, limit=1):
        """
        Get scopes info for the given component

        :param component_name: str, component_name
        :param limit: int, the number of scopes to retrieve
        :return: `release_machine.release_machine.proto.structures.message_pb2.ScopesResponse`
        """
        from release_machine.release_machine.services.release_engine.services import ModelClient
        from release_machine.release_machine.proto.structures import message_pb2

        rm_model_client = ModelClient.from_address(rm_const.Urls.RM_HOST)
        response = rm_model_client.get_scopes(message_pb2.ScopesRequest(
            component_name=component_name,
            start_scope_number=last_scope_number,
            limit=limit,
        ))
        return response

    def _get_released_scopes(self, component_name, limit):
        """
        Get released scopes info for the given component

        :param component_name: str, component_name
        :param limit: int, max number of scopes to get
        :return: a list of `release_machine.release_machine.proto.structures.message_pb2.ReleasedScopesInfoItem`
        """

        if (
            component_name in self._released_scopes_cache and
            self._released_scopes_cache[component_name]['limit'] >= limit
        ):
            return self._released_scopes_cache[component_name]['scopes'][:limit]

        from release_machine.release_machine.services.release_engine.services import RMStatsClient
        from release_machine.release_machine.proto.structures import message_pb2

        rm_stats_client = RMStatsClient.from_address(rm_const.Urls.RM_HOST)
        released_scopes_info = rm_stats_client.get_released_scopes_info(message_pb2.ReleasedScopesInfoRequest(
            component_name=component_name,
            limit=limit,
        ))

        self._released_scopes_cache[component_name] = {
            'limit': limit,
            'scopes': released_scopes_info.scopes,
        }

        return released_scopes_info.scopes

    @staticmethod
    def _get_russian_holidays(date_from, date_to):
        """
        Retrieves holidays in Russia between `date_from` and `date_to` from Calendar API.

        :param date_from: "%Y-%m-%d"-formatted date str, get holidays starting from this date
        :param date_to: "%Y-%m-%d"-formatted date str, get holidays before this date
        :return: a list of dicts of holidays (see https://wiki.yandex-team.ru/calendar/api/new-web/#get-holidays)
        """
        url = "{url}?from={date_from}&to={date_to}&for=rus&outMode=holidays".format(
            url=os.path.join(rm_const.Urls.CALENDAR_API_URL, 'get-holidays'),
            date_from=date_from,
            date_to=date_to,
        )

        return requests_wrapper.get_r(url).json().get('holidays', [])

    @classmethod
    def _seconds_to_days(cls, seconds):
        """Converts seconds to days."""
        return seconds / cls.DAY_SECONDS

    def _get_processed_components_suppressed_str_list(self):
        """
        Get a comma-separated list (string) of names of components processed by the task. Keep only
        `self.NUMBER_OF_PROCESSED_COMPONENTS_TO_PRINT_IN_INFO` in output (the rest is denoted by
        "and N others" where N is the number of remaining components)

        :return: str with comma-separated list of processed components' names
        """
        components_to_print = ", ".join(
            self.component_names_to_process[:self.NUMBER_OF_PROCESSED_COMPONENTS_TO_PRINT_IN_INFO]
        )
        components_remain_count = len(
            self.component_names_to_process[self.NUMBER_OF_PROCESSED_COMPONENTS_TO_PRINT_IN_INFO:]
        )

        if components_remain_count:
            return "{} and {} others".format(components_to_print, components_remain_count)

        return components_to_print

    def log_exception(self, message, exc):
        self.set_info("{}. See log for more info".format(message))
        eh.log_exception(message, exc, task=self)
