"""
!!!PY3 ONLY!!!
"""

import typing
import datetime
import logging

from dateutil import parser as dt_parser

from sandbox.projects.common import gsid_parser


TaskInfoData = typing.TypedDict(
    "TaskInfoData",
    {
        "id": int,
        "context.__GSID": str,
        "context.__CI_CONTEXT": typing.Optional[dict],
        "time.created": str,
        "type": str,
    }
)


SolomonKwargsData = typing.TypedDict(
    "SolomonKwargsData",
    {
        "sensor": str,
        "value": typing.Union[int, float],
        "ts_datetime": datetime.datetime,
        "labels": typing.Dict[str, str],
    }
)


def get_revision_commit_datetime(revision):
    # type: (int) -> datetime.datetime

    import release_machine.vcs_indexer.client.python as vcs_indexer_lib

    vcs_indexer = vcs_indexer_lib.VcsIndexer()

    batch_info = vcs_indexer.batch_info(
        vcs_indexer_lib.BatchInfoRequest(
            start_revision=revision,
            end_revision=revision,
            need_paths=False,
            offset=0,
            limit=1,
        ),
    )

    return dt_parser.parse(batch_info["logentries"][0]["date"])


def build_task_lag_solomon_kwarg_data(task_info):
    # type: (TaskInfoData) -> typing.Optional[SolomonKwargsData]

    logging.info("Building task lag data for %s", task_info["id"])

    task_type = task_info["type"]
    task_started_at = dt_parser.parse(task_info["time.created"])

    if task_info["context.__CI_CONTEXT"]:
        revision = task_info["context.__CI_CONTEXT"]["target_revision"]["number"]
        component_name = "ab_exp"
    else:
        revision = gsid_parser.get_svn_revision_from_gsid(task_info["context.__GSID"])
        component_name = "ab_experiments"

    if not revision:
        logging.info("Cannot parse revision for task %s - skipping",  task_info["id"])
        return

    revision = int(revision)

    sensor = task_type.lower()
    commit_datetime = get_revision_commit_datetime(revision)
    task_start_lag = (task_started_at - commit_datetime).total_seconds()

    return SolomonKwargsData(
        sensor=sensor,
        ts_datetime=commit_datetime,
        value=task_start_lag,
        labels={
            "component_name": component_name,
            "revision": revision,
        },
    )


def calculate_and_push_task_lag_data(sandbox_client, solomon_client, task_type, limit, offset=0):

    response = sandbox_client.task.read(
        type=task_type,
        limit=limit,
        offset=offset,
        fields=list(TaskInfoData.__annotations__.keys()),
    )

    for task_info in response["items"]:

        data = build_task_lag_solomon_kwarg_data(task_info)

        if not data:
            continue

        solomon_client.set_value(**data)
